print.summarytools <- function(x, method="pander", include.footer=TRUE, silent=FALSE, ...) {

  if(!method %in% c("pander", "browser", "viewer", "html_noshow"))
    stop("method must be one of 'pander', 'browser' or 'viewer'")

  # Build info.table and prepare the field  -----------------------------------

  # Function to add '#' in non-plain-ascii pander tables
  addHash <- function(string, n=4) {
    if(attr(x, "pander.args")$plain.ascii)
      string
    else
      paste(paste0(rep("#",n), collapse = ""), string)
  }
  
  if(attr(x, "st.type") %in% c("freq", "descr")) {
    var.info <- attr(x, "var.info")
    var.info <- var.info[!is.na(var.info)]
    
    if(method=="pander")
      var.info <- paste(addHash(names(var.info)), var.info, sep=": ", collapse="\n")
    else
      var.info <- as.matrix(paste(names(var.info), var.info, sep=": "))
  }

  if(method != "pander" && include.footer) {
    htmlfooter <- paste0("Generated by <a href='https://github.com/dcomtois/summarytools'>summarytools</a> package version ",
                         packageVersion(pkg = "summarytools"),
                         " (<a href='http://www.r-project.org/'>R</a> version ", getRversion(), ")",
                         "<br/>", Sys.Date())
  } else {
    htmlfooter <- ""
  }
  
  # printing freq objects -----------------------------------------------------
  if(attr(x, "st.type") == "freq") {
    report.title <- ifelse("weights" %in% names(attributes(x)),
                           "Weighted Frequencies",
                           "Frequencies")
      
    if(method == "pander") {
      cat("\n", addHash(report.title, 3), "\n\n", var.info, sep = "")
      cat(do.call(pander::pander, append(attr(x, "pander.args"), list(x = quote(x)))))
    } else {
      # method = viewer / browser / html_noshow ------------------------------
      freq.table.html <-
        xtable::print.xtable(xtable::xtable(x = x, align = "rccccc",
                                            digits = c(0,
                                                       attr(x, "pander.args")$round * as.numeric("weights" %in% names(attributes(x))),
                                                       rep(attr(x, "pander.args")$round, 4))),
                             type = "html", print.results = FALSE,
                             html.table.attributes = 'class="table table-striped table-bordered"')

      stpath <- find.package("summarytools")

      html.content <- tags$html(
        HTML(text = "<head>"),
        tags$meta(charset="UTF-8"),
        includeCSS(path = paste(stpath,"includes/stylesheets/bootstrap.min.css", sep="/")),
        includeCSS(path = paste(stpath,"includes/stylesheets/custom.css", sep="/")),
        HTML(text = "</head>"),
        tags$body(
          div(class="container", # style="width:80%",
              h2(report.title),
              apply(var.info, 1, h4),
              br(),
              HTML(gsub("<td> ", "<td>", freq.table.html)), # To avoid initial space in cells
              HTML(text = htmlfooter)
          )
        )
      )
    }
  }

  # Printing descr objects ----------------------------------------------------
  if(attr(x, "st.type") == "descr") {

    if(!silent && "ignored" %in% names(attributes(x)))
      message("Non-numerical variable(s) ignored: ", attr(x, "ignored"))
    
    report.title <- ifelse("weights" %in% names(attributes(x)),
                           "Weighted Descriptive Statistics",
                           "Descriptive Statistics")
    
    # With method="pander" --------------------------------------
    if(method=="pander") {
      cat("\n", addHash(report.title, 3), "\n\n", var.info, sep = "")
      cat(do.call(pander::pander, append(attr(x, "pander.args"), list(x=quote(x$stats)))))
      cat(addHash("Observations"))
      cat(do.call(pander::pander, append(attr(x, "pander.args"), list(x=quote(x$observ)))))
    } else {
      # method = viewer / browser / html_noshow --------------------------
      descr.table.html <-
        xtable::print.xtable(xtable::xtable(x = x$stats,
                                            align = paste0("r", paste(rep("c",ncol(x$stats)),
                                                                     collapse="")),
                                            digits = c(0,rep(attr(x, "pander.args")$round,
                                                             ncol(x$stats)))),
                             type = "html", print.results = FALSE,
                             html.table.attributes = 'class="table table-striped table-bordered"')

      obs.table.html <-
        xtable::print.xtable(xtable::xtable(x = x$observ, align = paste0("r", paste(rep("c",ncol(x$observ)),collapse="")),
                                            digits = c(0,rep(attr(x, "pander.args")$round,ncol(x$observ)))),
                             type = "html", print.results = FALSE,
                             html.table.attributes = 'class="table table-striped table-bordered"')

      stpath <- find.package("summarytools")

      html.content <- tags$html(
        HTML(text = "<head>"),
        tags$meta(charset="UTF-8"),
        includeCSS(path = paste(stpath,"includes/stylesheets/bootstrap.min.css", sep="/")),
        includeCSS(path = paste(stpath,"includes/stylesheets/custom.css", sep="/")),
        HTML(text = "</head>"),
        tags$body(
          div(class="container", # style="width:80%",
              h2(report.title),
              apply(var.info, 1, h4),
              br(),
              HTML(gsub("<td> ", "<td>", descr.table.html)),
              h3("Observations"),
              HTML(gsub("<td> ", "<td>", obs.table.html)),
              HTML(text = htmlfooter)
          )
        )
      )
    }
  }

  # Printing dfSummary objects ------------------------------------------------
  if(attr(x, "st.type") == "dfSummary") {
    if(method=="pander") {
      cat("\n", addHash("Dataframe Summary", 3), "\n",
          "\n", addHash(attr(x, "df.name"), 4), "\n", sep = "")
      cat(do.call(pander::pander, append(attr(x, "pander.args"), list(x=quote(x)))))
    } else {
      # method = viewer / browser / html_noshow --------------------------------
      dfSummary.html <-
        xtable::print.xtable(xtable::xtable(x = x,digits = 0,
                                            align = paste0("c", paste(rep("l",ncol(x)),collapse=""))),
                             include.rownames = FALSE, type = "html", print.results = FALSE,
                             sanitize.colnames.function = function(x) gsub("\\.", " ", x),
                             html.table.attributes = 'class="table table-striped table-bordered"')

      stpath <- find.package("summarytools")

      html.content <- tags$html(
          HTML(text = "<head>"),
          tags$meta(charset="UTF-8"),
          tags$title(paste("dfSummary -", attr(x, "df.name"))),
          includeCSS(path = paste(stpath,"includes/stylesheets/bootstrap.min.css", sep="/")),
          includeCSS(path = paste(stpath,"includes/stylesheets/custom.css", sep="/")),
          HTML(text = "</head>"),
        tags$body(
          div(class="container",
              h3("Dataframe Summary"),
              h2(attr(x, "df.name")),
              if("rows.subset" %in% names(attributes(x)$var.info))
                p("Rows subset:", attr(x,"subset")),
              h4("Number of rows: ", attr(x, "n.obs")),
              br(),
              HTML(gsub("<td> ", "<td>", dfSummary.html)),
              HTML(text = htmlfooter)
          )
        )
      )
    }
  }

  # Create and open the output html file --------------------------------------------
  if(method %in% c('browser', 'viewer', 'html_noshow')) {
    htmlfile <- paste0(tempfile(),".html")
    html.content.utf8 <- iconv(as.character(html.content, from=getOption("encoding"), to="UTF-8"))
    capture.output(cat("<!DOCTYPE html>\n", html.content.utf8), file = htmlfile)
  }

  if(method=="viewer") {
    if(.Platform$GUI == "RStudio") 
      rstudioapi::viewer(htmlfile)
    else {
      message("Method 'viewer' only valid within RStudio. Switching method to 'browser'.")
      method <- "browser"
    }
  }

  # For method "browser", we don't use utils::browseURL() because of compatibility issues with RStudio
  if(method=="browser") {
    switch(Sys.info()[['sysname']],
           Windows = {shell.exec(file = paste0("file:///", htmlfile))},
           Linux   = {system(paste('/usr/bin/xdg-open', htmlfile), wait = FALSE, ignore.stdout = TRUE)},
           Darwin  = {system(paste('open', htmlfile), wait = FALSE, ignore.stderr = TRUE)})
  }

  # return file path and update tmpfiles vector when method = browser or viewer ----------------------
  if(method %in% c("browser", "viewer", "html_noshow")) {
    if(!silent)
      message(paste0("Temporary file created: ", htmlfile, '. To delete, use cleartmp().'))
    .st.env$tmpfiles <- c(.st.env$tmpfiles, htmlfile)
    return(invisible(normalizePath(htmlfile)))
  } else {
    return(invisible())
  }
}
