% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/id.cv.R
\name{id.cv}
\alias{id.cv}
\title{Changes in volatility identification of SVAR models}
\usage{
id.cv(x, SB, start = NULL, end = NULL, frequency = NULL,
  format = NULL, dateVector = NULL, max.iter = 50, crit = 0.001,
  restriction_matrix = NULL)
}
\arguments{
\item{x}{An object of class 'vars', 'vec2var', 'nlVar'. Estimated VAR object}

\item{SB}{Integer, vector or date character. The structural break is specified either by an integer (number of observations in the pre-break period),
a vector of ts() frequencies if a ts object is used in the VAR or a date character. If a date character is provided, either a date vector containing the whole time line
in the corresponding format (see examples) or common time parameters need to be provided}

\item{start}{Character. Start of the time series (only if dateVector is empty)}

\item{end}{Character. End of the time series (only if dateVector is empty)}

\item{frequency}{Character. Frequency of the time series (only if dateVector is empty)}

\item{format}{Character. Date format (only if dateVector is empty)}

\item{dateVector}{Vector. Vector of time periods containing SB in corresponding format}

\item{max.iter}{Integer. Number of maximum GLS iterations}

\item{crit}{Numeric. Critical value for the precision of the GLS estimation}

\item{restriction_matrix}{Matrix or vector. A matrix containing presupposed entries for matrix B, NA if no restriction is imposed (entries to be estimated). Alternatively, a K^2*J (or J*K^2) matrix can be passed, where J is the number of restrictions and K the number of time series. (as suggested in Luetkepohl, 2017, section 5.2.1).}
}
\value{
A list of class "svars" with elements
\item{Lambda}{Estimated unconditional heteroscedasticity matrix \eqn{\Lambda}}
\item{Lambda_SE}{Matrix of standard errors of Lambda}
\item{B}{Estimated structural impact matrix B, i.e. unique decomposition of the covariance matrix of reduced form residuals}
\item{B_SE}{Standard errors of matrix B}
\item{n}{Number of observations}
\item{Fish}{Observed Fisher information matrix}
\item{Lik}{Function value of likelihood}
\item{wald_statistic}{Results of pairwise Wald tests}
\item{iteration}{Number of GLS estimations}
\item{method}{Method applied for identification}
\item{SB}{Structural break (number of observations)}
\item{A_hat}{Estimated VAR parameter via GLS}
\item{type}{Type of the VAR model, e.g. 'const'}
\item{SBcharacter}{Structural break (date; if provided in function arguments)}
\item{restrictions}{Number of specified restrictions}
\item{restriction_matrix}{Specified restriction matrix}
\item{y}{Data matrix}
\item{p}{Number of lags}
\item{K}{Dimension of the VAR}
}
\description{
Given an estimated VAR model, this function applies changes in volatility to identify the structural impact matrix B of the corresponding SVAR model
\deqn{y_t=c_t+A_1 y_{t-1}+...+A_p y_{t-p}+u_t
=c_t+A_1 y_{t-1}+...+A_p y_{t-p}+B \epsilon_t.}
Matrix B corresponds to the decomposition of the pre-break covariance matrix \eqn{\Sigma_1=B B'}.
The post-break covariance corresponds to \eqn{\Sigma_2=B\Lambda B'} where \eqn{\Lambda} is the estimated unconditional heteroskedasticity matrix.
}
\examples{
#' # data contains quartlery observations from 1965Q1 to 2008Q2
# assumed structural break in 1979Q3
# x = output gap
# pi = inflation
# i = interest rates
set.seed(23211)
v1 <- vars::VAR(USA, lag.max = 10, ic = "AIC" )
x1 <- id.cv(v1, SB = 59)
summary(x1)

# switching columns according to sign patter
x1$B <- x1$B[,c(3,2,1)]
x1$B[,3] <- x1$B[,3]*(-1)

# Impulse response analysis
i1 <- irf(x1, n.ahead = 30)
plot(i1, scales = 'free_y')

# Restrictions
# Assuming that the interest rate doesn't influence the output gap on impact
restMat <- matrix(rep(NA, 9), ncol = 3)
restMat[1,3] <- 0
x2 <- id.cv(v1, SB = 59, restriction_matrix = restMat)
summary(x2)

# In alternative Form
restMat <- c(rep(0,6), 1, 0,0)
x2 <- id.cv(v1, SB = 59, restriction_matrix = restMat)
summary(x2)

#Structural brake via Dates
# given that time series vector with dates is available
dateVector = seq(as.Date("1965/1/1"), as.Date("2008/7/1"), "quarter")
x3 <- id.cv(v1, SB = "1979-07-01", format = "\%Y-\%m-\%d", dateVector = dateVector)
summary(x3)

# or pass sequence arguments directly
x4 <- id.cv(v1, SB = "1979-07-01", format = "\%Y-\%m-\%d", start = "1965-01-01", end = "2008-07-01",
frequency = "quarter")
summary(x4)

# or provide ts date format (For quarterly, monthly, weekly and daily frequencies only)
x5 <- id.cv(v1, SB = c(1979, 3))
summary(x5)

}
\references{
Rigobon, R., 2003. Identification through Heteroskedasticity. The Review of Economics and Statistics, 85, 777-792.\cr
 Herwartz, H. & Ploedt, M., 2016. Simulation Evidence on Theory-based and Statistical Identification under Volatility Breaks Oxford Bulletin of Economics and Statistics, 78, 94-112.
}
\seealso{
For alternative identification approaches see \code{\link{id.st}}, \code{\link{id.cvm}}, \code{\link{id.dc}} or \code{\link{id.ngml}}
}
