% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstimationML.R
\name{EstimationMaxLinear}
\alias{EstimationMaxLinear}
\title{Estimation of the parameters of the max-linear model}
\usage{
EstimationMaxLinear(x, indices, k, method, Bmatrix = NULL, Ldot = NULL,
  biascorr = FALSE, k1 = (nrow(x) - 10), tau = 5, startingValue,
  Omega = diag(nrow(indices)), iterate = FALSE, covMat = TRUE,
  GoFtest = FALSE, dist = 0.01, EURO = FALSE)
}
\arguments{
\item{x}{An \eqn{n} x \eqn{d} data matrix.}

\item{indices}{A \eqn{q} x \eqn{d} matrix containing at least 2 non-zero elements per row, representing the values in which we will evaluate the stable tail dependence function.}

\item{k}{An integer between 1 and \eqn{n - 1}; the threshold parameter in the definition of the empirical stable tail dependence function.}

\item{method}{Choose between \code{Mestimator} and \code{WLS}.}

\item{Bmatrix}{A function that converts the parameter vector theta to a parameter matrix B. If nothing is provided, then a simple 2-factor model is assumed.}

\item{Ldot}{For \code{method = "WLS"} only. A \eqn{q} x \eqn{p} matrix, where \eqn{p} is the number of parameters of the model. Represents the total derivative of the function L defined in Einmahl et al. (2016). If nothing is provided, then a simple 2-factor model is assumed.}

\item{biascorr}{For \code{method = "WLS"} only. If \code{TRUE}, then the bias-corrected estimator of the stable tail dependence function is used. Defaults to \code{FALSE}.}

\item{k1}{For \code{biascorr = TRUE} only. The value of \eqn{k_1} in the definition of the bias-corrected estimator of the stable tail dependence function.}

\item{tau}{For \code{biascorr = TRUE} only. The parameter of the power kernel.}

\item{startingValue}{Initial value of the parameters in the minimization routine.}

\item{Omega}{A \eqn{q} x \eqn{q} matrix specifying the metric with which the distance between the parametric and nonparametric estimates will be computed. The default is the identity matrix, i.e., the Euclidean metric.}

\item{iterate}{A Boolean variable. For \code{method = "WLS"} only. If \code{TRUE}, then continuous updating is used. Defaults to \code{FALSE}.}

\item{covMat}{A Boolean variable. For \code{method = "WLS"} only. If \code{TRUE} (the default), the covariance matrix is calculated.}

\item{GoFtest}{A Boolean variable. For \code{method = "WLS"} only. If \code{TRUE}, then the goodness-of-fit test of Corollary 2.6 from Einmahl et al. (2016) is performed. Defaults to \code{FALSE}.}

\item{dist}{A positive scalar. If \code{GoFtest = TRUE}, only eigenvalues \eqn{\nu} larger than \code{dist} are used; see Corollary 2.6 (Einmahl et al., 2016). Defaults to 0.01.}

\item{EURO}{A Boolean variable. If \code{TRUE}, then the model from Einmahl et al. (2016, Section 4) is assumed, and the corresponding \code{Bmatrix} and \code{Ldot} are used.}
}
\value{
For \code{Mestimator}, the estimator \code{theta} is returned. For \code{WLS}, a list with the following components:
\tabular{ll}{
\code{theta} \tab The estimator with estimated optimal weight matrix. \cr
\code{theta_pilot} \tab The estimator without the optimal weight matrix.\cr
\code{covMatrix} \tab The estimated covariance matrix for the estimator. \cr
\code{value} \tab The value of the minimized function at \code{theta}. \cr
\code{GoFresult} \tab A list of length two, returning the value of the test statistic and \code{s}. \cr
}
}
\description{
Estimation the parameters of the max-linear model, using either the pairwise M-estimator or weighted least squares (WLS).
}
\details{
The matrix \code{indices} can be either user defined or returned by \code{selectGrid}.
For \code{method = "Mestimator"}, only a grid with exactly two ones per row is accepted,
representing the pairs to be used. The functions \code{Bmatrix} and \code{Ldot} can be defined
such that they represent a max-linear model on a directed acyclic graph: see the vignette for this package for an example.
}
\examples{
## Generate data
set.seed(1)
n <- 1000
fr <- matrix(-1/log(runif(2*n)), nrow = n, ncol = 2)
data <- cbind(pmax(0.3*fr[,1],0.7*fr[,2]),pmax(0.5*fr[,1],0.5*fr[,2]),pmax(0.9*fr[,1],0.1*fr[,2]))
## Transform data to unit Pareto margins
x <- apply(data, 2, function(i) n/(n + 0.5 - rank(i)))
## Define indices in which we evaluate the estimator
indices <- selectGrid(cst = c(0,0.5,1), d = 3)
EstimationMaxLinear(x, indices, k = 100, method = "WLS", startingValue = c(0.3,0.5,0.9))
indices <- selectGrid(cst = c(0,1), d = 3)
EstimationMaxLinear(x, indices, k = 100, method = "Mestimator", startingValue = c(0.3,0.5,0.9))
}
\references{
Einmahl, J.H.J., Kiriliouk, A., and Segers, J. (2016). A continuous updating weighted least squares estimator of tail dependence in high dimensions. See http://arxiv.org/abs/1601.04826.
}
\seealso{
\code{\link{selectGrid}}
}

