% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival_biomarkers_subgroups.R
\name{survival_biomarkers_subgroups}
\alias{survival_biomarkers_subgroups}
\alias{tabulate_survival_biomarkers}
\title{Tabulate biomarker effects on survival by subgroup}
\usage{
tabulate_survival_biomarkers(
  df,
  vars = c("n_tot", "n_tot_events", "median", "hr", "ci", "pval"),
  groups_lists = list(),
  control = control_coxreg(),
  label_all = "All Patients",
  time_unit = NULL,
  na_str = default_na_str(),
  .indent_mods = 0L
)
}
\arguments{
\item{df}{(\code{data.frame})\cr containing all analysis variables, as returned by
\code{\link[=extract_survival_biomarkers]{extract_survival_biomarkers()}}.}

\item{vars}{(\code{character})\cr the names of statistics to be reported among:
\itemize{
\item \code{n_tot_events}: Total number of events per group.
\item \code{n_tot}: Total number of observations per group.
\item \code{median}: Median survival time.
\item \code{hr}: Hazard ratio.
\item \code{ci}: Confidence interval of hazard ratio.
\item \code{pval}: p-value of the effect.
Note, one of the statistics \code{n_tot} and \code{n_tot_events}, as well as both \code{hr} and \code{ci} are required.
}}

\item{groups_lists}{(named \code{list} of \code{list})\cr optionally contains for each \code{subgroups} variable a
list, which specifies the new group levels via the names and the
levels that belong to it in the character vectors that are elements of the list.}

\item{control}{(\code{list})\cr a list of parameters as returned by the helper function \code{\link[=control_coxreg]{control_coxreg()}}.}

\item{label_all}{(\code{string})\cr label for the total population analysis.}

\item{time_unit}{(\code{string})\cr label with unit of median survival time. Default \code{NULL} skips displaying unit.}

\item{na_str}{(\code{string})\cr string used to replace all \code{NA} or empty values in the output.}

\item{.indent_mods}{(named \code{integer})\cr indent modifiers for the labels. Defaults to 0, which corresponds to the
unmodified default behavior. Can be negative.}
}
\value{
An \code{rtables} table summarizing biomarker effects on survival by subgroup.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Tabulate the estimated effects of multiple continuous biomarker variables
across population subgroups.
}
\details{
These functions create a layout starting from a data frame which contains
the required statistics. The tables are then typically used as input for forest plots.
}
\section{Functions}{
\itemize{
\item \code{tabulate_survival_biomarkers()}: Table-creating function which creates a table
summarizing biomarker effects on survival by subgroup.

}}
\note{
In contrast to \code{\link[=tabulate_survival_subgroups]{tabulate_survival_subgroups()}} this tabulation function does
not start from an input layout \code{lyt}. This is because internally the table is
created by combining multiple subtables.
}
\examples{
library(dplyr)

adtte <- tern_ex_adtte

# Save variable labels before data processing steps.
adtte_labels <- formatters::var_labels(adtte)

adtte_f <- adtte \%>\%
  filter(PARAMCD == "OS") \%>\%
  mutate(
    AVALU = as.character(AVALU),
    is_event = CNSR == 0
  )
labels <- c("AVALU" = adtte_labels[["AVALU"]], "is_event" = "Event Flag")
formatters::var_labels(adtte_f)[names(labels)] <- labels

# Typical analysis of two continuous biomarkers `BMRKR1` and `AGE`,
# in multiple regression models containing one covariate `RACE`,
# as well as one stratification variable `STRATA1`. The subgroups
# are defined by the levels of `BMRKR2`.

df <- extract_survival_biomarkers(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    biomarkers = c("BMRKR1", "AGE"),
    strata = "STRATA1",
    covariates = "SEX",
    subgroups = "BMRKR2"
  ),
  data = adtte_f
)
df

# Here we group the levels of `BMRKR2` manually.
df_grouped <- extract_survival_biomarkers(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    biomarkers = c("BMRKR1", "AGE"),
    strata = "STRATA1",
    covariates = "SEX",
    subgroups = "BMRKR2"
  ),
  data = adtte_f,
  groups_lists = list(
    BMRKR2 = list(
      "low" = "LOW",
      "low/medium" = c("LOW", "MEDIUM"),
      "low/medium/high" = c("LOW", "MEDIUM", "HIGH")
    )
  )
)
df_grouped

## Table with default columns.
tabulate_survival_biomarkers(df)

## Table with a manually chosen set of columns: leave out "pval", reorder.
tab <- tabulate_survival_biomarkers(
  df = df,
  vars = c("n_tot_events", "ci", "n_tot", "median", "hr"),
  time_unit = as.character(adtte_f$AVALU[1])
)

## Finally produce the forest plot.
\donttest{
g_forest(tab, xlim = c(0.8, 1.2))
}

}
\seealso{
\code{\link[=h_tab_surv_one_biomarker]{h_tab_surv_one_biomarker()}} which is used internally, \code{\link[=extract_survival_biomarkers]{extract_survival_biomarkers()}}.
}
