# A geom_linerange but with sensible defaults for displaying multiple intervals
#
# Author: mjskay
###############################################################################


# Names that should be suppressed from global variable check by codetools
# Names used broadly should be put in _global_variables.R
globalVariables(c(".lower", ".upper", ".width"))


#' Multiple probability interval plots (ggplot geom)
#'
#' Modified versions of \code{\link{geom_linerange}} and \code{\link{geom_linerangeh}} with default aesthetics
#' designed for use with output from \code{\link{point_interval}}.
#'
#' \code{geom_interval} is a modified version of \code{\link{geom_linerange}}, and \code{geom_intervalh} is
#' a modified version of \code{\link{geom_linerangeh}}. These geoms set some default aesthetics equal
#' to the \code{.lower}, \code{.upper}, and \code{.width} columns generated by the \code{point_interval} family
#' of functions, making them often more convenient than vanilla \code{\link{geom_linerange}} or
#' \code{\link{geom_linerangeh}} when used with functions like \code{\link{median_qi}}, \code{\link{mean_qi}},
#' \code{\link{mode_hdi}}, etc.
#'
#' Specifically, \code{geom_interval} acts as if its default aesthetics are
#' \code{aes(ymin = .lower, ymax = .upper, color = fct_rev(ordered(.width)))}. \code{geom_intervalh} acts as if
#' its default aesthetics are \code{aes(xmin = .lower, xmax = .upper, color = fct_rev(ordered(.width)))}.
#'
#' @param mapping The aesthetic mapping, usually constructed with
#' \code{\link{aes}} or \code{\link{aes_string}}. Only needs to be set at the
#' layer level if you are overriding the plot defaults.
#' @param data A layer specific dataset - only needed if you want to override
#' the plot defaults.
#' @param stat The statistical transformation to use on the data for this layer.
#' @param position The position adjustment to use for overlapping points on this layer.
#' @param ...  Other arguments passed to \code{\link{layer}}.
#' @param na.rm	If \code{FALSE}, the default, missing values are removed with a warning. If \code{TRUE}, missing
#' values are silently removed.
#' @param show.legend Should this layer be included in the legends? \code{NA}, the default, includes if any aesthetics
#' are mapped. \code{FALSE} never includes, and \code{TRUE} always includes.
#' @param inherit.aes If \code{FALSE}, overrides the default aesthetics, rather than combining with them. This is
#' most useful for helper functions that define both data and aesthetics and shouldn't inherit behavior from the
#' default plot specification, e.g. borders.
#' @author Matthew Kay
#' @seealso See \code{\link{geom_lineribbon}} for a similar geom designed for curves plus probability bands. See
#' \code{\link{geom_linerange}} and \code{\link{geom_linerangeh}} for the geoms these are based on.
#' @keywords manip
#' @examples
#'
#' library(magrittr)
#' library(ggplot2)
#'
#' data(RankCorr, package = "tidybayes")
#'
#' RankCorr %>%
#'   spread_draws(u_tau[i]) %>%
#'   median_qi(.width = c(.5, .8, .95, .99)) %>%
#'   ggplot(aes(y = i, x = u_tau)) +
#'   geom_intervalh() +
#'   scale_color_brewer()
#'
#' RankCorr %>%
#'   spread_draws(u_tau[i]) %>%
#'   median_qi(.width = c(.5, .8, .95, .99)) %>%
#'   ggplot(aes(x = i, y = u_tau)) +
#'   geom_interval() +
#'   scale_color_brewer()
#'
#' @import ggplot2
#' @export
geom_interval <- function(mapping = NULL, data = NULL,
  stat = "identity", position = "identity",
  ...,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE) {

  l = layer(
    data = data,
    mapping = mapping,
    stat = stat,
    geom = GeomInterval,
    position = position,
    show.legend = show.legend,
    inherit.aes = inherit.aes,
    params = list(
      na.rm = na.rm,
      ...
    )
  )

  #provide some default computed aesthetics
  default_computed_aesthetics = aes(ymin = .lower, ymax = .upper, color = forcats::fct_rev(ordered(.width)))

  compute_aesthetics = l$compute_aesthetics
  l$compute_aesthetics = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    compute_aesthetics(data, plot)
  }

  map_statistic = l$map_statistic
  l$map_statistic = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    map_statistic(data, plot)
  }

  l
}

#' @rdname tidybayes-ggproto
#' @format NULL
#' @usage NULL
#' @importFrom grid grobName gTree gList
#' @import ggplot2
#' @export
GeomInterval <- ggproto("GeomInterval", Geom,
  default_aes = aes(colour = "black", size = 4, linetype = 1, shape = 19,
    fill = NA, alpha = NA, stroke = 1),

  draw_key = draw_key_path,

  required_aes = c("x", "y", "ymin", "ymax"),

  draw_panel = function(data, panel_scales, coord) {
    # draw all the intervals
    interval_grobs = data %>%
      dlply("group", function(d) {
        group_grobs = list(GeomLinerange$draw_panel(d, panel_scales, coord))
        list(
          width = d %$% mean(abs(ymax - ymin)),
          grobs = group_grobs
        )
      })

    # this is a slightly hackish approach to getting the draw order correct for the common
    # use case of fit lines / curves: draw the intervals in order from largest mean width to
    # smallest mean width, so that the widest intervals are on the bottom.
    interval_grobs = interval_grobs[order(-map_dbl(interval_grobs, "width"))] %>%
      map("grobs") %>%
      reduce(c)

    ggname("geom_interval",
      gTree(children = do.call(gList, interval_grobs))
    )
  }
)
