# A combination of geom_line and geom_ribbon  with sensible defaults for displaying multiple bands
#
# Author: mjskay
###############################################################################


# Names that should be suppressed from global variable check by codetools
# Names used broadly should be put in _global_variables.R
globalVariables(c(".lower", ".upper", ".width"))


#' Line + multiple probability ribbon plots (ggplot geom)
#'
#' A combination of \code{\link{geom_line}} and \code{\link{geom_ribbon}} with default aesthetics
#' designed for use with output from \code{\link{point_interval}}.
#'
#' \code{geom_lineribbon} is a combination version of a \code{\link{geom_line}}, and \code{geom_ribbon} designed for use
#' with output from \code{\link{point_interval}}. This geom sets some default aesthetics equal to the \code{.lower},
#' \code{.upper}, and \code{.width} columns generated by the \code{point_interval} family of functions, making them
#' often more convenient than a vanilla \code{\link{geom_ribbon}} + \code{\link{geom_line}}.
#'
#' Specifically, \code{geom_lineribbon} acts as if its default aesthetics are
#' \code{aes(ymin = .lower, ymax = .upper, size = -.width)}.
#'
#' @param mapping The aesthetic mapping, usually constructed with
#' \code{\link{aes}} or \code{\link{aes_string}}. Only needs to be set at the
#' layer level if you are overriding the plot defaults.
#' @param data A layer specific dataset - only needed if you want to override
#' the plot defaults.
#' @param stat The statistical transformation to use on the data for this layer.
#' @param position The position adjustment to use for overlapping points on this layer.
#' @param ...  Other arguments passed to \code{\link{layer}}.
#' @param na.rm	If \code{FALSE}, the default, missing values are removed with a warning. If \code{TRUE}, missing
#' values are silently removed.
#' @param show.legend Should this layer be included in the legends? \code{NA}, the default, includes if any aesthetics
#' are mapped. \code{FALSE} never includes, and \code{TRUE} always includes.
#' @param inherit.aes If \code{FALSE}, overrides the default aesthetics, rather than combining with them. This is
#' most useful for helper functions that define both data and aesthetics and shouldn't inherit behavior from the
#' default plot specification, e.g. borders.
#' @author Matthew Kay
#' @seealso See \code{\link{stat_lineribbon}} for a version that does summarizing of samples into points and intervals
#' within ggplot. See \code{\link{geom_pointinterval}} / \code{\link{geom_pointintervalh}} for a similar geom intended
#' for point summaries and intervals. See \code{\link{geom_ribbon}} and \code{\link{geom_line}} for the geoms this is
#' based on.
#' @keywords manip
#' @examples
#'
#' library(dplyr)
#' library(ggplot2)
#'
#' data_frame(x = 1:10) %>%
#'   group_by_all() %>%
#'   do(data_frame(y = rnorm(100, .$x))) %>%
#'   median_qi(.width = c(.5, .8, .95)) %>%
#'   ggplot(aes(x = x, y = y)) +
#'   # automatically uses aes(ymin = .lower, ymax = .upper, fill = fct_rev(ordered(.width)))
#'   geom_lineribbon() +
#'   scale_fill_brewer()
#'
#' @importFrom forcats fct_rev
#' @import ggplot2
#' @export
geom_lineribbon <- function(mapping = NULL, data = NULL,
  stat = "identity", position = "identity",
  ...,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE) {

  l = layer(
    data = data,
    mapping = mapping,
    stat = stat,
    geom = GeomLineribbon,
    position = position,
    show.legend = show.legend,
    inherit.aes = inherit.aes,
    params = list(
      na.rm = na.rm,
      ...
    )
  )

  #provide some default computed aesthetics
  default_computed_aesthetics = aes(ymin = .lower, ymax = .upper, fill = forcats::fct_rev(ordered(.width)))

  compute_aesthetics = l$compute_aesthetics
  l$compute_aesthetics = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    compute_aesthetics(data, plot)
  }

  map_statistic = l$map_statistic
  l$map_statistic = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    map_statistic(data, plot)
  }

  l
}

#' @importFrom grid grobTree
draw_key_lineribbon <- function(data, params, size) {
  if (is.na(data$fill)) {
    draw_key_path(data, params, size)
  } else {
    draw_key_rect(data, params, size)
  }
}

#' @rdname tidybayes-ggproto
#' @format NULL
#' @usage NULL
#' @importFrom grid grobName gTree gList
#' @importFrom plyr dlply
#' @importFrom purrr map map_dbl
#' @import ggplot2
#' @export
GeomLineribbon <- ggproto("GeomLineribbon", Geom,
  default_aes = aes(colour = "red", size = 1.25, linetype = 1, shape = 19,
    fill = NA, alpha = NA, stroke = 1),

  draw_key = draw_key_lineribbon,

  required_aes = c("x", "y", "ymin", "ymax"),

  draw_panel = function(data, panel_scales, coord) {
    # ribbons do not autogroup by color/fill/linetype, so if someone groups by changing the color
    # of the line or by setting fill, the ribbons might give an error. So we will do the
    # grouping ourselves
    grouping_columns = names(data) %>%
      intersect(c("colour", "fill", "linetype", "group"))

    # draw all the ribbons
    ribbon_grobs = data %>%
      dlply(grouping_columns, function(d) {
        group_grobs = list(GeomRibbon$draw_panel(transform(d, size = NA), panel_scales, coord))
        list(
          width = d %$% mean(abs(ymax - ymin)),
          grobs = group_grobs
        )
      })

    # this is a slightly hackish approach to getting the draw order correct for the common
    # use case of fit lines / curves: draw the ribbons in order from largest mean width to
    # smallest mean width, so that the widest intervals are on the bottom.
    ribbon_grobs = ribbon_grobs[order(-map_dbl(ribbon_grobs, "width"))] %>%
      map("grobs") %>%
      reduce(c)

    # now draw all the lines
    line_grobs = data %>%
      dlply(grouping_columns, function(d) {
        if (!is.null(d$y)) {
          list(GeomLine$draw_panel(d, panel_scales, coord))
        } else {
          list()
        }
      })

    line_grobs = reduce(line_grobs, c)

    grobs = c(ribbon_grobs, line_grobs)

    ggname("geom_lineribbon",
      gTree(children = do.call(gList, grobs))
    )
  }
)
