% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_expand.R
\name{time_expand}
\alias{time_expand}
\alias{time_complete}
\title{A time based extension to \code{tidyr::complete()}.}
\usage{
time_expand(
  data,
  time = NULL,
  ...,
  .by = NULL,
  time_by = NULL,
  from = NULL,
  to = NULL,
  time_type = c("auto", "duration", "period"),
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1),
  expand_type = c("nesting", "crossing"),
  sort = TRUE,
  keep_class = TRUE,
  roll_month = "preday",
  roll_dst = "pre",
  log_limit = 8
)

time_complete(
  data,
  time = NULL,
  ...,
  .by = NULL,
  time_by = NULL,
  from = NULL,
  to = NULL,
  time_type = c("auto", "duration", "period"),
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1),
  expand_type = c("nesting", "crossing"),
  sort = TRUE,
  keep_class = TRUE,
  fill = NA,
  roll_month = "preday",
  roll_dst = "pre",
  log_limit = 8
)
}
\arguments{
\item{data}{A data frame.}

\item{time}{Time variable.}

\item{...}{Groups to expand.}

\item{.by}{(Optional). A selection of columns to group by for this operation.
Columns are specified using tidy-select.}

\item{time_by}{Time unit. \cr
Must be one of the three:
\itemize{
\item string, specifying either the unit or the number and unit, e.g
\code{time_by = "days"} or \code{time_by = "2 weeks"}
\item named list of length one, the unit being the name, and
the number the value of the list, e.g. \code{list("days" = 7)}.
For the vectorized time functions, you can supply multiple values,
e.g. \code{list("days" = 1:10)}.
\item Numeric vector. If time_by is a numeric vector and x is not a date/datetime,
then arithmetic is used, e.g \code{time_by = 1}.
}}

\item{from}{Time series start date.}

\item{to}{Time series end date.}

\item{time_type}{If "auto", \code{periods} are used for
the time expansion when days, weeks, months or years are specified,
and \code{durations} are used otherwise.}

\item{time_floor}{Should \code{from} be floored to the
nearest unit specified through the \code{time_by}
argument? This is particularly useful for
starting sequences at the beginning of a week or month for example.}

\item{week_start}{day on which week starts following ISO conventions - 1
means Monday (default), 7 means Sunday.
This is only used when \code{floor_date = TRUE}.}

\item{expand_type}{Type of time expansion to use where "nesting"
finds combinations already present in the data,
"crossing" finds all combinations of values in the group variables.}

\item{sort}{Logical. If \code{TRUE} expanded/completed variables are sorted.}

\item{keep_class}{Logical. If \code{TRUE} then the class of the input data is retained.
If \code{FALSE}, which is sometimes faster, a \code{data.table} is returned.}

\item{roll_month}{Control how impossible dates are handled when
month or year arithmetic is involved.
Options are "preday", "boundary", "postday", "full" and "NA".
See \code{?timechange::time_add} for more details.}

\item{roll_dst}{See \code{?timechange::time_add} for the full list of details.}

\item{log_limit}{The maximum log10 number of rows that can be expanded.
Anything exceeding this will throw an error.}

\item{fill}{A named list containing value-name pairs to fill the named implicit missing values.}
}
\value{
A \code{data.frame} of expanded time by or across groups.
}
\description{
A time based extension to \code{tidyr::complete()}.
}
\details{
This works much the same as \code{tidyr::complete()}, except that
you can supply an additional \code{time} argument to allow for filling in time gaps,
expansion of time, as well as aggregating time to a higher unit.
\code{lubridate} is used for handling time, while \code{data.table} and \code{collapse} are used for
the data frame expansion.

At the moment, within group combinations are ignored. This means when \code{expand_type = nesting},
existing combinations of supplied groups across the entire dataset are used, and
when \code{expand_type = crossing}, all possible combinations of supplied groups across the \bold{entire}
dataset are used as well.
}
\examples{
library(timeplyr)
library(dplyr)
library(lubridate)
library(nycflights13)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
x <- flights$time_hour

time_num_gaps(x) # Missing hours

flights_count <- flights \%>\%
  fcount(time_hour)

# Fill in missing hours
flights_count \%>\%
  time_complete(time = time_hour)

# You can specify units too
flights_count \%>\%
  time_complete(time = time_hour, time_by = "hours")
flights_count \%>\%
  time_complete(time = as_date(time_hour), time_by = "days") #  Nothing to complete here

# Where time_expand() and time_complete() really shine is how fast they are with groups
flights \%>\%
  group_by(origin, dest) \%>\%
  time_expand(time = time_hour, time_by = dweeks(1))
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
