% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coersion-tk_ts.R
\name{tk_ts}
\alias{tk_ts}
\alias{tk_ts_}
\title{Coerce time series objects and tibbles with date/date-time columns to ts.}
\usage{
tk_ts(
  data,
  select = NULL,
  start = 1,
  end = numeric(),
  frequency = 1,
  deltat = 1,
  ts.eps = getOption("ts.eps"),
  silent = FALSE
)

tk_ts_(
  data,
  select = NULL,
  start = 1,
  end = numeric(),
  frequency = 1,
  deltat = 1,
  ts.eps = getOption("ts.eps"),
  silent = FALSE
)
}
\arguments{
\item{data}{A time-based tibble or time-series object.}

\item{select}{\strong{Applicable to tibbles and data frames only}.
The column or set of columns to be coerced to \code{ts} class.}

\item{start}{the time of the first observation.  Either a single
    number or a vector of two integers, which specify a natural time
    unit and a (1-based) number of samples into the time unit.  See
    the examples for the use of the second form.}

\item{end}{the time of the last observation, specified in the same way
    as \code{start}.}

\item{frequency}{the number of observations per unit of time.}

\item{deltat}{the fraction of the sampling period between successive
    observations; e.g., 1/12 for monthly data.  Only one of
    \code{frequency} or \code{deltat} should be provided.}

\item{ts.eps}{time series comparison tolerance.  Frequencies are
    considered equal if their absolute difference is less than
    \code{ts.eps}.}

\item{silent}{Used to toggle printing of messages and warnings.}
}
\value{
Returns a \code{ts} object.
}
\description{
Coerce time series objects and tibbles with date/date-time columns to ts.
}
\details{
\code{tk_ts()} is a wrapper for \code{stats::ts()} that is designed
to coerce \code{tibble} objects that have a "time-base" (meaning the values vary with time)
to \code{ts} class objects. There are two main advantages:
\enumerate{
\item Non-numeric columns get removed instead of being populated by NA's.
\item The returned \code{ts} object retains a "timetk index" (and various other attributes) if detected.
The "timetk index" can be used to coerce between \code{tbl}, \code{xts}, \code{zoo}, and \code{ts} data types.
}

The \code{select} argument is used to select subsets
of columns from the incoming data.frame.
Only columns containing numeric data are coerced. \emph{At a minimum, a \code{frequency}
and a \code{start} should be specified.}

For non-data.frame object classes (e.g. \code{xts}, \code{zoo}, \code{timeSeries}, etc) the objects are coerced
using \code{stats::ts()}.

\code{tk_ts_} is a nonstandard evaluation method.
}
\examples{
library(tidyverse)
library(timetk)

### tibble to ts: Comparison between tk_ts() and stats::ts()
data_tbl <- tibble::tibble(
    date = seq.Date(as.Date("2016-01-01"), by = 1, length.out = 5),
    x    = rep("chr values", 5),
    y    = cumsum(1:5),
    z    = cumsum(11:15) * rnorm(1))

# as.ts: Character columns introduce NA's; Result does not retain index
stats::ts(data_tbl[,-1], start = 2016)

# tk_ts: Only numeric columns get coerced; Result retains index in numeric format
data_ts <- tk_ts(data_tbl, start = 2016)
data_ts

# timetk index
tk_index(data_ts, timetk_idx = FALSE)   # Regularized index returned
tk_index(data_ts, timetk_idx = TRUE)    # Original date index returned

# Coerce back to tibble
data_ts \%>\% tk_tbl(timetk_idx = TRUE)


### Using select
tk_ts(data_tbl, select = y)


### NSE: Enables programming
select   <- "y"
tk_ts_(data_tbl, select = select)

}
\seealso{
\code{\link[=tk_index]{tk_index()}}, \code{\link[=tk_tbl]{tk_tbl()}}, \code{\link[=tk_xts]{tk_xts()}}, \code{\link[=tk_zoo]{tk_zoo()}}, \code{\link[=tk_zooreg]{tk_zooreg()}}
}
