% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn-loss.R
\name{nn_triplet_margin_with_distance_loss}
\alias{nn_triplet_margin_with_distance_loss}
\title{Triplet margin with distance loss}
\usage{
nn_triplet_margin_with_distance_loss(
  distance_function = NULL,
  margin = 1,
  swap = FALSE,
  reduction = "mean"
)
}
\arguments{
\item{distance_function}{(callable, optional): A nonnegative, real-valued function that
quantifies the closeness of two tensors. If not specified,
\code{\link[=nn_pairwise_distance]{nn_pairwise_distance()}} will be used.  Default: \code{None}}

\item{margin}{(float, optional): A non-negative margin representing the minimum difference
between the positive and negative distances required for the loss to be 0. Larger
margins penalize cases where the negative examples are not distant enough from the
anchors, relative to the positives. Default: \eqn{1}.}

\item{swap}{(bool, optional): Whether to use the distance swap described in the paper
\href{http://www.bmva.org/bmvc/2016/papers/paper119/index.html}{Learning shallow convolutional feature descriptors with triplet losses} by
V. Balntas, E. Riba et al. If TRUE, and if the positive example is closer to the
negative example than the anchor is, swaps the positive example and the anchor in
the loss computation. Default: \code{FALSE}.}

\item{reduction}{(string, optional): Specifies the (optional) reduction to apply to the output:
\code{'none'} | \code{'mean'} | \code{'sum'}. \code{'none'}: no reduction will be applied,
\code{'mean'}: the sum of the output will be divided by the number of
elements in the output, \code{'sum'}: the output will be summed. Default: \code{'mean'}}
}
\description{
Creates a criterion that measures the triplet loss given input
tensors \eqn{a}, \eqn{p}, and \eqn{n} (representing anchor,
positive, and negative examples, respectively), and a nonnegative,
real-valued function ("distance function") used to compute the relationship
between the anchor and positive example ("positive distance") and the
anchor and negative example ("negative distance").
}
\details{
The unreduced loss (i.e., with \code{reduction} set to \code{'none'})
can be described as:

\deqn{
  \ell(a, p, n) = L = \{l_1,\dots,l_N\}^\top, \quad
l_i = \max \{d(a_i, p_i) - d(a_i, n_i) + {\rm margin}, 0\}
}

where \eqn{N} is the batch size; \eqn{d} is a nonnegative, real-valued function
quantifying the closeness of two tensors, referred to as the \code{distance_function};
and \eqn{margin} is a non-negative margin representing the minimum difference
between the positive and negative distances that is required for the loss to
be 0.  The input tensors have \eqn{N} elements each and can be of any shape
that the distance function can handle.
If \code{reduction} is not \code{'none'}
(default \code{'mean'}), then:

\deqn{
\ell(x, y) =
\begin{array}{ll}
\mbox{mean}(L), &  \mbox{if reduction} = \mbox{`mean';}\\
            \mbox{sum}(L),  &  \mbox{if reduction} = \mbox{`sum'.}
\end{array}
}

See also \code{\link[=nn_triplet_margin_loss]{nn_triplet_margin_loss()}}, which computes the triplet
loss for input tensors using the \eqn{l_p} distance as the distance function.
}
\section{Shape}{

\itemize{
\item Input: \eqn{(N, *)} where \eqn{*} represents any number of additional dimensions
as supported by the distance function.
\item Output: A Tensor of shape \eqn{(N)} if \code{reduction} is \code{'none'}, or a scalar
otherwise.
}
}

\examples{
if (torch_is_installed()) {
# Initialize embeddings
embedding <- nn_embedding(1000, 128)
anchor_ids <- torch_randint(1, 1000, 1, dtype = torch_long())
positive_ids <- torch_randint(1, 1000, 1, dtype = torch_long())
negative_ids <- torch_randint(1, 1000, 1, dtype = torch_long())
anchor <- embedding(anchor_ids)
positive <- embedding(positive_ids)
negative <- embedding(negative_ids)

# Built-in Distance Function
triplet_loss <- nn_triplet_margin_with_distance_loss(
  distance_function = nn_pairwise_distance()
)
output <- triplet_loss(anchor, positive, negative)

# Custom Distance Function
l_infinity <- function(x1, x2) {
  torch_max(torch_abs(x1 - x2), dim = 1)[[1]]
}

triplet_loss <- nn_triplet_margin_with_distance_loss(
  distance_function = l_infinity, margin = 1.5
)
output <- triplet_loss(anchor, positive, negative)

# Custom Distance Function (Lambda)
triplet_loss <- nn_triplet_margin_with_distance_loss(
  distance_function = function(x, y) {
    1 - nnf_cosine_similarity(x, y)
  }
)

output <- triplet_loss(anchor, positive, negative)
}
}
