
#' @export
#' @rdname TVECM.sim
VECM.sim <- function(data,B,VECMobject,  beta, n=200, lag=1, 
                     type=c("simul","boot", "check"),  
                     include = c("const", "trend","none", "both"), 
                     starting=NULL, innov=rmnorm(n,  varcov=varcov), varcov=diag(1,k), 
                     show.parMat=FALSE, seed){
  k<- if(!missing(VECMobject)) VECMobject$k else if(!missing(B)) nrow(B) else if(!missing(data)) ncol(data)
  TVECM.sim(data=data,B=B,TVECMobject=VECMobject, nthresh=0,  beta=beta, n=n, 
            lag=lag, type=type,  include = include, 
            starting=starting, innov=innov, varcov=varcov, 
            show.parMat=show.parMat, seed=seed)
}

#' @export
#' @rdname TVECM.sim
#' @param check When performing a bootstrap replication, check if taking original residuals (instead of resampled) 
#' leads to the original data. 
VECM.boot <- function(VECMobject, show.parMat=FALSE,  seed, check=TRUE){
  if(VECMobject$num_exogen!=0) stop("VECM.boot() does not work for VECM() with exogen variables")
  if(check){
    ch <- TVECM.sim(TVECMobject=VECMobject,  type="check")
    if(!isTRUE(all.equal(as.matrix(ch), as.matrix(VECMobject$model[,1:VECMobject$k]), check.attributes=FALSE)))
      warning("Pseudo Bootstrap was not able to replicate original data, there might be an issue")
  }
  TVECM.sim(TVECMobject=VECMobject,  type="boot", show.parMat=show.parMat, seed=seed)
}

#' @export
#' @rdname TVECM.sim
TVECM.boot <- function(TVECMobject, show.parMat=FALSE,  seed, check=TRUE){
  if(check){
    ch <- TVECM.sim(TVECMobject=TVECMobject,  type="check")
    if(!isTRUE(all.equal(as.matrix(ch), as.matrix(TVECMobject$model[,1:TVECMobject$k]), check.attributes=FALSE)))
      warning("Pseudo Bootstrap was not able to replicate original data, there might be an issue")
  }
  TVECM.sim(TVECMobject=TVECMobject,  type="boot", show.parMat=show.parMat, seed=seed)
}

check.VECM.boot <- function(VECMobject, show.parMat=FALSE,  seed, check=TRUE){
  if(VECMobject$num_exogen!=0) stop("VECM.boot() does not work for VECM() with exogen variables")
  ch <- TVECM.sim(TVECMobject=VECMobject,  type="check")
  res <- isTRUE(all.equal(as.matrix(ch), as.matrix(VECMobject$model[,1:VECMobject$k]), check.attributes=FALSE))
  res
}

check.TVECM.boot <- function(TVECMobject){
  ch <- TVECM.sim(TVECMobject=TVECMobject,  type="check")
  res <- isTRUE(all.equal(as.matrix(ch), as.matrix(TVECMobject$model[,1:TVECMobject$k]), check.attributes=FALSE))
  res
}

as.matrix.ts <-
  function(x, ...)
  {
    # A function implemented by Diethelm Wuertz
    ans = as.matrix.default(unclass(x))
    attr(ans, "tsp")<-NULL
     rownames(ans)<-NULL # colnames(ans)<-NULL
    ans
  }

#'Simulation and bootstrap of bivariate VECM/TVECM
#'
#'Estimate or bootstraps a multivariate Threshold VAR
#'
#'This function offers the possibility to generate series following a
#'VECM/TVECM from two approaches: bootstrap or simulation. \code{VECM.sim} is
#'just a wrapper for \code{\link{TVECM.sim}}.
#'
#'When the argument \code{matrix} is given, on can only simulate a VECM
#'(\code{nthresh}=0) or TVECM (\code{nthresh}=1 or 2). One can have a
#'specification with constant (\code{"const"}), \code{"trend"}, \code{"both"}
#'or \code{"none"} (see argument \code{include}). Order for the parameters is
#'ECT/include/lags for VECM and ECT1/include1/lags1/ECT2/include2/lags2 for
#'TVECM. To be sure that once is using it correctly, setting \code{show.parMat
#'= TRUE} will show the matrix of parameters together with their values and
#'names.
#'
#'The argument \code{beta} is the contegrating value on the right side of the
#'long-run relationship, and hence the function use the vector (1,-beta). The
#'\code{innov} argument specifies the innovations. It should be given as a
#'matrix of dim nxk, (here \var{n} does not include the starting values!), by
#'default it uses a multivariate normal distribution, with covariance matrix
#'specified by \code{varcov}.
#'
#'The starting values (of dim lags x k) can be given through argument
#'\code{starting}. The user should take care for their choice, since it is not
#'sure that the simulated values will cross the threshold even once. Notice
#'that only one cointegrating value is allowed. User interested in simulating a
#'VECM with more cointegrating values should do use the VAR representation and
#'use \code{\link{TVAR.sim}}.
#'
#'The second possibility is to bootstrap series. This is done on a object
#'generated by \code{\link{TVECM}} (or \code{\link{VECM}}). A simple residual
#'bootstrap is done, or one can simulate a series with the same parameter
#'matrix and with normal distributed residuals (with variance pre-specified),
#'corresponding to Monte-carlo simulations.
#'
#'One can alternatively give only the series, and then the function will call
#'internally \code{\link{TVECM}}.
#'
#'@aliases TVECM.sim VECM.sim
#'@param data matrix of parameter to simulate
#'@param B Matrix of coefficients to simulate
#'@param TVECMobject,VECMobject Object computed by function \code{\link{TVECM}}
#'or linear \code{\link{VECM}}
#'@param nthresh number of threshold (see details)
#'@param Thresh The threshold value(s). Vector of length nthresh
#'@param beta The cointegrating value
#'@param n Number of observations to create when type="simul"
#'@param lag Number of lags to include in each regime
#'@param type Whether a bootstrap or simulation is to employ. See details
#'@param include Type of deterministic regressors to include. NOT WORKING
#'PROPERLY CURRENTLY if not const
#'@param starting Starting values when a simulation with given parameter matrix
#'is made
#'@param innov Innovations used for simulation. Should be matrix of dim nxk. By
#'default multivariate normal.
#'@param varcov Variance-covariance matrix for the innovations. By default
#'multivariate normal is used.
#'@param show.parMat Logical. Should the parameter matrix be shown? Useful to
#'understand how to give right input
#'@param seed Optional. Seed for the random number generation.
#'@return A matrix with the simulated/bootstraped series.
#'@author Matthieu Stigler
#'@seealso \code{\link{TVECM}} to estimate a TVECM, \code{\link{VAR.sim}} to
#'simulate/bootstrap a VAR.
#'@keywords ts
#'@export
#'@examples
#'
#'
#'###reproduce example in Enders (2004, 2 edition) p. 350, 
#' # (similar example in Enders (2010, 3 edition) 301-302). 
#'
#'if(require(mnormt)){
#'#see that the full "VAR" coefficient matrix is:
#'  A <- matrix(c(-0.2, 0.2, 0.2, -0.2), byrow=TRUE, ncol=2)
#'
#'# but this is not the input of VECM.sim. You should decompose into the a and b matrix:
#'  a<-matrix(c(-0.2, 0.2), ncol=1)
#'  b<-matrix(c(1,-1), nrow=1)
#'
#'# so that:
#'  a%*%b
#'
#'# The a matrix is the input under argument B, while the b matrix is under argument beta: 
#' # (the other zeros in B are for the not-specified lags)
#'  innov<-rmnorm(100, varcov=diag(2))
#'  startVal <- matrix(0, nrow=2, ncol=1)
#'  Bvecm <- rbind(c(-0.2, 0,0), c(0.2, 0,0))
#'  vecm1 <- VECM.sim(B=Bvecm, beta=1,n=100, lag=1,include="none", innov=innov, starting=startVal)
#'  ECT <- vecm1[,1]-vecm1[,2]
#'
#'#add an intercept as in panel B
#'  Bvecm2 <- rbind(c(-0.2, 0.1,0,0), c(0.2,0.4, 0,0))
#'  vecm2 <- VECM.sim(B=Bvecm2,  n=100,beta=1, lag=1,include="const", innov=innov, starting=startVal)
#'
#'  par(mfrow=c(2,1))
#'  plot(vecm1[,1], type="l", main="Panel a: no drift or intercept", ylab="", xlab="")
#'  lines(vecm1[,2], lty=2)
#'  plot(vecm2[,1], type="l", main="Panel b: drift terms (0.1)", ylab="", xlab="")
#'  lines(vecm2[,2], lty=2)
#'}
#'##Bootstrap a TVAR with 1 threshold (two regimes)
#'data(zeroyld)
#'dat<-zeroyld
#'TVECMobject<-TVECM(dat, nthresh=1, lag=1, ngridBeta=20, ngridTh=20, plot=FALSE)
#'TVECM.sim(TVECMobject=TVECMobject,type="boot")
#'
#'##Check the bootstrap
#' TVECM.sim.check <- TVECM.sim(TVECMobject=TVECMobject,type="check")
#' all(TVECM.sim.check==dat)
#'
TVECM.sim<-function(data,B,TVECMobject, nthresh=1, Thresh, beta, n=200, lag=1, 
                    type=c("simul","boot", "check"),  
                    include = c("const", "trend","none", "both"), 
                    starting=NULL, innov=rmnorm(n, varcov=varcov), varcov=diag(1,k), 
                    show.parMat=FALSE, seed){

  if(!missing(data)&!missing(B))
    stop("You have to provide either B or y, but not both")
  p<-lag
  type<-match.arg(type)
  include<-match.arg(include)
  isMissingB <- missing(B)

###check correct arguments
if(!nthresh%in%c(0,1,2))
  stop("Arg nthresh should  be either 0, 1 or 2")
if(!missing(n)&any(!missing(data), !missing(TVECMobject)))
  stop("arg n should not be given with arg data or TVECMobject")
if(!missing(TVECMobject)&any(!missing(Thresh), !missing(nthresh), !missing(lag)))
  warning("When object TVECMobject is given, only args 'type' and 'round' are relevant, others are not considered")

##include term
  ninc<- switch(include, "none"=0, "const"=1, "trend"=1, "both"=2)
  incVal<- switch(include, "none"=NULL, "const"="const", "trend"="trend", "both"=c("const","trend"))

### possibility 1: only parameters matrix is given
if(!missing(B)){
  if(missing(beta))
    stop("please provide arg beta (cointegrating value)")
  if(type!="simul"){
    type<-"simul"
    warning("Type check or boot are only avalaible with pre specified data. The type simul was used")
  }
  nB<-nrow(B)
  if(nB==1) stop("B matrix should at least have two rows for two variables\n")
  ndig<-4
  esp<-p*nB+1+ninc #number of lags +ecm
  ## naming of variables:
  pa<-switch(as.character(nthresh), "0"="", "1"=c("_low", "_upr"),"2"=c("_low", "_mid","_upr"))
  lags<-as.vector(outer("L{x", 1:nB,  paste, sep=""))
  lags2<-paste(rep(lags, times=p),"}{", rep(1:p,each=p),"}",sep="")
  if(esp*(nthresh+1)!=ncol(B)){
    colnames_Matrix_input<-as.vector(outer(c("ECT",incVal, lags2), pa, paste, sep=""))
    cat("Matrix B badly specified: should have ", esp*(nthresh+1), "columns, but has", ncol(B), "\n")
    print(matrix(NA, nrow=nB, ncol=esp*(nthresh+1), dimnames=list(paste("Equ x", 1:nB, sep=""), colnames_Matrix_input)))
    stop()
  }
  rownames(B)<- paste("Equ x", 1:nB, ":",sep="")
  y<-matrix(0,ncol=nB, nrow=n)
  if(!is.null(starting)){
    if(all(dim(as.matrix(starting))==c(nB,p)))
      y[seq_len(p),]<-starting
    else
	stop("Bad specification of starting values. Should have nrow = lag and ncol = number of variables")
  }
    Bmat<-B
    k <- ncol(y) 		#Number of variables
    T <- nrow(y) 		#Size of start sample
  if(is.vector(beta)){
    if(length(beta)==k-1) beta <- c(1, -beta)
    tBETA<-matrix(beta, nrow=1)
    r <- 1
  } else {
    if(nrow(beta)!=k) stop("beta should have k rows and r cols")
    r <- ncol(beta)
    tBETA <- t(beta)
  }
}

### possibility 2: only data is given: compute it with linear or selectSETAR
else if(!missing(data)){
  if(nthresh==0){
    TVECMobject<-lineVar(data, lag=p, include=include, model="VECM")
  } else { 
    if(!missing(Thresh)){
      if(nthresh==1) {
        TVECMobject<-TVECM(data, lag=p, include=include, nthresh=nthresh, plot=FALSE, trace=FALSE, th1=list(exact=Thresh))
      } else if(nthresh==2){
        TVECMobject<-TVECM(data, lag=p, include=include, nthresh=nthresh, plot=FALSE, trace=FALSE, th1=list(exact=Thresh[1]),th2=list(exact=Thresh[2]))
      }
    } else {
      TVECMobject<-TVECM(data, lag=p, include=include, nthresh=nthresh, plot=FALSE, trace=FALSE)
    }
  }
}
### possibility 3: setarobject is given by user (or by poss 2)
if(!missing(TVECMobject)){
  k<-TVECMobject$k
  T<-TVECMobject$T
  p<-TVECMobject$lag
  include<-TVECMobject$include
  if(include %in% c("trend", "both"))
    warning(paste("Accuracy of function (tested with arg type=check) is not good when arg include=",include," is given\n"))
  modSpe<-TVECMobject$model.specific
  LRinclude <- modSpe$LRinclude
  nthresh <- modSpe$nthresh
  if(nthresh>0 &&modSpe$model=="only_ECT") stop("TVECM.sim() does not work for 'common=only_ECT'")
  if(LRinclude!="none") stop("TVECM.sim() does not work for 'LRinclude!='none'")
  beta<- -modSpe$coint[2,1]
  tBETA <- t(modSpe$coint)
  r <- modSpe$r
  res<-residuals(TVECMobject)
  Bmat<-coefMat(TVECMobject)
  y<-as.matrix(TVECMobject$model)[,1:k]
  ndig<-getndp(y[,1])
  if(nthresh>0){
    Thresh<-modSpe$Thresh
    nthresh<-modSpe$nthresh
  }
}

t <- T-p-1 		#Size of end sample
npar<-k*(p+ninc+1)

##### put coefficients vector in right form according to arg include (arg both need no modif)
  if(include!="both"){
    aa1 <- r+switch(include, "none"=1:2, "const"=2, "trend"=1, "both"=NULL)
    aa <- sort(rep(aa1, each=nthresh+1)+ (0:nthresh)*(p*k+max(aa1)))
    Bmat<-myInsertCol(Bmat, c=aa, 0)
  }
  nparBmat<-p*k+2+1

  
##############################
###Reconstitution boot/simul
##############################
#initial values

#initial values
Yb<-matrix(0, nrow=nrow(y), ncol=k)
Yb[1:(p+1),]<-y[1:(p+1),]		

trend<-c(rep(NA, T-t),1:t)

#resampling/ simulation of residual/innovations
if(type=="simul"&&dim(innov)!=c(n,k))
  stop(paste("input innov is not of right dim, should be matrix with", n,"rows and ", k, "cols\n"))
if(!missing(seed)) set.seed(seed)
resids<-switch(type, "boot"=res[sample(seq_len(t), replace=TRUE),], "simul"= innov, "check"=res)
resb<-rbind(matrix(0,nrow=p+1, ncol=k),resids)

if(nthresh==0){
  for(i in (p+2):T){
    ECT<-Bmat[,1:r]%*%tBETA%*%matrix(Yb[i-1,], ncol=1)
    Yb[i,]<-rowSums(cbind(Yb[i-1,],Bmat[,r+1], Bmat[,r+2]*trend[i], ECT,Bmat[,-c(1:(r+2))]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
  }
} else if(nthresh==1){
  BD<-Bmat[,seq_len(nparBmat)]
  BU<-Bmat[,-seq_len(nparBmat)]
  
  for(i in (p+2):(nrow(y))){
    ECT<-tBETA%*%matrix(Yb[i-1,], ncol=1)
    if(round(ECT,ndig)<=Thresh){
      Yb[i,]<-rowSums(cbind(Yb[i-1,],BD[,1]%*%ECT, BD[,2], BD[,3]*trend[i],BD[,-c(1,2,3)]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
    }  else{
      Yb[i,]<-rowSums(cbind(Yb[i-1,],BU[,1]%*%ECT, BU[,2], BU[,3]*trend[i],BU[,-c(1,2,3)]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
    }
  }
} else if(nthresh==2){
  BD <- Bmat[,seq_len(nparBmat)]
  BM <- Bmat[,seq_len(nparBmat)+nparBmat]
  BU <- Bmat[,seq_len(nparBmat)+2*nparBmat]
  for(i in (p+2):(nrow(y))){
    ECT<-tBETA%*%matrix(Yb[i-1,], ncol=1)
    if(round(ECT,ndig)<=Thresh[1]){ 
      Yb[i,]<-rowSums(cbind(Yb[i-1,],BD[,1]%*%ECT,BD[,2], BD[,3]*trend[i], BD[,-c(1,2,3)]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
    } else if(round(ECT,ndig)>Thresh[2]) {
      Yb[i,]<-rowSums(cbind(Yb[i-1,],BU[,1]%*%ECT,BU[,2], BU[,3]*trend[i],BU[,-c(1,2,3)]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
    } else{
      Yb[i,]<-rowSums(cbind(Yb[i-1,],BM[,1]%*%ECT,BM[,2], BM[,3]*trend[i],BM[,-c(1,2,3)]%*%matrix(t(Yb[i-c(1:p),]-Yb[i-c(2:(p+1)),]), ncol=1),resb[i,]))
    }
  }
}

if(show.parMat){
  if(!isMissingB){
    colnames_Matrix_system<-as.vector(outer(c("ECT","Const", "Trend", lags2), pa, paste, sep=""))
    colnames(Bmat)<- colnames_Matrix_system
  } else if(include!="both"){
    add <- switch(include, "const"="Trend", "trend"="Const", "none"=c("Const", "Trend"))
    colnames(Bmat)[aa] <- rep(add, nthresh+1)
  }
  print(Bmat)
}
res<-round(Yb, ndig)
return(res)
}

if(FALSE){
library(tsDyn)
environment(TVECM.sim)<-environment(star)

##Simulation of a TVAR with 1 threshold
B<-rbind(c(-0.2, 0,0), c(0.2, 0,0))
a<-TVECM.sim(B=B, nthresh=0, beta=1, lag=1,include="none", starting=c(2,2))
ECT<-a[,1]-a[,2]

layout(matrix(1:2, ncol=1))
plot(a[,1], type="l", xlab="", ylab="", ylim=range(a, ECT))
lines(a[,2], col=2, type="l")

plot(ECT, type="l")

B<-rbind(c(0.2, 0.11928245, 1.00880447, -0.009974585, 0.3, -0.089316, 0.95425564, 0.02592617),c( -0.1, 0.25283578, 0.09182279,  0.914763741, 0.35,-0.0530613, 0.02248586, 0.94309347))
sim<-TVECM.sim(B=B,beta=1, nthresh=1,n=500, type="simul",Thresh=5, starting=c(5.2, 5.5))
#estimate the new serie
TVECM(sim, lag=1)

##Bootstrap a TVAR with two threshold (three regimes)
#data(zeroyld)
dat<-zeroyld
TVECMobject<-TVECM(dat, lag=1, nthresh=2, plot=FALSE, trace=FALSE, th1=list(exact=-1),th2=list(exact=1))
TVECMobject<-TVECM(dat, lag=1, nthresh=2)#, plot=FALSE, trace=FALSE, th1=list(exact=7),th2=list(exact=9))

TVECM.sim(data=dat,nthresh=2, type="boot", Thresh=c(7,9))

##Check the bootstrap
linObject<-lineVar(dat, lag=1, model="VECM")
all(TVECM.sim(TVECMobject=linObject,type="check")==dat)
all(TVECM.sim(TVECMobject=lineVar(dat, lag=1, model="VECM", include="none"),type="check")==dat)

#not working: (probably trend coefficients too small so digits errors)
all(TVECM.sim(TVECMobject=lineVar(dat, lag=1, model="VECM", include="trend"),type="check")==dat)
all(TVECM.sim(TVECMobject=lineVar(dat, lag=1, model="VECM", include="both"),type="check")==dat)

#nthresh=1
TVECMobject<-TVECM(dat, nthresh=1, lag=1, ngridBeta=20, ngridTh=20, plot=FALSE)
all(TVECM.sim(TVECMobject=TVECMobject,type="check")==dat)

all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=1, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE),type="check")==dat)
all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=1, lag=1, ngridBeta=20, ngridTh=20, plot=FALSE, include="none"),type="check")==dat)
all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=1, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE, include="none"),type="check")==dat)

#nthresh=2
TVECMobject2<-TVECM(dat, nthresh=2, lag=1, ngridBeta=20, ngridTh=20, plot=FALSE)
all(TVECM.sim(TVECMobject=TVECMobject2,type="check")==dat)
all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=2, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE),type="check")==dat)

all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=2, lag=1, ngridBeta=20, ngridTh=20, plot=FALSE, include="none"),type="check")==dat) 
#famous rounding problem...
all(TVECM.sim(TVECMobject=TVECM(dat, nthresh=2, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE, include="none"),type="check")==dat)

###TODO:
#improve trend/both case
#TVECM: possibility to give args!
TVECM(dat, nthresh=1, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE, th1=list(exact=-1.4),include="none")
TVECM(dat, nthresh=1, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE, th1=list(exact=-1.4),beta=list(exact=1),include="none")
TVECM(dat, nthresh=2, lag=2, ngridBeta=20, ngridTh=20, plot=FALSE, th1=list(exact=-1.4),th2=list(exact=0.5),include="none")
}
