get_vars <- function(inp){
  if(is.list(inp)){
    out <- unique(unlist(lapply(inp, all.vars)))
  } else {
    out <- all.vars(inp)
  }
  names(out) <- out
  out
}

var_data <- function(var, guide, n){
  out <- rep(NA, n)
  gv <- guide[[var]]
  if(is.null(gv)){
    out <- stats::rnorm(n, 0, 1)
  } else if(inherits(gv, "factor")){
    levs <- levels(gv)
    out <- factor(sample(levs, n, replace=TRUE), levels=levs)
  } else{
    gv$n <- n
    out <- do.call(gv$dist, gv[!names(gv)=="dist"])
  }
  out
}

generate_data <- function(formulas, guide, n){
  vars <- get_vars(formulas)
  if(length(vars)==0) return(NULL)
  as.data.frame(lapply(vars, var_data, guide=guide, n=n))
}

capitalize <- function(inp){
  paste0(toupper(substring(inp,1,1)),
         substring(inp,2,nchar(inp)))
}

parse_func_name <- function(inp){
  if(!is.character(inp)){
    stop("Argument must be a character string", call.=FALSE)
  }
  capitalize(inp)
}

blank_umFit <- function(fit_function){
  type <- parse_func_name(fit_function)
  type <- ifelse(type=="Pcount", "PCount", type)
  type <- ifelse(type=="MultinomPois", "MPois", type)
  type <- ifelse(type=="Distsamp", "DS", type)
  type <- ifelse(type=="Colext", "ColExt", type)
  type <- ifelse(type=="Gdistsamp", "GDS", type)
  type <- ifelse(type=="Gpcount", "GPC", type)
  type <- ifelse(type=="Gmultmix", "GMM", type)
  type <- ifelse(type=="PcountOpen", "PCO", type)
  type <- ifelse(type=="DistsampOpen", "DSO", type)
  type <- ifelse(type=="MultmixOpen", "MMO", type)
  type <- ifelse(type=="Gdistremoval", "GDR", type)
  type <- paste0("unmarkedFit", type)
  new(type)
}


setGeneric("get_umf_components", function(object, ...) standardGeneric("get_umf_components"))

setMethod("get_umf_components", "unmarkedFit",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$state, guide, design$M)
  oc <- generate_data(formulas$det, guide, design$J*design$M)
  yblank <- matrix(0, design$M, design$J)
  list(y=yblank, siteCovs=sc, obsCovs=oc)
})


setGeneric("simulate_fit", function(object, ...) standardGeneric("simulate_fit"))

setMethod("simulate_fit", "unmarkedFitOccu",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  umf <- unmarkedFrameOccu(y=parts$y, siteCovs=parts$siteCovs,
                           obsCovs=parts$obsCovs)
  occu(as.formula(paste(deparse(formulas$det), deparse(formulas$state))),
       data=umf, se=FALSE, control=list(maxit=1))
})

setMethod("simulate_fit", "unmarkedFitPCount",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  umf <- unmarkedFramePCount(y=parts$y, siteCovs=parts$siteCovs,
                             obsCovs=parts$obsCovs)
  args <- list(...)
  K <- ifelse(is.null(args$K), 100, args$K)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  pcount(as.formula(paste(deparse(formulas$det), deparse(formulas$state))),
       data=umf, mixture=mixture, K=K, se=FALSE, control=list(maxit=1))
})

setMethod("simulate_fit", "unmarkedFitOccuRN",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  umf <- unmarkedFrameOccu(y=parts$y, siteCovs=parts$siteCovs,
                           obsCovs=parts$obsCovs)
  occuRN(as.formula(paste(deparse(formulas$det), deparse(formulas$state))),
       data=umf, se=FALSE, control=list(maxit=1))
})


setMethod("simulate", "character",
  function(object, nsim=1, seed=NULL, formulas, coefs=NULL, design, guide=NULL, ...){
  model <- blank_umFit(object)
  fit <- suppressWarnings(simulate_fit(model, formulas, guide, design, ...))
  coefs <- check_coefs(coefs, fit)
  fit <- replace_estimates(fit, coefs)
  ysims <- simulate(fit, nsim)
  umf <- fit@data
  # fix this
  umfs <- lapply(ysims, function(x){
    if(object=="occuMulti"){
      umf@ylist <- x
    } else if(object=="gdistremoval"){
      umf@yDistance=x$yDistance
      umf@yRemoval=x$yRemoval
    } else {
      umf@y <- x
    }
    umf
  })
  if(length(umfs)==1) umfs <- umfs[[1]]
  umfs
})

setMethod("get_umf_components", "unmarkedFitMPois",
          function(object, formulas, guide, design, ...){
  args <- list(...)
  sc <- generate_data(formulas$state, guide, design$M)
  oc <- generate_data(formulas$det, guide, design$J*design$M)
  J <- ifelse(args$type=="double", 3, design$J)
  yblank <- matrix(0, design$M, design$J)
  list(y=yblank, siteCovs=sc, obsCovs=oc)
})

setMethod("simulate_fit", "unmarkedFitMPois",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  type <- ifelse(is.null(args$type), "removal", args$type)
  umf <- unmarkedFrameMPois(y=parts$y, siteCovs=parts$siteCovs,
                           obsCovs=parts$obsCovs, type=type)
  multinomPois(as.formula(paste(deparse(formulas$det), deparse(formulas$state))),
               data=umf, se=FALSE, control=list(maxit=1))
})

setMethod("get_umf_components", "unmarkedFitDS",
          function(object, formulas, guide, design, ...){
  #args <- list(...)
  sc <- generate_data(formulas$state, guide, design$M)
  sc2 <- generate_data(formulas$det, guide, design$M)
  dat <- list(sc, sc2)
  keep <- sapply(dat, function(x) !is.null(x))
  dat <- dat[keep]
  sc <- do.call(cbind, dat)
  yblank <- matrix(0, design$M, design$J)
  list(y=yblank, siteCovs=sc)
})

setMethod("simulate_fit", "unmarkedFitDS",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(is.null(args$tlength)) args$tlength <- 0
  umf <- unmarkedFrameDS(y=parts$y, siteCovs=parts$siteCovs,
                         tlength=args$tlength, survey=args$survey, unitsIn=args$unitsIn,
                         dist.breaks=args$dist.breaks)
  keyfun <- ifelse(is.null(args$keyfun), "halfnorm", args$keyfun)
  output <- ifelse(is.null(args$output), "density", args$output)
  unitsOut <- ifelse(is.null(args$unitsOut), "ha", args$unitsOut)

  distsamp(as.formula(paste(deparse(formulas$det), deparse(formulas$state))),
               data=umf, se=FALSE, control=list(maxit=1), keyfun=keyfun,
          output=output, unitsOut=unitsOut)
})


setMethod("get_umf_components", "unmarkedFitColExt",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$psi, guide, design$M)
  ysc <- generate_data(list(formulas$col, formulas$ext), guide, design$M*design$T)
  oc <- generate_data(formulas$det, guide, design$J*design$M*design$T)
  yblank <- matrix(0, design$M, design$T*design$J)
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc, obsCovs=oc)
})


setMethod("simulate_fit", "unmarkedFitColExt",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  umf <- unmarkedMultFrame(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           obsCovs=parts$obsCovs, numPrimary=design$T)
  colext(psiformula=formulas$psi, gammaformula=formulas$col,
         epsilonformula=formulas$ext,pformula=formulas$det,
         data=umf, se=FALSE, control=list(maxit=1))
})

setMethod("get_umf_components", "unmarkedFitOccuTTD",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$psi, guide, design$M)
  ysc <- NULL
  if(design$T>1){
    ysc <- generate_data(list(formulas$col, formulas$ext), guide, design$M*design$T)
  }
  oc <- generate_data(formulas$det, guide, design$J*design$M*design$T)
  yblank <- matrix(0, design$M, design$T*design$J)
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc, obsCovs=oc)
})


setMethod("simulate_fit", "unmarkedFitOccuTTD",
  function(object, formulas, guide, design, ...){
  if(is.null(design$T)) design$T <- 1
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  umf <- unmarkedFrameOccuTTD(y=parts$y,
                              surveyLength=args$surveyLength,
                              siteCovs=parts$siteCovs,
                              yearlySiteCovs=parts$yearlySiteCovs,
                              obsCovs=parts$obsCovs, numPrimary=design$T)
  linkPsi <- ifelse(is.null(args$linkPsi), "logit", args$linkPsi)
  ttdDist <- ifelse(is.null(args$ttdDist), "exp", args$ttdDist)
  occuTTD(psiformula=formulas$psi, gammaformula=formulas$col,
         epsilonformula=formulas$ext,detformula=formulas$det,
         linkPsi=linkPsi, ttdDist=ttdDist,
         data=umf, se=FALSE, control=list(maxit=1))
})


setMethod("get_umf_components", "unmarkedFitGMM",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$lambda, guide, design$M)
  ysc <- generate_data(formulas$phi, guide, design$M*design$T)
  yblank <- matrix(0, design$M, design$T*design$J)
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc)
})


setMethod("simulate_fit", "unmarkedFitGDS",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(args$survey=="line"){
    umf <- unmarkedFrameGDS(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           numPrimary=design$T,
                           tlength=args$tlength, survey=args$survey,
                           unitsIn=args$unitsIn, dist.breaks=args$dist.breaks)
  } else if(args$survey=="point"){
    umf <- unmarkedFrameGDS(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           numPrimary=design$T, survey=args$survey,
                           unitsIn=args$unitsIn, dist.breaks=args$dist.breaks)
  }

  keyfun <- ifelse(is.null(args$keyfun), "halfnorm", args$keyfun)
  output <- ifelse(is.null(args$output), "density", args$output)
  unitsOut <- ifelse(is.null(args$unitsOut), "ha", args$unitsOut)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  K <- ifelse(is.null(args$K), 100, args$K)

  gdistsamp(lambdaformula=formulas$lambda, phiformula=formulas$phi,
            pformula=formulas$det, data=umf, keyfun=keyfun, output=output,
            unitsOut=unitsOut, mixture=mixture, K=K,
            se=FALSE, control=list(maxit=1))
})

setMethod("simulate_fit", "unmarkedFitGPC",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  umf <- unmarkedFrameGPC(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           numPrimary=design$T)
  K <- ifelse(is.null(args$K), 100, args$K)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)

  gpcount(lambdaformula=formulas$lambda, phiformula=formulas$phi,
          pformula=formulas$det, data=umf, mixture=mixture, K=K,
          se=FALSE, control=list(maxit=1))
})


setMethod("simulate_fit", "unmarkedFitGMM",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  umf <- unmarkedFrameGMM(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           numPrimary=design$T, type=args$type)
  K <- ifelse(is.null(args$K), 100, args$K)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)

  gmultmix(lambdaformula=formulas$lambda, phiformula=formulas$phi,
           pformula=formulas$det, data=umf, mixture=mixture, K=K,
           se=FALSE, control=list(maxit=1))
})


setMethod("get_umf_components", "unmarkedFitDailMadsen",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$lambda, guide, design$M)
  ysc <- generate_data(list(formulas$gamma, formulas$omega), guide, design$M*design$T)
  oc <- generate_data(formulas$det, guide, design$M*design$T*design$J)
  yblank <- matrix(0, design$M, design$T*design$J)
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc, obsCovs=oc)
})

setMethod("simulate_fit", "unmarkedFitPCO",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(is.null(args$primaryPeriod)){
    args$primaryPeriod <- matrix(1:design$T, design$M, design$T, byrow=TRUE)
  }
  umf <- unmarkedFramePCO(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           numPrimary=design$T, primaryPeriod=args$primaryPeriod)
  K <- ifelse(is.null(args$K), 100, args$K)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  dynamics <- ifelse(is.null(args$dynamics), "constant", args$dynamics)
  fix <- ifelse(is.null(args$fix), "none", args$fix)
  immigration <- ifelse(is.null(args$immigration), FALSE, args$immigration)
  iotaformula <- args$iotaformula
  if(is.null(iotaformula)) iotaformula <- ~1

  pcountOpen(lambdaformula=formulas$lambda, gammaformula=formulas$gamma,
             omegaformula=formulas$omega, pformula=formulas$det,
             data=umf, mixture=mixture, K=K, dynamics=dynamics, fix=fix,
             se=FALSE, method='SANN', control=list(maxit=1), immigration=immigration,
             iotaformula=iotaformula)
})

setMethod("simulate_fit", "unmarkedFitMMO",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(is.null(args$primaryPeriod)){
    args$primaryPeriod <- matrix(1:design$T, design$M, design$T, byrow=TRUE)
  }
  umf <- unmarkedFrameMMO(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           type=args$type,
                           numPrimary=design$T, primaryPeriod=args$primaryPeriod)
  K <- ifelse(is.null(args$K), 100, args$K)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  dynamics <- ifelse(is.null(args$dynamics), "constant", args$dynamics)
  fix <- ifelse(is.null(args$fix), "none", args$fix)
  immigration <- ifelse(is.null(args$immigration), FALSE, args$immigration)
  iotaformula <- args$iotaformula
  if(is.null(iotaformula)) iotaformula <- ~1

  multmixOpen(lambdaformula=formulas$lambda, gammaformula=formulas$gamma,
             omegaformula=formulas$omega, pformula=formulas$det,
             data=umf, mixture=mixture, K=K, dynamics=dynamics, fix=fix,
             se=FALSE, method='SANN', control=list(maxit=1), immigration=immigration,
             iotaformula=iotaformula)
})

setMethod("get_umf_components", "unmarkedFitDSO",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(formulas$lambda, guide, design$M)
  ysc <- generate_data(list(formulas$gamma, formulas$omega, formulas$det),
                       guide, design$M*design$T)
  yblank <- matrix(0, design$M, design$T*design$J)
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc)
})

setMethod("simulate_fit", "unmarkedFitDSO",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(is.null(args$primaryPeriod)){
    args$primaryPeriod <- matrix(1:design$T, design$M, design$T, byrow=TRUE)
  }
  if(args$survey=="line"){
  umf <- unmarkedFrameDSO(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,
                           tlength=args$tlength, survey=args$survey,
                           unitsIn=args$unitsIn, dist.breaks=args$dist.breaks,
                           numPrimary=design$T, primaryPeriod=args$primaryPeriod)
  } else if(args$survey == "point"){
  umf <- unmarkedFrameDSO(y=parts$y, siteCovs=parts$siteCovs,
                           yearlySiteCovs=parts$yearlySiteCovs,survey=args$survey,
                           unitsIn=args$unitsIn, dist.breaks=args$dist.breaks,
                           numPrimary=design$T, primaryPeriod=args$primaryPeriod)
  }
  K <- ifelse(is.null(args$K), 100, args$K)
  keyfun <- ifelse(is.null(args$keyfun), "halfnorm", args$keyfun)
  output <- ifelse(is.null(args$output), "density", args$output)
  unitsOut <- ifelse(is.null(args$unitsOut), "ha", args$unitsOut)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  dynamics <- ifelse(is.null(args$dynamics), "constant", args$dynamics)
  fix <- ifelse(is.null(args$fix), "none", args$fix)
  immigration <- ifelse(is.null(args$immigration), FALSE, args$immigration)
  iotaformula <- args$iotaformula
  if(is.null(iotaformula)) iotaformula <- ~1
  distsampOpen(lambdaformula=formulas$lambda, gammaformula=formulas$gamma,
             omegaformula=formulas$omega, pformula=formulas$det,
             keyfun=keyfun, unitsOut=unitsOut, output=output,
             data=umf, mixture=mixture, K=K, dynamics=dynamics, fix=fix,
             se=FALSE, method='SANN', control=list(maxit=1), immigration=immigration,
             iotaformula=iotaformula)
})


setMethod("get_umf_components", "unmarkedFitOccuMulti",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(lapply(formulas$state, as.formula), guide, design$M)
  oc <- generate_data(lapply(formulas$det, as.formula), guide, design$J*design$M)
  nspecies <- length(formulas$det)
  yblank <- lapply(1:nspecies, function(x) matrix(0, design$M, design$J))
  list(y=yblank, siteCovs=sc, obsCovs=oc)
})

setMethod("simulate_fit", "unmarkedFitOccuMulti",
  function(object, formulas, guide, design, ...){
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  if(is.null(args$maxOrder)) args$maxOrder <- length(parts$y)
  umf <- unmarkedFrameOccuMulti(y=parts$y, siteCovs=parts$siteCovs,
                                obsCovs=parts$obsCovs, maxOrder=args$maxOrder)
  occuMulti(formulas$det, formulas$state, data=umf, maxOrder=args$maxOrder,
            se=FALSE, control=list(maxit=1))
})

setMethod("get_umf_components", "unmarkedFitOccuMS",
          function(object, formulas, guide, design, ...){
  sc <- generate_data(lapply(formulas$state, as.formula), guide, design$M)
  ysc <- NULL
  if(!is.null(formulas$phi)){
    ysc <- generate_data(lapply(formulas$phi, as.formula), guide, design$M*design$T*design$J)
  }
  oc <- generate_data(lapply(formulas$det, as.formula), guide, design$J*design$M)
  nspecies <- length(formulas$det)
  yblank <- matrix(0, design$M, design$T*design$J)
  yblank[1,1] <- 2 # To bypass sanity checker in unmarkedFrameOccuMS
  list(y=yblank, siteCovs=sc, yearlySiteCovs=ysc, obsCovs=oc)
})

setMethod("simulate_fit", "unmarkedFitOccuMS",
  function(object, formulas, guide, design, ...){
  if(is.null(design$T)) design$T <- 1
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  umf <- unmarkedFrameOccuMS(y=parts$y, siteCovs=parts$siteCovs,
                                yearlySiteCovs=parts$yearlySiteCovs,
                                obsCovs=parts$obsCovs, numPrimary=design$T)
  if(is.null(args$parameterization)) args$parameterization <- "multinomial"
  occuMS(formulas$det, formulas$state, formulas$phi, data=umf,
         parameterization=args$parameterization,
         se=FALSE, control=list(maxit=1))
})

setMethod("get_umf_components", "unmarkedFitGDR",
          function(object, formulas, guide, design, ...){
  if(any(! c("M","Jdist","Jrem") %in% names(design))){
    stop("Required design components are M, Jdist, and Jrem")
  }
  sc <- generate_data(list(formulas$lambda, formulas$dist), guide, design$M)
  ysc <- NULL
  if(design$T > 1){
    ysc <- generate_data(formulas$phi, guide, design$M*design$T)
  }
  oc <- generate_data(formulas$rem, guide, design$M*design$T*design$Jrem)

  list(yDistance=matrix(0, design$M, design$T*design$Jdist),
       yRemoval=matrix(0, design$M, design$T*design$Jrem),
       siteCovs=sc, yearlySiteCovs=ysc, obsCovs=oc)
})

setMethod("simulate_fit", "unmarkedFitGDR",
  function(object, formulas, guide, design, ...){
  if(is.null(design$T)) design$T <- 1
  if(is.null(formulas$phi)) formulas$phi <- ~1
  parts <- get_umf_components(object, formulas, guide, design, ...)
  args <- list(...)
  umf <- unmarkedFrameGDR(yDistance=parts$yDistance, yRemoval=parts$yRemoval,
                          numPrimary=design$T, siteCovs=parts$siteCovs,
                          obsCovs=parts$obsCovs, yearlySiteCovs=parts$yearlySiteCovs,
                          dist.breaks=args$dist.breaks, unitsIn=args$unitsIn,
                          period.lengths=args$period.lengths)

  keyfun <- ifelse(is.null(args$keyfun), "halfnorm", args$keyfun)
  output <- ifelse(is.null(args$output), "density", args$output)
  unitsOut <- ifelse(is.null(args$unitsOut), "ha", args$unitsOut)
  mixture <- ifelse(is.null(args$mixture), "P", args$mixture)
  K <- ifelse(is.null(args$K), 100, args$K)

  gdistremoval(lambdaformula=formulas$lambda, phiformula=formulas$phi,
               removalformula=formulas$rem, distanceformula=formulas$dist,
               data=umf, keyfun=keyfun, output=output, unitsOut=unitsOut,
               mixture=mixture, K=K, se=FALSE, control=list(maxit=1))
})
