#' Cairo wrapper function
#'
#' Cairo wrapper function returning NULL if not \code{type} is specified
#'
#' \code{openGraphCairo()} \code{Cairo()} wrapper function. Differences to \code{Cairo:}
#' a) prematurely ends the function call to \code{Cairo()} returning NULL, if no output \code{type}  of types "png", "jpeg", "pdf", "svg", "ps" or "tiff"
#' is provided.
#' b) 
#' The \code{file} argument of the underlying Cairo function is generated by \code{file.path(fileDirectory,paste(fileName,".", type, sep = ""))}.

#' @param width see \code{Cairo()}
#' @param height see \code{Cairo()}
#' @param fileName name of file to be created. Does not include both file extension ".\code{type}" and  file \code{filedirectory}. Default file name "visstat_plot". 
#' @param type Supported output types are "png", "jpeg", "pdf", "svg", "ps" and "tiff". See \code{Cairo()}
#' @param fileDirectory path of directory, where plot is stored. Default current working directory.
#' @param pointsize see \code{Cairo()}
#' @param bg see \code{Cairo()}
#' @param canvas see \code{Cairo()}
#' @param units see \code{Cairo()}
#' @param dpi DPI used for the conversion of units to pixels. Default value 150. 
#' @name openGraphCairo
#' @return NULL, if no \code{type} is specified. Otherwise see \code{Cairo()}
#' @examples
#'
#' ##  adapted from example in \code{Cairo()}
#' openGraphCairo(fileName="normal_dist",type="pdf", fileDirectory=tempdir()) 
#' plot(rnorm(4000),rnorm(4000),col="#ff000018",pch=19,cex=2) 
#' dev.off() # creates a file "normal_dist.pdf" in the directory specified in fileDirectory
#  ## remove the plot from fileDirectory
#' file.remove(file.path(tempdir(),"normal_dist.pdf"))


#' @import Cairo
#' 
#' @export openGraphCairo

openGraphCairo =  function(width = 640,
                           height = 480 , fileName = NULL, type = NULL, fileDirectory=getwd(), pointsize = 12,
                           bg = "transparent", canvas = "white", units = "px",
                           dpi=150) 
  
  
  
{
  oldparCairo <- par(no.readonly = TRUE)  
  oldparCairo$new=FALSE
  on.exit(par(oldparCairo))
  
  if (is.null(type))
  {return()}
  
  else{
    
   
    #set default fileName to "visstat_plot"
    if (is.null(fileName)) {
      fileName = "visstat_plot"}
    
    fullfilename = paste(fileName,".", type, sep = "")
    Cairofilename = file.path(fileDirectory,fullfilename)
    
    if (type == "png") {
      CairoPNG(filename = Cairofilename)  
    }else if (type == "pdf") {
      CairoPDF(file = Cairofilename)    
    }else if (type == "jpeg") {
      CairoJPEG(filename = Cairofilename)    
    }else if (type == "tiff") {
      CairoTIFF(filename = Cairofilename)     
    }else if (type == "svg") { 
      CairoSVG(file = Cairofilename) 
    }else if (type == "ps") { 
      CairoPS(file=Cairofilename,family = "Helvetica")
  
    }else{
      warning("Chosen output type not supported. No graphics saved.")
      return()
    }  
    
  }   
}


