% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eocusum_sim.R
\name{eocusum_crit_sim}
\alias{eocusum_crit_sim}
\title{Compute alarm threshold of EO-CUSUM control charts using simulation}
\usage{
eocusum_crit_sim(L0, pmix, k, RQ = 1, side = "low", yemp = FALSE,
  m = 10000, nc = 1, hmax = 30, jmax = 4, verbose = FALSE)
}
\arguments{
\item{L0}{Double. Prespecified in-control Average Run Length.}

\item{pmix}{Data Frame. A three column data frame. First column is the operation outcome.
Second column are the predicted probabilities from the risk model. Third column can be either the
 predicted probabilities from the risk model or average outcome.}

\item{k}{Double. Reference value of the CUSUM control chart. Either \code{0} or a positive
value. Can be determined with function \code{\link{optimal_k}}.}

\item{RQ}{Double. Defines the true performance of a surgeon with the odds ratio ratio of death
\code{RQ}. Use \code{RQ = 1} to compute the in-control ARL and other values to compute the
out-of-control ARL.}

\item{side}{Character. Default is \code{"low"} to calculate ARL for the upper arm of the V-mask.
If side = \code{"up"}, calculate the lower arm of the V-mask.}

\item{yemp}{Logical. If \code{TRUE} use observed outcome value, if \code{FALSE} use estimated
binary logistc regression model.}

\item{m}{Integer. Number of simulation runs.}

\item{nc}{Integer. Number of cores used for parallel processing. Value is passed to
\code{\link{parSapply}}.}

\item{hmax}{Integer. Maximum value of \code{h} for the grid search.}

\item{jmax}{Integer. Number of digits for grid search.}

\item{verbose}{Logical. If \code{TRUE} verbose output is included, if \code{FALSE} a quiet
calculation of \code{h} is done.}
}
\value{
Returns a single value which is the control limit \code{h} for a given in-control ARL.
}
\description{
Compute alarm threshold of EO-CUSUM control charts using simulation.
}
\details{
Determines the control limit ("\code{h}") for given in-control ARL (\code{"L0"})
applying a grid search using \code{\link{eocusum_arl_sim}} and \code{\link{parSapply}}.
}
\examples{
\dontrun{
library(vlad)
library(dplyr)
data("cardiacsurgery", package = "spcadjust")

## preprocess data to 30 day mortality
SALL <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
         phase = factor(ifelse(date < 2*365, "I", "II")))
SI <- subset(SALL, phase == "I")
y <- subset(SALL, select = y)
GLM <- glm(y ~ s, data = SI, family = "binomial")
pi1 <- predict(GLM, type = "response", newdata = data.frame(s = SALL$s))
pmix <- data.frame(y, pi1, pi1)

## (Deterioration)
kopt <- optimal_k(pmix = pmix, RA = 2)
h <- eocusum_crit_sim(L0=370, pmix=pmix, k=kopt, side = "low", verbose=TRUE, nc=4)

## parameters to set up a tabular CUSUM or V-Mask (upper arm)
d <- h/kopt
theta <- atan(kopt)*180/pi
cbind(kopt, h, theta, d)

## (Improvement)
kopt <- optimal_k(pmix = pmix, RA = 1/2)
h <- eocusum_crit_sim(L0=370, pmix=pmix, k=kopt, side = "up", verbose=TRUE, nc=4)

## parameters to set up a tabular CUSUM or V-Mask (lower arm)
d <- h/kopt
theta <- atan(kopt)*180/pi
cbind(kopt, h, theta, d)
}
}
\references{
Barnard GA (1959). Control charts and stochastic processes.
\emph{J R Stat Soc Series B Stat Methodol}, \strong{21}(2), pp. 239--271.

Kemp KW (1961). The Average Run Length of the Cumulative Sum Chart
when a V-mask is used. \emph{J R Stat Soc Series B Stat Methodol}, \strong{23}(1),pp. 149--153.

Wittenberg P, Gan FF, Knoth S (2018).
A simple signaling rule for variable life-adjusted display derived from
an equivalent risk-adjusted CUSUM chart.
\emph{Statistics in Medicine}, \strong{37}(16), pp 2455--2473.
}
\author{
Philipp Wittenberg
}
