#' @title Twitter API authentication
#'
#' @description Twitter authentication uses OAuth and typically requires four developer API keys generated when you
#'   create a twitter app via the twitter developer web site. These keys and the authentication object produced are
#'   referenced as developer keys and developer access in this package. Software using developer access to the API can
#'   operate with full access to the API including within the user-context but with the understanding that it is an app.
#'   The \pkg{rtweet} package refers to this as bot authentication.
#'
#'   There is another method available commonly used by third-party apps in which an app can be authorized by a user to
#'   use the twitter API on their behalf. The implementation of this method in \pkg{vosonSML} does not require a
#'   developer account but does still require the user to have access to an apps two consumer API keys (generated by the
#'   app developer). The authentication object with token produced from this method allows the user to access the API
#'   within their own user-context and rate-limits.
#'
#'   If an individual has applied for and been granted Twitter API access they will also have a bearer token associated
#'   with their app. This token allows read-only access to the API but higher rate-limits so it is the most suited
#'   method for this package and data collection.
#'
#'   The twitter OAuth process is described here:
#'   \url{https://developer.twitter.com/en/docs/authentication/overview}.
#'
#' @note \pkg{vosonSML} uses the \pkg{rtweet} package for twitter data collection and also gratefully acknowledges the
#'   techniques and code written by its authors reproduced in this package for creating twitter API access tokens.
#'
#' @param socialmedia Character string. Identifier for social media API to authenticate, set to \code{"twitter"}.
#' @param appName Character string. Registered twitter app name associated with the API keys.
#' @param apiKey Character string. API consumer key to authenticate (also called API key).
#' @param apiSecret Character string. API consumer secret to authenticate (also called API secret).
#' @param accessToken Character string. API access token to authenticate.
#' @param accessTokenSecret Character string. API access token secret to authenticate.
#' @param bearerToken Character string. Twitter app bearer token. Default is \code{NULL}.
#' @param verbose Logical. Output additional information. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#'
#' @return A \code{credential} object containing an access token \code{$auth} and social media type descriptor
#'   \code{$socialmedia} set to \code{"twitter"}. Object has the class names \code{"credential"} and \code{"twitter"}.
#'
#' @examples
#' \dontrun{
#' # twitter API access using an app bearer token
#' app_auth <- Authenticate(
#'   "twitter",
#'   bearerToken = "xxxxxxxxxxxx"
#' )
#'
#' # twitter authentication using developer app API keys
#' dev_auth <- Authenticate(
#'   "twitter",
#'   appName = "My App",
#'   apiKey = "xxxxxxxxxxxx",
#'   apiSecret = "xxxxxxxxxxxx",
#'   accessToken = "xxxxxxxxxxxx",
#'   accessTokenSecret = "xxxxxxxxxxxx"
#' )
#'
#' # twitter user authentication via authorization of an app
#' # requires the apps consumer API keys apiKey and apiSecret parameters are
#' # equivalent to the apps consumer key and secret will open a web browser
#' # to twitter prompting the user to log in and authorize the app
#' user_auth <- Authenticate(
#'   "twitter",
#'   appName = "An App",
#'   apiKey = "xxxxxxxxxxxx",
#'   apiSecret = "xxxxxxxxxxxx"
#' )
#' }
#'
#' @export
Authenticate.twitter <-
  function(socialmedia,
           appName,
           apiKey,
           apiSecret,
           accessToken,
           accessTokenSecret,
           bearerToken = NULL,
           verbose = FALSE,
           ...) {

    msg("Creating twitter token...\n")

    credential <- list(socialmedia = "twitter", token = NULL)
    class(credential) <-
      append(class(credential), c("credential", "twitter"))

    # if bearer token provided
    if (!is.null(bearerToken)) {
      bearerToken <- check_chr(bearerToken, param = "bearerToken", min = 1)

      credential$auth <- structure(
        list(token = bearerToken),
        class = "rtweet_bearer"
      )

      msg("Done.\n")

      return(credential)
    }

    if (missing(appName)) {
      appName <- "r twitter app"
    } else {
      appName <- check_chr(appName, param = "appName", min = 1)
    }

    if (missing(apiKey) || missing(apiSecret)) {
      stop("Missing twitter API consumer keys.", call. = FALSE)
    }

    apiKey <- check_chr(apiKey, param = "apiKey", min = 1)
    apiSecret <- check_chr(apiSecret, param = "apiSecret", min = 1)

    app <- httr::oauth_app(
      appName,
      key = apiKey,
      secret = apiSecret
    )

    # for user context token
    if (missing(accessToken) || missing(accessTokenSecret)) {
      prompt_and_stop("httpuv", "Authenticate.twitter")

      token <- twitter_Token1.0$new(
        app = app,
        endpoint = httr::oauth_endpoints("twitter"),
        params = list(as_header = TRUE),
        cache_path = FALSE
      )

      credential$auth <- token

      msg("Done.\n")

      return(credential)
    }

    if (missing(accessToken) || missing(accessTokenSecret)) {
      stop("Missing twitter API access tokens.", call. = FALSE)
    }

    accessToken <- check_chr(accessToken, param = "accessToken", min = 1)
    accessTokenSecret <- check_chr(accessTokenSecret, param = "accessTokenSecret", min = 1)

    # for developer context token
    credentials <- list(
      oauth_token = accessToken,
      oauth_token_secret = accessTokenSecret
    )

    token <- httr::Token1.0$new(
      app = app,
      endpoint = httr::oauth_endpoints("twitter"),
      params = list(as_header = TRUE),
      credentials = credentials,
      cache_path = FALSE
    )

    credential$auth <- token

    msg("Done.\n")

    credential
  }

