context("test-themes")

test_theme_file <- function(theme = "duo", theme_file = paste0(theme, ".css"), ...) {
  theme_fun <- switch(
    theme,
    "duo" = style_duo,
    "duo_accent" = style_duo_accent,
    "duo_accent_inverse" = style_duo_accent_inverse,
    "mono_accent" = style_mono_accent,
    "mono_accent_inverse" = style_mono_accent_inverse,
    "mono_dark" = style_mono_dark,
    "mono_light" = style_mono_light,
    "solarized_dark" = style_solarized_dark,
    "solarized_light" = style_solarized_light,
    style_xaringan
  )

  tmpfile <- tempfile()

  theme_fun(outfile = tmpfile, ..., text_font_google = google_font("Noto Serif"))
  theme_css <- readLines(tmpfile)
  # Mask package version in test files
  theme_css <- sub("( \\*  Version: )[\\d.-]+", "\\1a.b.c.d.eeee", theme_css, perl = TRUE)
  theme_css <- paste(theme_css, collapse = "\n")
  expect_known_output(cat(theme_css), test_path("css", theme_file))
}

test_that("style_duo()", test_theme_file("duo"))
test_that("style_duo_accent()", test_theme_file("duo_accent"))
test_that("style_duo_accent_inverse()", test_theme_file("duo_accent_inverse"))
test_that("style_mono_accent()", test_theme_file("mono_accent"))
test_that("style_mono_accent_inverse()", test_theme_file("mono_accent_inverse"))
test_that("style_mono_dark()", test_theme_file("mono_dark"))
test_that("style_mono_light()", test_theme_file("mono_light"))
test_that("style_solarized_dark()", test_theme_file("solarized_dark"))
test_that("style_solarized_light()", test_theme_file("solarized_light"))
test_that("google fonts in theme", test_theme_file(
  "google_fonts",
  header_font_google = google_font("IBM Plex Serif", "700"),
  code_font_google = google_font("IBM Plex Mono")
))

test_that("header_background_auto = TRUE", {
  test_theme_file("duo", "duo-header_bg.css", header_background_auto = TRUE)
  test_theme_file("mono_light", "mono_light-header_bg.css", header_background_auto = TRUE)
  test_theme_file("solarized_dark", "solarized_dark-header_bg.css", header_background_auto = TRUE)
})

test_that("style colors are added to themes", {
  test_theme_file("xaringan", colors = c('light-blue' = "#bad4ed"))
})

test_that("setting google font overrides individual font", {
  theme_vars <- with_clean_session(function() {
    xf <- xaringanthemer::style_xaringan(
      text_font_family = "Damogran",
      header_font_family = "Magrathea",
      code_font_google = xaringanthemer::google_font("IBM Plex Mono")
    )
    xaringanthemer:::read_css_vars(xf)
  })

  expect_equal(theme_vars$text_font_family, "Damogran")
  expect_false(theme_vars$text_font_is_google)
  expect_equal(theme_vars$header_font_family, "Magrathea")
  expect_false(theme_vars$header_font_is_google)
  expect_equal(theme_vars$code_font_family, "'IBM Plex Mono'")
  expect_true(theme_vars$code_font_is_google)
})

test_that("default fonts are correctly identified as google font", {
  theme_vars <- with_clean_session(function() {
    xf <- xaringanthemer::style_xaringan()
    xaringanthemer:::read_css_vars(xf)
  })

  expect_equal(theme_vars$text_font_family,
               quote_elements_w_spaces(xaringanthemer_font_default("text_font_family")))
  expect_true(theme_vars$text_font_is_google)
  expect_equal(theme_vars$header_font_family,
               quote_elements_w_spaces(xaringanthemer_font_default("header_font_family")))
  expect_true(theme_vars$header_font_is_google)
})


test_that("NULL output returns CSS as text", {
  expect_false(
    with_clean_session(function() {
      xaringanthemer::style_xaringan(outfile = NULL)
      file.exists("xaringan-themer.css")
    })
  )

  xt <- with_clean_session(function() {
    xaringanthemer::style_xaringan(outfile = NULL)
  })

  expect_type(xt, "character")
  expect_true(any(grepl("generated by xaringanthemer", xt)))
})


test_that("style_xaringan() warns about non-hex colors used by theme_xaringan()", {
  expect_error(
    with_clean_session(function() {
      options(warn = 2)
      xaringanthemer::style_xaringan(text_color = "rgb(100, 100, 100)", background_color = "white", outfile = NULL)
    }),
    regexp = "Colors.+used by.+theme_xaringan",
    class = "callr_status_error"
  )
})
