% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module3_ata.R
\name{ata}
\alias{ata}
\alias{ata}
\alias{print.ata}
\alias{plot.ata}
\alias{ata_obj_relative}
\alias{ata_obj_absolute}
\alias{ata_constraint}
\alias{ata_item_use}
\alias{ata_item_enemy}
\alias{ata_item_fixedvalue}
\alias{ata_solve}
\title{Automated Test Assembly (ATA)}
\usage{
ata(pool, num_form = 1, len = NULL, max_use = NULL, ...)

\method{print}{ata}(x, ...)

\method{plot}{ata}(x, ...)

ata_obj_relative(x, coef, mode = c("max", "min"), negative = FALSE,
  forms = NULL, collapse = FALSE, internal_index = FALSE, ...)

ata_obj_absolute(x, coef, target, forms = NULL, collapse = FALSE,
  internal_index = FALSE, ...)

ata_constraint(x, coef, min = NA, max = NA, level = NULL, forms = NULL,
  collapse = FALSE, internal_index = FALSE)

ata_item_use(x, min = NA, max = NA, items = NULL)

ata_item_enemy(x, items)

ata_item_fixedvalue(x, items, min = NA, max = NA, forms)

ata_solve(x, solver = c("lpsolve", "glpk"), as.list = TRUE,
  details = TRUE, time_limit = 10, message = FALSE, ...)
}
\arguments{
\item{pool}{item pool (must include a, b, c parameters), a data.frame}

\item{num_form}{the number of forms to be assembled}

\item{len}{the length of each form}

\item{max_use}{the maximum use of each item}

\item{...}{options, e.g. group, common_items, overlap_items}

\item{x}{the ata object}

\item{coef}{the coefficients of the objective function or the constraint}

\item{mode}{the optimization mode: 'max' for maximization and 'min' for minimization}

\item{negative}{\code{TRUE} when the expected value of the objective function is negative}

\item{forms}{the indices of forms where objectives are added. \code{NULL} for all forms}

\item{collapse}{\code{TRUE} to collapse all forms into one objective function}

\item{internal_index}{\code{TRUE} to use internal form indices}

\item{target}{the target values of the objective function}

\item{min}{the lower bound of the constraint}

\item{max}{the upper bound of the constraint}

\item{level}{the level of the categorical variable to be constrained}

\item{items}{a vector of item indices}

\item{solver}{use 'lpsolve' or 'glpk' solver}

\item{as.list}{\code{TRUE} to return results in a list; otherwise, data frame}

\item{details}{\code{TRUE} to print detailed information}

\item{time_limit}{the time limit in seconds passed along to solvers}

\item{message}{\code{TRUE} to print messages from solvers}
}
\description{
Automated Test Assembly (ATA)

\code{ata} creates an \code{ata} object

\code{ata_obj_relative} adds relative (maximize/minimize) objectives to LP

\code{ata_obj_absolute} adds absolute objectives to the LP model

\code{ata_constraint} adds a constraint to the LP model

\code{ata_item_use} sets the minimum and maximum usage for items

\code{ata_item_enemy} adds enemy item relationships to the LP model

\code{ata_item_fixedvalue} sets a fixed value range for items

\code{ata_solve} solves the LP model
}
\details{
The \code{ata} object stores the definition of a LP model. \code{ata_solve} 
converts definitions to the real LP object and attempts to solve it. When
successfully solved, \code{result} (a matrix of binary values) and \code{items}
(a list or data.frame of items) are added to the \code{ata} object.

\code{ata_obj_relative}, maximize (y - beta) subject to y <= sum(x) <= y + beta, or
minimize (y + beta) subject to y - beta <= subject sum(x) <= y.
When \code{negative=TRUE}, y and beta are negative; otherwise positive.  
When \code{coef} is a numeric vector of the same length with the pool or forms, the given coefficients 
are used directly; when a variable name, variable values are used; when a numeric vector with different 
length with the pool, informations at those theta points are used.

\code{ata_obj_absolute} minimizes y0+y1 subject to t-y0 <= sum(x) <= t+y1.

When \code{level} is NA, it is assumed to impose constraints on a quantitative item property;
otherwise, a categorical item property. \code{coef} can be a variable name, a constant, or 
a numeric vector of the same length with the pool.

\code{ata_solve} takes control options in \code{...}. 
For lpsolve, see \code{lpSolveAPI::lp.control.options}.
For glpk, see \code{glpkAPI::glpkConstants}
}
\examples{
\dontrun{
## generate a 100-item pool
nitems <- 100
pool <- with(model_3pl_gendata(1, nitems), data.frame(id=1:nitems, a=a, b=b, c=c))
pool$id <- 1:nitems
pool$content <- sample(1:3, nitems, replace=TRUE)
pool$time <- round(rlnorm(nitems, log(60), .2))
pool$group <- sort(sample(1:round(nitems/3), nitems, replace=TRUE))

## ex. 1: 6 forms, 10 items, maximize b parameter
x <- ata(pool, 6, len=10, max_use=1)
x <- ata_obj_relative(x, "b", "max")
x <- ata_solve(x, timeout=5)
sapply(x$items, function(x) c(mean=mean(x$b), std=sd(x$b)))

## ex. 2: 3 forms, 10 items, minimize b parameter
x <- ata(pool, 3, len=10, max_use=1)
x <- ata_obj_relative(x, "b", "min", negative=TRUE)
x <- ata_solve(x, as.list=FALSE, timeout=5)
with(x$items, aggregate(b, by=list(form=form), mean))

## ex. 3: 2 forms, 10 items, mean(b) = 0, sd(b) = 1.0, content = (3, 3, 4)
x <- ata(pool, 2, len=10, max_use=1)
x <- ata_obj_absolute(x, pool$b, 0 * 10)
x <- ata_obj_absolute(x, (pool$b - 0)^2, 1 * 10)
x <- ata_constraint(x, "content", min=3, max=3, level=1)
x <- ata_constraint(x, "content", min=3, max=3, level=2)
x <- ata_constraint(x, "content", min=4, max=4, level=3)
x <- ata_solve(x, timeout=5)
sapply(x$items, function(x) c(mean=mean(x$b), sd=sd(x$b)))

## ex. 4: same with ex. 3, but group-based
x <- ata(pool, 2, len=10, max_use=1, group="group")
x <- ata_obj_absolute(x, pool$b, 0 * 10)
x <- ata_obj_absolute(x, (pool$b - 0)^2, 1 * 10)
x <- ata_constraint(x, "content", min=3, max=3, level=1)
x <- ata_constraint(x, "content", min=3, max=3, level=2)
x <- ata_constraint(x, "content", min=4, max=4, level=3)
x <- ata_solve(x, timeout=5)
sapply(x$items, function(x) c(mean=mean(x$b), sd=sd(x$b), 
       n_items=length(unique(x$id)), n_groups=length(unique(x$group))))

# ex. 5: 2 forms, 10 items, flat TIF over [-1, 1]
x <- ata(pool, 2, len=10, max_use=1)
x <- ata_obj_relative(x, seq(-1, 1, .5), "max")
x <- ata_solve(x)
plot(x)
}
}
