% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class-sens.R
\name{sens}
\alias{sens}
\alias{sens.data.frame}
\alias{sens_vec}
\title{Sensitivity}
\usage{
sens(data, ...)

\method{sens}{data.frame}(data, truth, estimate, estimator = NULL, na_rm = TRUE, ...)

sens_vec(truth, estimate, estimator = NULL, na_rm = TRUE, ...)
}
\arguments{
\item{data}{Either a \code{data.frame} containing the \code{truth} and \code{estimate}
columns, or a \code{table}/\code{matrix} where the true class results should be
in the columns of the table.}

\item{...}{Not currently used.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:quasiquotation]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimate}{The column identifier for the predicted class
results (that is also \code{factor}). As with \code{truth} this can be
specified different ways but the primary method is to use an
unquoted variable name. For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimator}{One of: \code{"binary"}, \code{"macro"}, \code{"macro_weighted"},
or \code{"micro"} to specify the type of averaging to be done. \code{"binary"} is
only relevant for the two class case. The other three are general methods for
calculating multiclass metrics. The default will automatically choose \code{"binary"}
or \code{"macro"} based on \code{estimate}.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{sens_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
These functions calculate the \code{\link[=sens]{sens()}} (sensitivity) of a measurement system
compared to a reference result (the "truth" or gold standard).
Highly related functions are \code{\link[=spec]{spec()}}, \code{\link[=ppv]{ppv()}}, and \code{\link[=npv]{npv()}}.
}
\details{
The sensitivity (\code{sens()}) is defined as the proportion of positive
results out of the number of samples which were actually
positive.

When the denominator of the calculation is \code{0}, sensitivity is undefined.
This happens when both \verb{# true_positive = 0} and \verb{# false_negative = 0}
are true, which mean that there were no true events. When computing binary
sensitivity, a \code{NA} value will be returned with a warning. When computing
multiclass sensitivity, the individual \code{NA} values will be removed, and the
computation will procede, with a warning.
}
\section{Relevant Level}{


There is no common convention on which factor level should
automatically be considered the "event" or "positive" result.
In \code{yardstick}, the default is to use the \emph{first} level. To
change this, a global option called \code{yardstick.event_first} is
set to \code{TRUE} when the package is loaded. This can be changed
to \code{FALSE} if the \emph{last} level of the factor is considered the
level of interest by running: \code{options(yardstick.event_first = FALSE)}.
For multiclass extensions involving one-vs-all
comparisons (such as macro averaging), this option is ignored and
the "one" level is always the relevant result.
}

\section{Multiclass}{


Macro, micro, and macro-weighted averaging is available for this metric.
The default is to select macro averaging if a \code{truth} factor with more
than 2 levels is provided. Otherwise, a standard binary calculation is done.
See \code{vignette("multiclass", "yardstick")} for more information.
}

\section{Implementation}{


Suppose a 2x2 table with notation:

\tabular{rcc}{ \tab Reference \tab \cr Predicted \tab Positive \tab Negative
\cr Positive \tab A \tab B \cr Negative \tab C \tab D \cr }

The formulas used here are:

\deqn{Sensitivity = A/(A+C)}
\deqn{Specificity = D/(B+D)}
\deqn{Prevalence = (A+C)/(A+B+C+D)}
\deqn{PPV = (Sensitivity * Prevalence) / ((Sensitivity * Prevalence) + ((1-Specificity) * (1-Prevalence)))}
\deqn{NPV = (Specificity * (1-Prevalence)) / (((1-Sensitivity) * Prevalence) + ((Specificity) * (1-Prevalence)))}

See the references for discussions of the statistics.
}

\examples{
# Two class
data("two_class_example")
sens(two_class_example, truth, predicted)

# Multiclass
library(dplyr)
data(hpc_cv)

hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  sens(obs, pred)

# Groups are respected
hpc_cv \%>\%
  group_by(Resample) \%>\%
  sens(obs, pred)

# Weighted macro averaging
hpc_cv \%>\%
  group_by(Resample) \%>\%
  sens(obs, pred, estimator = "macro_weighted")

# Vector version
sens_vec(two_class_example$truth, two_class_example$predicted)

# Making Class2 the "relevant" level
options(yardstick.event_first = FALSE)
sens_vec(two_class_example$truth, two_class_example$predicted)
options(yardstick.event_first = TRUE)

}
\references{
Altman, D.G., Bland, J.M. (1994) ``Diagnostic tests 1:
sensitivity and specificity,'' \emph{British Medical Journal},
vol 308, 1552.
}
\seealso{
Other class metrics: 
\code{\link{accuracy}()},
\code{\link{bal_accuracy}()},
\code{\link{detection_prevalence}()},
\code{\link{f_meas}()},
\code{\link{j_index}()},
\code{\link{kap}()},
\code{\link{mcc}()},
\code{\link{npv}()},
\code{\link{ppv}()},
\code{\link{precision}()},
\code{\link{recall}()},
\code{\link{spec}()}

Other sensitivity metrics: 
\code{\link{npv}()},
\code{\link{ppv}()},
\code{\link{spec}()}
}
\author{
Max Kuhn
}
\concept{class metrics}
\concept{sensitivity metrics}
