% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fasthmmfit.R
\name{fasthmmfit}
\alias{fasthmmfit}
\title{Fast gradient descent / stochastic gradient descent algorithm to learn the parameters
in a specialized zero-inflated hidden Markov model, where zero-inflation only happens
in State 1. And if there were covariates, they could only be the same ones for the
state-dependent log Poisson means and the logit structural zero proportion.}
\usage{
fasthmmfit(y, x = NULL, ntimes = NULL, M, prior_init, tpm_init, emit_init,
  zero_init, yceil = NULL, stochastic = FALSE, nmin = 1000,
  nupdate = 100, power = 0.7, rate = c(1, 0.05), method = "Nelder-Mead",
  hessian = FALSE, ...)
}
\arguments{
\item{y}{observed time series values}

\item{x}{matrix of covariates for the log poisson means and logit zero proportion.
Default to NULL.}

\item{ntimes}{a vector specifying the lengths of individual, 
i.e. independent, time series. If not specified, the responses are assumed to 
form a single time series, i.e. ntimes=length(y).}

\item{M}{number of latent states}

\item{prior_init}{a vector of initial values for prior probability for each state}

\item{tpm_init}{a matrix of initial values for transition probability matrix}

\item{emit_init}{a vector of initial values for the means for each poisson distribution}

\item{zero_init}{a scalar initial value for the structural zero proportion}

\item{yceil}{a scalar defining the ceiling of y, above which the values will be
truncated. Default to NULL.}

\item{stochastic}{Logical. Should the stochastic gradient descent methods be used.}

\item{nmin}{a scalar for the minimum number of observations before the first
iteration of stochastic gradient descent. Default to 1000.}

\item{nupdate}{a scalar specifying the total number of updates for stochastic
gradient descent. Default to 100.}

\item{power}{a scalar representing the power of the learning rate, which should lie
between (0.5,1]. Default to 0.7}

\item{rate}{a vector of learning rate in stochastic gradient descent for the logit
parameters and log parameters. Default to c(1,0.05).}

\item{method}{method to be used for direct numeric optimization. See details in
the help page for optim() function. Default to Nelder-Mead.}

\item{hessian}{Logical. Should a numerically differentiated Hessian matrix be returned?
Note that the hessian is for the working parameters, which are the generalized logit of prior 
probabilities (except for state 1), the generalized logit of the transition probability 
matrix(except 1st column), the logit of non-zero zero proportions, and the log of 
each state-dependent poisson means}

\item{...}{Further arguments passed on to the optimization methods}
}
\value{
the maximum likelihood estimates of the zero-inflated hidden Markov model
}
\description{
Fast gradient descent / stochastic gradient descent algorithm to learn the parameters
in a specialized zero-inflated hidden Markov model, where zero-inflation only happens
in State 1. And if there were covariates, they could only be the same ones for the
state-dependent log Poisson means and the logit structural zero proportion.
}
\examples{
#1. no covariates
set.seed(135)
prior_init <- c(0.5,0.2,0.3)
emit_init <- c(10,40,70)
zero_init <- c(0.5,0,0)
omega <- matrix(c(0.5,0.3,0.2,0.4,0.3,0.3,0.2,0.4,0.4),3,3,byrow=TRUE)
result <- hmmsim(n=10000,M=3,prior=prior_init, tpm_parm=omega,
                emit_parm=emit_init,zeroprop=zero_init)
y <- result$series

time <- proc.time()
fit1 <-  fasthmmfit(y,x=NULL,ntimes=NULL,M=3,prior_init,omega,
              emit_init,0.5, hessian=FALSE,
              method="BFGS", control=list(trace=1))
proc.time() - time


#2. with covariates
priorparm <- 0
tpmparm <- c(-2,2)
zeroindex <- c(1,0)
zeroparm <- c(0,-1,1)
emitparm <- c(2,0.5,-0.5,3,0.3,-0.2)
workparm <- c(priorparm,tpmparm,zeroparm,emitparm)

designx <- matrix(rnorm(20000),nrow=10000,ncol=2)
x <- cbind(1,designx) #has to make the additional 1st column of 1 for intercept
result <- hmmsim2(workparm,2,10000,zeroindex,emit_x=designx,zeroinfl_x=designx)
y <- result$series

time <- proc.time()
fit2 <-  fasthmmfit(y=y,x=x,ntimes=NULL,M=2,prior_init=c(0.5,0.5),
              tpm_init=matrix(c(0.9,0.1,0.1,0.9),2,2),
              zero_init=0.4,emit_init=c(7,21), hessian=FALSE,
              method="BFGS", control=list(trace=1))
proc.time() - time
fit2

#3. stochastic gradient descent without covariates
#no covariates
prior_init <- c(0.5,0.2,0.3)
emit_init <- c(10,40,70)
zero_init <- c(0.5,0,0)
omega <- matrix(c(0.5,0.3,0.2,0.4,0.3,0.3,0.2,0.4,0.4),3,3,byrow=TRUE)
result <- hmmsim(n=50000,M=3,prior=prior_init, tpm_parm=omega,
                emit_parm=emit_init,zeroprop=zero_init)
y <- result$series

initparm2 <- c(-1,-0.5,  -0.3,-0.3,-0.4,-0.4,0.5,0.5,  0,2,3,4)
time <- proc.time()
fitst <- fasthmmfit(y=y,x=NULL,ntimes=NULL,M=3,prior_init=c(0.4,0.3,0.3),
              tpm_init=matrix(c(0.6,0.3,0.1,0.3,0.4,0.3,0.1,0.3,0.6),3,3,byrow=TRUE),
              zero_init=0.3,emit_init=c(8,35,67),stochastic=TRUE,
              nmin=1000,nupdate=1000,power=0.6,rate=c(1,0.05))
proc.time() - time
str(fitst)

#with covariates
priorparm <- 0
tpmparm <- c(-2,2)
zeroindex <- c(1,0)
zeroparm <- c(0,-1,1)
emitparm <- c(2,0.5,-0.5,3,0.3,-0.2)
workparm <- c(priorparm,tpmparm,zeroparm,emitparm)

designx <- matrix(rnorm(100000),nrow=50000,ncol=2)
x <- cbind(1,designx) #has to make the additional 1st column of 1 for intercept
result <- hmmsim2(workparm,2,50000,zeroindex,emit_x=designx,zeroinfl_x=designx)
y <- result$series

initparm <- c(0, -1.8,1.8, 0,-0.8,0.8, 1.8,0.6,-0.6,3.1,0.4,-0.3)

time <- proc.time()
fitst <- fasthmmfit(y=y,x=x,ntimes=NULL,M=2,prior_init=c(0.4,0.6),
              tpm_init=matrix(c(0.8,0.2,0.2,0.8),2,2,byrow=TRUE),
              zero_init=0.3,emit_init=c(10,25),stochastic=TRUE,
              nmin=1000,nupdate=1000,power=0.6,rate=c(1,0.05))
proc.time() - time
str(fitst)
}
\references{
Walter Zucchini, Iain L. MacDonald, Roland Langrock. Hidden Markov Models for 
Time Series: An Introduction Using R, Second Edition. Chapman & Hall/CRC
}
