use crate::job::*;
use crate::registry::Registry;
use crate::unwind;
use std::mem;
use std::sync::Arc;






















































pub fn spawn<F>(func: F)
where
    F: FnOnce() + Send + 'static,
{

    unsafe { spawn_in(func, &Registry::current()) }
}




pub(super) unsafe fn spawn_in<F>(func: F, registry: &Arc<Registry>)
where
    F: FnOnce() + Send + 'static,
{





    let abort_guard = unwind::AbortIfPanic; // just in case we are wrong, and code CAN panic
    let job_ref = spawn_job(func, registry);
    registry.inject_or_push(job_ref);
    mem::forget(abort_guard);
}

unsafe fn spawn_job<F>(func: F, registry: &Arc<Registry>) -> JobRef
where
    F: FnOnce() + Send + 'static,
{


    registry.increment_terminate_count();

    HeapJob::new({
        let registry = Arc::clone(registry);
        move || {
            registry.catch_unwind(func);
            registry.terminate(); // (*) permit registry to terminate now
        }
    })
    .into_static_job_ref()
}



























pub fn spawn_fifo<F>(func: F)
where
    F: FnOnce() + Send + 'static,
{

    unsafe { spawn_fifo_in(func, &Registry::current()) }
}




pub(super) unsafe fn spawn_fifo_in<F>(func: F, registry: &Arc<Registry>)
where
    F: FnOnce() + Send + 'static,
{





    let abort_guard = unwind::AbortIfPanic; // just in case we are wrong, and code CAN panic
    let job_ref = spawn_job(func, registry);



    match registry.current_thread() {
        Some(worker) => worker.push_fifo(job_ref),
        None => registry.inject(job_ref),
    }
    mem::forget(abort_guard);
}

#[cfg(test)]
mod test;
