% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PlantArea.R
\docType{methods}
\name{plantAreaIndex}
\alias{plantAreaIndex}
\title{Plant Area Index (PAI)}
\usage{
plantAreaIndex(
  vxsp,
  vx,
  type = c("av", "ag", "bc", "xy"),
  pattern.pad = "^pad_*"
)
}
\arguments{
\item{vxsp}{a \code{\linkS4class{VoxelSpace}} object.}

\item{vx}{a subset of voxel index. A data.table with \verb{i, j, k} columns.
Missing parameter means whole voxel space.}

\item{type}{a character vector, the type of PAI profile.
\itemize{
\item \code{"av"} Averaged value on every voxel
\item \code{"ag"} Above ground vertical profile
\item \code{"bc"} Below canopy vertical profile
\item \code{"xy"} Spatial profile
}}

\item{pattern.pad}{character string containing a
\link[base:regex]{regular expression} to be matched in the voxel space
variable names, for selecting PAD variables. Typing the name of a specific
PAD variable works just fine.}
}
\value{
Returns a list of PAI profiles for requested PAD variables and PAI
types.
\subsection{\code{av} Averaged PAI}{

Returns a single value. Calculated as the sum of PAD values multiplied by
voxel volume and divided by ground surface with vegetation.
}

\subsection{\code{ag & bc} Above ground and below canopy PAI vertical profile}{

Returns a vertical profile of PAI values either from ground distance or
canopy depth. Calculated as the averaged PAD values per layer (a layer
being defined by either the distance to ground or canopy level) multiplied
by voxel size along z (equivalent to multiplying PAD by voxel volume and
dividing by voxel ground surface).
}

\subsection{\code{xy} Spatial PAI profile}{

Returns a list a PAI values by i, j index. Calculated as the sum of PAD on
(i, j) column multiplied by voxel size along z (equivalent to multiplying
PAD by voxel volume and dividing by voxel ground surface).
}
}
\description{
Computes Plant Area Index (PAI) from Plant Area Density (PAD).
PAI is defined as the plant area per unit ground surface area (PAI = plant
area / ground area = m^2 / m^2).

The function can estimate PAI on the whole voxel space or any region of
interest (parameter vx subset of voxels). It can compute PAI from several
perspectives : either an averaged PAI value, a two-dimensions (i, j) PAI
array or vertical profiles either above ground or below canopy.
}
\examples{
vxsp <- readVoxelSpace(system.file("extdata", "tls_sample.vox", package = "AMAPVox"))
vxsp@data <- merge(vxsp@data, plantAreaDensity(vxsp), by = c("i", "j", "k"))
\dontrun{
lai <- plantAreaIndex(vxsp)
names(lai)
library(ggplot2)
ggplot(data = lai[["pad_transmittance.pai.ag" ]], aes(x=pai, y=ground_distance)) +
  geom_path() + geom_point()
}
# PAI on a subset
ni <- round(dim(vxsp)[1]/2)
vx <- vxsp@data[i < ni, .(i, j, k)]
lai <- plantAreaIndex(vxsp, vx)

}
\seealso{
\code{\link[=plantAreaDensity]{plantAreaDensity()}}
}
