#' Compute conditional probabilities
#'
#' \code{computeVorobTerms} compute the future uncertainty.
#'
#' @param i the index for which conditional probabilities are computed. It ranges from 1 to \eqn{2^{nrow(X.new)}} which represents all possible combinations.
#' @param object an object of class \code{\link[GPCsign]{gpcm}}.
#' @param intpoints.oldmean vectors of size \code{ncol(integration.points)} corresponding to the mean  at the integration points before adding the batchsize points \code{x} to the design of experiments.
#' @param krig2 a list containing the output of the function  \code{\link[ARCHISSUR]{predict_update_gpc_parallel}}.
#' @param sk.new conditional standard deviations vector at the points \code{intpoints}.
#' @param alpha a scalar representing the Vorob'ev threshold.
#' @param gpc  a list containing the output of the predict function  \code{\link[GPCsign]{predict}} at \code{X.new}.
#' @param X.new input vector of size \code{batchsize*d} at which one wants to evaluate the criterion.
#' @param seed to fix the seed.
#'
#' @return a list with:
#' \item{term2}{Vector equal to \eqn{p_{n+q}p(y)} where \eqn{p_{n+q}} is the updated feasibility probability at integration.points and \eqn{p(y)} is the probability of outcomes y at new points \code{X.new}}
#'  \item{term1}{Vector equal to the product of \code{term2} with the indicator of \eqn{p_{n+q}> \alpha}  where \eqn{\alpha} is the Vorob'ev threshold}
#'  \item{term3}{Vector equal to the product of \eqn{p_{y}} with the indicator of \eqn{p_{n+q}< \alpha} }
#'  
#'
#'
#' @importFrom TruncatedNormal rtmvnorm
#' @importFrom stats optim pnorm runif
#' @importFrom KrigInv vorob_threshold
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#'
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
#' @examples
#' #-------------------------------------------------------------------
#' #--------------------------- computeVorobTerms -----------------------------
#' #-------------------------------------------------------------------
#'
#' ## 20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#' require(DiceKriging)
#' fx <- apply(x, 1, branin)
#' f <- ifelse(fx < 14, -1, 1)
#' Xf <- as.matrix(x)
#'
#' require(future)  # load future package for parallelization 
#' plan(multisession) # activate parallel calculations (with available cores automatic detection)
#'
#' ## gpcm object
#' require(GPCsign) 
#' model <- gpcm(f, Xf, coef.m = -1.25, coef.cov = c(1.17,0.89))
#'
#' ##
#' n.grid <- 20
#' x.grid <- seq(0,1,length=n.grid)
#' newdata <- expand.grid(x.grid,x.grid)
#' newdata <- as.matrix(newdata)
#' pred1 <- predict(object=model,newdata=newdata)
#' precalc.data <- list()
#' precalc.data$c.K <- crossprod(pred1$c, model@invK)
#' newdata.oldsd <- sqrt(pred1$Zsimu_var)
#'
#' # new points added
#' new.x <- matrix(c(0.1,0.2),ncol=2,byrow=TRUE)
#'
#' # predicion at new points
#' pred2 <- predict(object=model,newdata=new.x)
#' Sigma.r <- pred2$cov
#'
#' newdata <- scale(x = newdata, center = model@X.mean, scale = model@X.std)
#' new.x <- scale(x = new.x, center = model@X.mean, scale = model@X.std)
#' kn <- computeQuickgpccov(object = model,
#'                          integration.points = newdata,
#'                          X.new = new.x,
#'                          precalc.data = precalc.data,
#'                          c.newdata = pred2$c)
#'
#' updated.predictions <- predict_update_gpc_parallel(Sigma.r = Sigma.r,
#'                                                    newdata.oldsd = newdata.oldsd,
#'                                                    kn = kn)
#' # parameters for comp_term function
#' i <- 1
#' intpoints.oldmean <- pred1$Zsimu_mean
#' sk.new <- updated.predictions$sd
#' pn <- pred1$prob
#' alpha <- KrigInv::vorob_threshold(pn)
#'
#' result <- computeVorobTerms(i = i, object = model,
#'                     intpoints.oldmean = intpoints.oldmean,
#'                     krig2 = updated.predictions,
#'                     sk.new = sk.new,
#'                     alpha = alpha,
#'                     gpc = pred2,
#'                     X.new = new.x)
#' plan(sequential) # deactivate parallel calculations: back to sequential mode 
computeVorobTerms <- function(i, object, intpoints.oldmean, krig2, sk.new, alpha, gpc, X.new, seed=NULL){
  ind_nonzero = sk.new != 0.
  d <- object@d
  proba.class <- cbind(gpc$prob, 1-gpc$prob)
  q <- length(gpc$prob) # batchsize
  Zmean.newdata <- matrix(gpc$Zsimu_mean,nrow=q)
  Zvar.newdata <- gpc$Zsimu_var
  cov.newdata <- gpc$cov
  c.newdata <- gpc$c
  cov.newdata <- 0.5*(cov.newdata + t(cov.newdata))  ## to solve symmetry problems due to machine precision
  n_zero <- sum(!ind_nonzero, na.rm = TRUE)
  nobs <- ncol(Zmean.newdata)
  N_int <- nrow(intpoints.oldmean)
  lq <- rep(list(c(1,-1)), q)
  f.new.value <- expand.grid(lq)[i,] # future value: i-th outcome
  indic_proba <- rep(1,q)
  indic_proba[f.new.value == -1] = 2

  #set bounds for f.new.value 
  l <- -Zmean.newdata
  u <- matrix(rep(Inf, nobs*q),nrow=q)
  l[f.new.value==-1,] <- -Inf
  u[f.new.value==-1,] <- -Zmean.newdata[f.new.value==-1,]

  # get precalculated data of gpcm model at X.new
  precalc.data.q <- list(3)
  precalc.data.q$c.K <- crossprod(gpc$c[,q] , object@invK)
  precalc.data.q$lambda.intpoints <- gpc$lambda[,q]
  precalc.data.q$pn.intpoints <- gpc$prob[q]

  zobs.new <- NULL

  # simulate future observations of latent process at X.new knowing future class values f.new.value  
  for(j in 1:nobs){
        set.seed(seed)
        zobs.new <- cbind(zobs.new,TruncatedNormal::rtmvnorm(lb=l[,j], ub=u[,j], sigma=cov.newdata, n=1))
  } 

  p_i = proba.class[,indic_proba][1] # probability to have f.new.value 
  # compute conditional probabilities Bernoulli
  if(q>1){
    for(k in 2:q)
    {
      zobs.new.q <- NULL
      for(j in 1:nobs){
        ## generation of (Z(^i)_j|Z^n_j)_j j=1,nobs
        set.seed(seed)
        zobs.new.q <- cbind(zobs.new.q,
                            TruncatedNormal::rtmvnorm(lb=l[1:(k-1),j], ub=u[1:(k-1),j], sigma=cov.newdata[1:(k-1),1:(k-1)], n=1))
      }
      knq <- computeQuickgpccov(object, integration.points =  matrix(X.new[k,],ncol=d), X.new = matrix(X.new[1:(k-1),],ncol=d),
                                  precalc.data = precalc.data.q, c.newdata = c.newdata[,1:(k-1)])
      krigq  <- predict_update_gpc_parallel(Sigma.r=cov.newdata[1:(k-1),1:(k-1)], newdata.oldsd=Zvar.newdata[k], kn=knq)
      mn.new.q <- Zmean.newdata[k,] + krigq$lambda %*%  zobs.new.q
      if(krigq$sd !=0){
        Zbar.q <- mn.new.q/krigq$sd
        Phi_barq <-  1 - pnorm(-Zbar.q)
        p.intermq <- mean(rowMeans(Phi_barq)) #p_i-1(x_i)
        if(f.new.value[q]==-1)  p.intermq<- 1-p.intermq
        p_i  <- p_i*p.intermq
        }
      else{
        Phi_barq <- matrix(0, ncol= ncol(mn.new.q), nrow = nrow(mn.new.q))
        Phi_barq[mn.new.q > 0] <- 1
        p.intermq <- mean(rowMeans(Phi_barq))
        if(f.new.value[q]==-1) p.intermq <- 1-p.intermq
        p_i <- p_i*p.intermq
        }
    }
  }


  #### computation of the future uncertainty based on vorob'ev dev

  ### future proba
  mn.new <- intpoints.oldmean +  krig2$lambda %*% zobs.new
  Zbar <- mn.new/sk.new
  Phi_bar <-  1 - pnorm(-Zbar[ind_nonzero,])

  Phi_bar_zero = NULL
  if(n_zero>0){
    Zbar_zero <- mn.new[!ind_nonzero,]
    if(length(dim(Zbar_zero)) < 2)
    {
      Zbar_zero <- matrix( Zbar_zero, ncol = dim(object@Z_obs)[2])
    }
    Phi_bar_zero <- matrix(0, ncol= ncol(Zbar_zero), nrow = nrow(Zbar_zero))
    Phi_bar_zero[Zbar_zero > 0] <- 1
  }
  p.interm <-  rowMeans(rbind(Phi_bar,Phi_bar_zero)) # p_n+q(x)

  ###
  pnew = p_i * p.interm

  indicatrice_term <- rep(0, N_int)
  indicatrice_term[p.interm<alpha] <- 1
  value_u1 <-  pnew * indicatrice_term
  value_u3 <-  p_i * abs(indicatrice_term -1)
  return(list(term1=value_u1, term3=value_u3, term2=pnew))
}

