#' QQ-Plot
#' QQ-Plot between observed quantiles and theoretical quantiles.
#' @param x Object of the class `accept_reject` returned by the function
#' `accept_reject()`.
#' @param ... Additional arguments to be passed to methods.
#' @returns An object of classes `gg` and `ggplot` with the QQ-Plot of
#' theoretical quantiles versus observed quantiles.
#' @details
#' Generic method to plot the QQ-Plot between observed quantiles and theoretical
#' quantiles. The generic method will call the specific method
#' `qqplot.accept_reject()`, which operates on objects of class accept_reject
#' returned by the function `accept_reject()`.
#'
#' @seealso [accept_reject()], [print.accept_reject()], [plot.accept_reject()]
#' and [inspect()].
#' @export
qqplot <- function(x, ...) {
  UseMethod("qqplot")
}

#' QQ-Plot
#' Plot the QQ-Plot between observed quantiles and theoretical quantiles.
#' @param x Object of the class accept_reject returned by the function `accept_reject()`.
#' @param alpha Transparency of the points and reference line representing where
#' the quantiles should be (theoretical quantiles).
#' @param color_points Color of the points (default is `"#F890C2"`).
#' @param color_line Color of the reference line (detault is `"#BB9FC9"`).
#' @param size_points Size of the points (default is `1`).
#' @param size_line Thickness of the reference line (default is `1`).
#' @param ... Additional arguments for the `quantile()` function. For instance,
#' it's possible to change the algorithm type for quantile calculation.
#' @details
#' The function `qqplot.accept_reject()` for samples larger than ten thousand,
#' the `geom_scattermost()` function from the
#' [**scattermore**](https://CRAN.R-project.org/package=scattermore) library
#' is used to plot the points, as it is more efficient than `geom_point()` from
#' the [**ggplot2**](https://CRAN.R-project.org/package=ggplot2) library.
#'
#' @return An object of classes `gg` and `ggplot` with the QQ-Plot between the
#' observed quantiles generated by the return of the function `accept_reject()`
#' and the theoretical quantiles of the true distribution.
#' @examples
#' set.seed(0) # setting a seed for reproducibility
#'
#' x <- accept_reject(
#'   n = 2000L,
#'   f = dbinom,
#'   continuous = FALSE,
#'   args_f = list(size = 5, prob = 0.5),
#'   xlim = c(0, 5)
#' )
#' qqplot(x)
#'
#' y <- accept_reject(
#'   n = 1000L,
#'   f = dnorm,
#'   continuous = TRUE,
#'   args_f = list(mean = 0, sd = 1),
#'   xlim = c(-4, 4)
#' )
#' qqplot(y)
#'
#' @seealso [qqplot.accept_reject()], [accept_reject()], [plot.accept_reject()], [inspect()] and
#' [qqplot()].
#'
#' @importFrom Rcpp evalCpp
#' @importFrom ggplot2 ggplot geom_point geom_abline labs theme element_text
#' coord_cartesian scale_x_continuous scale_y_continuous aes_string
#' @importFrom scattermore geom_scattermore
#' @importFrom parallel mclapply detectCores
#' @importFrom stats quantile
#' @export
qqplot.accept_reject <-
  function(x,
           alpha = 0.5,
           color_points = "#F890C2",
           color_line = "#BB9FC9",
           size_points = 1,
           size_line = 1,
           ...
  ) {

  continuous <- attr(x, "continuous")
  sample_quantiles <- sort(x)
  p <- seq(0, 1, length.out = length(x))
  theoretical_quantiles <- quantile(x, probs = p, ...)

  df <- data.frame(Theoretical = theoretical_quantiles, Sample = sample_quantiles)
  xlim <- attr(x, "xlim")

  plot <- ggplot(df, aes_string(x = "Theoretical", y = "Sample")) +
    geom_abline(slope = 1, intercept = 0, color = color_line, size = size_line)

  if (continuous && length(x) >= 10e3){
    plot <- plot +
      geom_scattermore(
        pointsize = size_points + 2,
        interpolate = TRUE,
        color = color_points
    )
  } else {
    plot <- plot +
      geom_point(alpha = alpha, color = color_points, size = size_points)
  }

  plot <-
    plot +
    coord_cartesian(xlim = xlim, ylim = xlim) +
    labs(x = "Theoretical Quantiles", y = "Sample Quantiles", title = "QQ-Plot") +
    theme(
      axis.title = ggplot2::element_text(face = "bold"),
      title = ggplot2::element_text(face = "bold"),
      legend.title = ggplot2::element_text(face = "bold"),
      plot.subtitle = ggplot2::element_text(face = "plain")
    )

  if (!continuous) {
    plot <- plot +
      scale_x_continuous(breaks = xlim[1L]:xlim[2L]) +
      scale_y_continuous(breaks = xlim[1L]:xlim[2L])
  }
  return(plot)
}
