% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seasFcstQBR.R
\name{seasFcstQBR}
\alias{seasFcstQBR}
\title{Seasonal Forecast of Wet Season Calendar (WSC) from Tercile Rainfall
Probabilities using Quantile Bin Resampling (QBR)}
\usage{
seasFcstQBR(
  hisYearStart,
  hisYearEnd,
  rainTerc,
  seasRain,
  hisWSCvar,
  fcstVarName,
  tercileMethod
)
}
\arguments{
\item{hisYearStart}{Starting year (YYYY) for historical resampling.}

\item{hisYearEnd}{Ending year (YYYY) for historical resampling.}

\item{rainTerc}{A data frame with tercile probabilities for rainfall.
Columns should be named "T1" (below normal), "T2" (normal), and "T3"
(above normal).}

\item{seasRain}{A data frame containing seasonal rainfall data with columns}

\item{hisWSCvar}{A data frame containing historical WSC simulations. This can
be the output from the \code{calcSeasCal} function or a similar data frame with
required columns.}

\item{fcstVarName}{A character string indicating the WSC variable to forecast
("Onset" or "Cessation").}

\item{tercileMethod}{options are "quantiles" or "fixedValues"}
}
\value{
A data frame containing the tercile probabilities for the WSC
variable ("BelowNormal", "Normal", and "AboveNormal").
}
\description{
Generates seasonal forecasts for Wet Season Calendar (WSC)
variables (onset or cessation) using tercile seasonal rainfall probabilities
}
\details{
Uses QBR (Quantile Bin Resampling) to produce forecasts for
onset or cessation of the rainy season. It first categorizes historical WSC
simulations based on seasonal rainfall terciles and then resamples based on
given rainfall probabilities to generate ensemble forecasts.
}
\examples{
\donttest{
library(dplyr)

## Load example data:
data(AgroClimateData)

## Estimate daily PET:
PET <- calcEto(AgroClimateData, method = "PM", Zh = 10)

## Add the estimated PET 'ET.Daily' to a new column in AgroClimateData:
AgroClimateData$Eto <- PET$ET.Daily

## Estimate daily water balance for the soil having 100mm of WHC:
watBal.list <- calcWatBal(data = AgroClimateData, soilWHC = 100)
watBal <- watBal.list$data

## seasonal calendar is estimated for the onset window ranges from
## 01 September to 31 January having a soil with 100mm of WHC:

soilWHC <- 100
onsetWind.start <- "09-01"
onsetWind.end <- "01-31"
cessaWind.end <- "06-30"

seasCal.dF <- calcSeasCal(
  data = watBal, onsetWind.start, onsetWind.end,
  cessaWind.end, soilWHC
)

## Tercile Rainfall Probabilities of seasonal Forecast for OND, 2023:
rainTerc <- data.frame(T1 = 0.15, T2 = 0.10, T3 = 0.75)

## Summarize rainfall data for October to December:
seasRain <- AgroClimateData \%>\%
  filter(Month \%in\% c(10, 11, 12)) \%>\%
  group_by(Year) \%>\%
  summarize(sRain = sum(Rain))

## Start of the historical resampling year
hisYearStart <- 1991

## End of the historical resampling year
hisYearEnd <- 2022

## Historical WSC Simulations:
hisWSCvar <- seasCal.dF

## WSC variable to forecast:
fcstVarName <- "Onset"
tercileMethod <- "quantiles"

SeasFcst.dF <- seasFcstQBR(
  hisYearStart, hisYearEnd, rainTerc,
  seasRain, hisWSCvar, fcstVarName,
  tercileMethod
)
}
}
\references{
\itemize{
\item MacLeod D, Quichimbo EA, Michaelides K, Asfaw DT, Rosolem R, Cuthbert MO,
et al. (2023) Translating seasonal climate forecasts into water balance
forecasts for decision making. PLOS Clim 2(3): e0000138.
https://doi.org/10.1371/journal.pclm.0000138
\item van den Dool HM. A New Look at Weather Forecasting through Analogues.
Monthly Weather Review. 1989; 117(10):2230–2247.
https://doi.org/10.1175/1520-0493(1989)117\%3C2230:ANLAWF\%3E2.0.CO;2
}
}
\seealso{
\code{\link{calcSeasCal}, \link{calcWatBal}}
}
