% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loss_fun.R
\name{loss_fun}
\alias{loss_fun}
\title{Loss Function for BKP and DKP Models}
\usage{
loss_fun(
  gamma,
  Xnorm,
  y = NULL,
  m = NULL,
  Y = NULL,
  model = c("BKP", "DKP"),
  prior = c("noninformative", "fixed", "adaptive"),
  r0 = 2,
  p0 = NULL,
  loss = c("brier", "log_loss"),
  kernel = c("gaussian", "matern52", "matern32")
)
}
\arguments{
\item{gamma}{A numeric vector of log10-transformed kernel hyperparameters.}

\item{Xnorm}{A numeric matrix of normalized input features (\code{[0,1]^d}).}

\item{y}{A numeric vector of observed successes (length \code{n}).}

\item{m}{A numeric vector of total binomial trials (length \code{n}),
corresponding to each \code{y}.}

\item{Y}{A numeric matrix of observed class counts (\code{n × q}), required
only when \code{model = "DKP"}, where \code{n} is the number of
observations and \code{q} the number of classes.}

\item{model}{A character string specifying the model type: \code{"BKP"}
(binary outcome) or \code{"DKP"} (multi-class outcome).}

\item{prior}{Type of prior: \code{"noninformative"} (default),
\code{"fixed"}, or \code{"adaptive"}.}

\item{r0}{Global prior precision (used when \code{prior = "fixed"} or
\code{"adaptive"}).}

\item{p0}{For BKP, a scalar in \code{(0,1)} specifying the prior mean of
success probability when \code{prior = "fixed"}. For DKP, a numeric vector
of length equal to the number of classes specifying the global prior mean,
which must sum to 1.}

\item{loss}{Loss function for kernel hyperparameter tuning: \code{"brier"}
(default) or \code{"log_loss"}.}

\item{kernel}{Kernel function for local weighting: \code{"gaussian"}
(default), \code{"matern52"}, or \code{"matern32"}.}
}
\value{
A single numeric value representing the total loss (to be minimized).
The value corresponds to either the Brier score (squared error) or the
log-loss (cross-entropy).
}
\description{
Computes the loss for fitting BKP (binary) or DKP (multi-class)
models. Supports Brier score (mean squared error) and log-loss
(cross-entropy) under different prior specifications.
}
\examples{
# -------------------------- BKP ---------------------------
set.seed(123)
n <- 10
Xnorm <- matrix(runif(2 * n), ncol = 2)
m <- rep(10, n)
y <- rbinom(n, size = m, prob = runif(n))
loss_fun(gamma = rep(0, 2), Xnorm = Xnorm, y = y, m = m, model = "BKP")

# -------------------------- DKP ---------------------------
set.seed(123)
n <- 10
q <- 3
Xnorm <- matrix(runif(2 * n), ncol = 2)
Y <- matrix(rmultinom(n, size = 10, prob = rep(1/q, q)), nrow = n, byrow = TRUE)
loss_fun(gamma = rep(0, 2), Xnorm = Xnorm, Y = Y, model = "DKP")

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\seealso{
\code{\link{fit_BKP}} for fitting BKP models, \code{\link{fit_DKP}}
for fitting DKP models, \code{\link{get_prior}} for constructing prior
parameters, \code{\link{kernel_matrix}} for computing kernel matrices.
}
