% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcov.R
\name{bcov.test}
\alias{bcov.test}
\title{Ball Covariance Test}
\usage{
bcov.test(x, y = NULL, R = 99, dst = FALSE, weight = FALSE,
  seed = 4, num.threads = 2)
}
\arguments{
\item{x}{a numeric vector, matirx, data.frame or \code{dist} object or list containing numeric vector, matrix, data.frame, or \code{dist} object.}

\item{y}{a numeric vector, matirx, data.frame or \code{dist} object.}

\item{R}{the number of replications. When \code{R = 0}, the function return ball covariance. Default: \code{R = 99}}

\item{dst}{if \code{dst = TRUE}, \code{x} and \code{y} will be considered as a distance matrix. Default: \code{dst = FALSE}}

\item{weight}{a logical or character value used to choose the form of weight. 
If \code{weight = FALSE}, the ball covariance / correlation with constant weight is used. 
Alternatively, \code{weight = TRUE} and \code{weight = "prob"} indicates the probability weight is chosen 
while setting \code{weight = "chisq"} means select the Chi-square weight. 
Note that this arguments actually only influences the printed result in R console and is 
only available for the \code{bcov.test} function at present. Default: \code{weight = FALSE}}

\item{seed}{the random seed.}

\item{num.threads}{Number of threads. Default \code{num.threads = 2}.}
}
\value{
bcov.test returns a list with class "htest" containing the following components:
\item{\code{statistic}}{ball covariance or ball correlation statistic.}            
\item{\code{p.value}}{the p-value for the test.}  
\item{\code{replicates}}{replicates of the test statistic.}
\item{\code{size}}{sample size.} 
\item{\code{complete.info}}{a \code{list} containing multiple statistics value and their corresponding $p$ value.}
\item{\code{alternative}}{a character string describing the alternative hypothesis.}
\item{\code{method}}{a character string indicating what type of test was performed.}
\item{\code{data.name}}{description of data.}
}
\description{
Ball covariance test of multivariate independence. 
Ball covariance are generic multivariate measures of dependence in Banach space.
}
\details{
\code{bcov.test} are non-parametric tests of multivariate independence in Banach space. 
The test decision is obtained via permutation, with \code{R} replicates.

If two samples are pass to arguments \code{x} and \code{y}, the sample sizes (i.e. number of rows or length of the vector) 
of the two variables must agree. If a \code{\link{list}} object is passed to \code{x}, 
each element must with same sample sizes. Moreover, data pass to \code{x} or \code{y} 
must not contain missing or infinite values. 
If we set \code{dst = TRUE}, arguments \code{x}, \code{y} can be a \code{dist} object or a
symmetric numeric matrix recording distance between samples; 
otherwise, these arguments are treated as data.

The \code{bcov.test} statistic is \code{bcov} or \code{ bcor} which are dependence measure 
in Banach space. The \code{bcor} test statistic is based on the normalized 
coefficient of ball covariance. (See the manual page for \code{\link{bcov}} or \code{\link{bcor}}.)

For the general problem of testing independence when the distributions of \eqn{X} and 
\eqn{Y} are unknown, the test based on \code{bcov} can be implemented as a permutation test.
See (Pan et al 2017) for theoretical properties of the test, including statistical consistency.
}
\examples{
set.seed(1)

################# Quick Start #################
error <- runif(50, min = -0.3, max = 0.3)
x <- runif(50, 0, 4*pi)
y <- cos(x) + error
# plot(x, y)
bcov.test(x, y)

################# Quick Start #################
x <- matrix(runif(50 * 2, -pi, pi), nrow = 50, ncol = 2)
error <- runif(50, min = -0.3, max = 0.3)
y <- (sin((x[,1])^2 + x[,2])) + error
bcov.test(x = x, y = y)

################# Ball Covariance Test for Non-Hilbert Data #################
# load data:
data("ArcticLake")
# Distance matrix between y:
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
# Distance matrix between x:
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with BCov:
bcov.test(x = Dx, y = Dy, dst = TRUE)

################  Weighted Ball Covariance Test  #################
data("ArcticLake")
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with weighted BCov:
bcov.test(x = Dx, y = Dy, dst = TRUE, weight = TRUE)

################# Mutual Independence Test #################
x <- rnorm(30)
y <- (x > 0) * x + rnorm(30)
z <- (x <= 0) * x + rnorm(30)
data_list <- list(x, y, z)
bcov.test(data_list)

################# Mutual Independence Test for Meteorology data #################
data("meteorology")
bcov.test(meteorology)

}
\seealso{
\code{\link{bcov}}, \code{\link{bcor}}
}
\author{
Wenliang Pan, Xueqin Wang, Heping Zhang, Hongtu Zhu, Jin Zhu
}
