#' MCMC trajectories and posterior distributions plot
#'
#' This function uses the ouput of \code{\link{coda.samples}} to plot the trajectories of MCMC and densities of the posterior distributions
#' of the age - if it is calculated, palaeodose and equivalent dose dispersion parameters of the sample.
#' This function is used in the function \code{\link{Age_Computation}}, \code{\link{AgeS_Computation}} and \code{\link{Palaeodose_Computation}}.
#'
#' @param sample MCMC.list: that is generated by \code{\link{jags.model}} in \code{\link{Age_Computation}}.
#' @param size integer: length of each chain.
#' @param SampleNames character vector: names of the samples, used in the figure titles.
#' @param Nb_sample integer (with default): number of analysed samples.
#' @param Nb_chaines integer (with default): number of independent chains for the model (for more informations, see \code{\link{jags.model}}).
#' @param value integer vector (with default): position idex used to select age (if available), palaeodose and equivalent dose dispersion parameters for the first sample.
#' @param param character vector (with default): names of the selected parameters with value \code{A} for age (if available),
#' \code{D} for palaeodose and \code{sD} for equivalent dose dispersion.
#'
#' @return
#' A pdf file with the MCMC trajectories and posterior distributions for each parameter defined in \code{param}.
#' There is one page per sample, which is divided by \code{length(param)} vertically and by \code{2} horizontally. \cr
#' The first line of the figure correponds to the first parameter defined in \code{param}, and so on.
#' On each line, the plot on the left represents the MCMC trajectories, and the one on the right the posterior distributions of the parameter.
#'
#' @author Claire Christophe, Anne Philippe, Guillaume Guerin
#'
#' @seealso \code{\link{Age_Computation}}, \code{\link{AgeS_Computation}}, \code{\link{Palaeodose_Computation}},
#'  \code{\link{coda.samples}} and \code{\link{rjags}} packages.
#'
#' @examples
#' data(MCMCsample,envir = environment())
#' MCMC_plot(sample=MCMCsample,size=2000,SampleNames="GDB3")
#'
#' @export

MCMC_plot<-function(sample,
                    size,
                    SampleNames,
                    Nb_sample=1,
                    Nb_chaines=3,
                    value=c(0,Nb_sample,2*Nb_sample),
                    param=c("A","D","sD")){
  ##--- plot the trajectory of the markov chains and the density of the posterior density

  par(mfrow=c(length(param),2),mar=c(2,2,3,2),oma=c(3,2,3,3))

  for(nb in 1:Nb_sample){
    nm=0
    for(i in value){
      nm=nm+1
      plot(sample[1:size[1],(i+nb)],col=1,type="l",main=param[nm],xlab="", ylab="")
      for(j in 2:Nb_chaines){
        lines(sample[(1+(j-1)*size[1]):(j*size[1]),(i+nb)],col=j)
      }
      u=density(sample[1:size[1],(i+nb)])
      plot(u$x,u$y,type='l',col=1,main="")
      for(j in 2:Nb_chaines){
        u=density(sample[(1+(j-1)*size[1]):(j*size[1]),(i+nb)])
        lines(u$x,u$y,col=j)
      }
      mtext(paste("MCMC plot of sample: ",SampleNames[nb]), outer = TRUE, cex = 1.5)
    }
  }
}
