% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VariableSelection.R
\name{VariableSelection}
\alias{VariableSelection}
\alias{VariableSelection.formula}
\alias{VariableSelection.BranchGLM}
\title{Variable Selection for GLMs}
\usage{
VariableSelection(object, ...)

\method{VariableSelection}{formula}(
  object,
  data,
  family,
  link,
  offset = NULL,
  method = "Fisher",
  type = "forward",
  metric = "AIC",
  keep = NULL,
  maxsize = NULL,
  grads = 10,
  parallel = FALSE,
  nthreads = 8,
  tol = 1e-04,
  maxit = NULL,
  contrasts = NULL,
  showprogress = TRUE,
  ...
)

\method{VariableSelection}{BranchGLM}(
  object,
  type = "forward",
  metric = "AIC",
  keep = NULL,
  maxsize = NULL,
  method = "Fisher",
  grads = 10,
  parallel = FALSE,
  nthreads = 8,
  tol = 1e-04,
  maxit = NULL,
  showprogress = TRUE,
  ...
)
}
\arguments{
\item{object}{a formula or a \code{BranchGLM} object.}

\item{...}{further arguments passed to other methods.}

\item{data}{a dataframe with the response and predictor variables.}

\item{family}{distribution used to model the data, one of "gaussian", "gamma", "binomial", or "poisson".}

\item{link}{link used to link mean structure to linear predictors. One of
"identity", "logit", "probit", "cloglog", "sqrt", "inverse", or "log".}

\item{offset}{offset vector, by default the zero vector is used.}

\item{method}{one of "Fisher", "BFGS", or "LBFGS". Fisher's scoring is recommended
for forward selection and branch and bound selection since they will typically 
fit many models with a small number of covariates.}

\item{type}{one of "forward", "backward", or "branch and bound" to indicate 
which type of variable selection to perform.}

\item{metric}{metric used to choose model, the default is "AIC", but "BIC" is also available.}

\item{keep}{vector of names to denote variables that must be in the model.}

\item{maxsize}{maximum number of variables to consider in a single model, the 
default is the total number of variables.
This number adds onto any variables specified in keep.}

\item{grads}{number of gradients used to approximate inverse information with, only for \code{method = "LBFGS"}.}

\item{parallel}{one of TRUE or FALSE to indicate if parallelization should be used}

\item{nthreads}{number of threads used with OpenMP, only used if \code{parallel = TRUE}.}

\item{tol}{tolerance used to determine model convergence.}

\item{maxit}{maximum number of iterations performed. The default for 
Fisher's scoring is 50 and for the other methods the default is 200.}

\item{contrasts}{see \code{contrasts.arg} of \code{model.matrix.default}.}

\item{showprogress}{whether to show progress updates for branch and bound.}
}
\value{
A \code{BranchGLMVS} object which is a list with the following components
\item{\code{finalmodel}}{ the final \code{BranchGLM} model selected}
\item{\code{variables}}{ a vector corresponding to the selected variables}
\item{\code{numchecked}}{ number of models fit}
\item{\code{order}}{ the order the variables were added to the model or removed from the model, this is not included for branch and bound selection}
\item{\code{type}}{ type of variable selection employed}
\item{\code{keep}}{ character vector of variables kept in each model, NULL if none specified}
\item{\code{metric}}{ metric used to select model}
\item{\code{bestmetric}}{ the best metric found in the search}
}
\description{
Performs forward selection, backward elimination, 
and branch and bound selection for generalized linear models.
}
\details{
The model in the formula or the formula from the fitted model is 
treated as the upper model. The variables specified in keep along with an 
intercept (if included in formula) is the lower model. When an intercept is 
included in the model formula it is kept in each model. Interaction terms 
are not properly handled, i.e. an interaction term may be kept while removing 
the lower-order terms. Factor variables are either kept in their entirety or 
entirely removed.

The branch and bound method makes use of an efficient branch and bound algorithm 
to find the optimal model. This is will find the best model according to the metric, but 
can be much faster than an exhaustive search. The amount of speedup attained by 
using the branch and bound method as opposed to an exhaustive search depends on 
the specific problem. Sometimes it may not be able to prune much and must 
fit most of the models and sometimes it may be able to prune off many of the models.

Fisher's scoring is recommended for branch and bound selection and forward selection.
All observations that have any missing values in the upper model are ignored.
}
\examples{
Data <- iris
Fit <- BranchGLM(Sepal.Length ~ ., data = Data, family = "gaussian", link = "identity")

### Doing branch and bound selection 
VariableSelection(Fit, type = "branch and bound", metric = "BIC")

### Now doing it in parallel (although it isn't necessary for this dataset)
VariableSelection(Fit, type = "branch and bound", parallel = TRUE, metric = "BIC")

### Using a formula
VariableSelection(Sepal.Length ~ ., data = Data, family = "gaussian", 
link = "identity", metric = "BIC", type = "branch and bound")

### Using the keep argument
VariableSelection(Fit, type = "branch and bound", keep = "Petal.Width", metric = "BIC")
}
