#' Finds the MMS estimator of the type II Pareto distribution parameters
#' using the Bayesian method (and the R code) developed by
#' Zhang and Stevens (2009).
#'
#' @title Estimation of parameters for the Pareto-II distribution (MMSE)
#' @param x a non-negative numeric vector.
#' @return
#' The list  with the following components is passed as a result:
#' \tabular{ll}{
#' \code{k} \tab	the estimated parameter of shape.\cr
#' \code{s} \tab	the estimated parameter of scale.\cr
#' }
#' @export
#' @seealso \code{\link{dpareto2}}, \code{\link{pareto2.goftest}}, \code{\link{pareto2.mlekestimate}}
#' @references
#' Zhang J., Stevens M.A., A New and Efficient Estimation Method for the Generalized Pareto Distribution, Technometrics 51(3), 2009, 316-325.\cr
pareto2.zsestimate <- function(x)
{
	if (mode(x) != "numeric") stop("'x' should be numeric");
	x <- x[!is.na(x)];
	n <- length(x);
	x <- sort(x);

	if (n < 2) stop("'x' should be of length at least 2");
	if (x[n] <= 0.0) stop("'x' should be non-negative");


	lx <- function(b, x)
	{
		k <- -mean(log(1-b*x));
		log(b/k)+k-1;
	}

	m <- 20+floor(sqrt(n))

	b <- w <- L <- 1/x[n]+(1-sqrt(m/((1:m)-0.5)))/3/x[floor(n/4+0.5)]

	for (i in 1:m) L[i] <- n*lx(b[i],x)

	for (i in 1:m) w[i]<- 1/sum(exp(L-L[i]))

	b <- sum(b*w);

	k <- 1/mean(log(1-b*x));
	s <- -1/b;

	if (k<=0)  warning("estimated shape parameter <= 0");
	if (s<=1)  warning("estimated scale parameter <= 1");

	list(k=k, s=s);
}


#' Finds the unbiased maximum likelihood estimator of the type II Pareto distribution's
#' shape parameter \eqn{k} for known scale parameter \eqn{s}.
#'
#' @title Estimation of shape parameter for the Pareto-II distribution (MLE)
#' @param x a non-negative numeric vector.
#' @param s scale parameter, \eqn{s>0}.
#' @return
#' A single numeric value is returned, the unbiased ML estimator of \eqn{k}.
#' @export
#' @seealso \code{\link{dpareto2}}, \code{\link{pareto2.goftest}}, \code{\link{pareto2.zsestimate}}
pareto2.mlekestimate <- function(x, s)
{
	if (mode(s) != "numeric" || length(s) != 1 || s <= 0) stop("'s' should be > 0");

	if (mode(x) != "numeric") stop("'x' should be numeric");
	x <- x[!is.na(x)];
	n <- length(x);

	if (n < 2) stop("'x' should be of length at least 2");
	if (any(x < 0.0)) stop("'x' should be non-negative");

	return((n-1)/sum(log(1+x/s)));
}


