#' Plots Brier Curve
#'
#' This function plots the brier curve based on a set of predictions generated by a binary classifier. Brier curves allow an evaluation of classifier performance in cost space.
#'
#' @param bc A \code{brierCurve} object created by the \code{brierCurve} function
#' @param curveType the type of Brier curve to be plotted. Shoul be "brierCost" or"brierSkew".
#' @export
#' @import data.table
#' @return None
#' @author Koen W. De Bock, \email{kdebock@@audencia.com}
#' @references Hernandez-Orallo, J., Flach, P., & Ferri, C. (2011). Brier Curves: a New Cost-Based Visualisation of Classifier Performance. Proceedings of the 28th International Conference on Machine Learning (ICML-11), 585–592.
#' @seealso \code{\link{brierCurve}}, \code{\link{CSMES.ensNomCurve}}
#' @examples
#' ##load data
#' library(rpart)
#' data(BFP)
#' ##generate random order vector
#' BFP_r<-BFP[sample(nrow(BFP),nrow(BFP)),]
#' size<-nrow(BFP_r)
#' ##size<-300
#' train<-BFP_r[1:floor(size/3),]
#' val<-BFP_r[ceiling(size/3):floor(2*size/3),]
#' test<-BFP_r[ceiling(2*size/3):size,]
#' ##train CART decision tree model
#' model=rpart(as.formula(Class~.),train,method="class")
#' ##generate predictions for the tes set
#' preds<-predict(model,newdata=test)[,2]
#' ##calculate brier curve
#' bc<-brierCurve(test[,"Class"],preds)
#' ##plot briercurve
#' plotBrierCurve(bc,curveType="cost")

plotBrierCurve<-function(bc,curveType=c("brierCost","brierSkew")){
  if (curveType=="brierCost"){
    plot(bc[[3]],type="l",ylab="Loss",xlab="Cost",title="Loss-based Brier curve")
  } else if (curveType=="brierSkew"){
    plot(bc[[4]],type="l",ylab="Loss",xlab="Skew",title="skew-based Brier curve")
  }
}
