\name{MackChainLadder}
\alias{MackChainLadder}
\encoding{UTF-8}
\title{ Mack Chain-Ladder Model }
\description{
  The Mack chain-ladder model forecasts future claims developments
  based on a historical cumulative claims development triangle and estimates
  the standard error around those.
}
\usage{
MackChainLadder(Triangle, weights = 1, alpha=1, est.sigma="log-linear",
tail=FALSE, tail.se=NULL, tail.sigma=NULL, mse.method="Mack")
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{Triangle}{cumulative claims triangle.  Assume columns are the development
    period, use transpose otherwise.
    A (mxn)-matrix \eqn{C_{ik}} 
    which is filled for \eqn{k \leq n+1-i; i=1,\ldots,m; m\geq n }, see
    \code{\link{qpaid}} for how to use (mxn)-development triangles with
    m<n, say higher development period frequency (e.g quarterly) than
    origin period frequency (e.g accident years).}
  \item{weights}{ weights. Default: 1, which sets the weights for all
    triangle entries to 1. Otherwise specify weights as a matrix of the same
    dimension as \code{Triangle} with all weight entries in [0; 1].
    Hence, any entry set to 0 or \code{NA} eliminates that age-to-age factor 
    from inclusion in the model. See also 'Details'}
  \item{alpha}{'weighting' parameters.
    Default: 1 for all development periods; alpha=1 gives the historical 
    chain-ladder age-to-age factors, alpha=0 gives the straight average of the
    observed individual development factors and alpha=2 is the result of
    an ordinary regression of \eqn{C_{i,k+1}} against \eqn{C_{i,k}} with
    intercept 0, see also 'Details' below, \code{\link{chainladder}} and Mack's 1999 paper}
  \item{est.sigma}{defines how to estimate \eqn{sigma_{n-1}}, the
    variability of the individual age-to-age factors at development time
    \eqn{n-1}.
    Default is "log-linear" for a log-linear regression, "Mack" for
    Mack's approximation from his 1999 paper. Alternatively the user can
    provide a numeric value. If the log-linear model appears to be
    inappropriate (p-value > 0.05) the 'Mack' method will be used
    instead and a warning message printed.
    Similarly, if Triangle is so small that log-linear regression is being
    attempted on a vector of only one non-NA average link ratio,
    the 'Mack' method will be used instead and a warning message printed.
    }  
  \item{tail}{can be logical or a numeric value.
    If \code{tail=FALSE} no tail factor will be applied, if \code{tail=TRUE} 
    a tail factor will be estimated via a linear extrapolation of
    \eqn{log(chain-ladder factors - 1)}, if \code{tail} is a numeric
    value than this value will be used instead.}
  \item{tail.se}{defines how the standard error of the tail factor is estimated. Only needed if
    a tail factor > 1 is provided. Default is \code{NULL}. If
    \code{tail.se} is \code{NULL}, \code{tail.se} is estimated via
    "log-linear" regression, if \code{tail.se} is a numeric
    value than this value will be used instead.}  
  \item{tail.sigma}{defines how to estimate individual tail variability. Only needed if
      a tail factor > 1 is provided.
    Default is \code{NULL}. If \code{tail.sigma} is \code{NULL},
    \code{tail.sigma} 
    is estimated via "log-linear" regression, if \code{tail.sigma} is a numeric
    value than this value will be used instead}  
  \item{mse.method}{
    method used for the recursive estimate of the parameter risk component 
    of the mean square error.
    Value "Mack" (default) coincides with Mack's formula;
    "Independence" includes the additional cross-product term
    as in Murphy and BBMW. 
    Refer to References below.
    }  
}
\details{
  Following Mack's 1999 paper let \eqn{C_{ik}} denote the cumulative loss 
  amounts of origin period (e.g. accident year) 
  \eqn{i=1,\ldots,m}, with losses known for development period
  (e.g. development year) \eqn{k \le n+1-i}.
  In order to forecast the amounts \eqn{C_{ik}} for
  \eqn{k > n+1-i} the Mack chain-ladder-model assumes:
  \deqn{\mbox{CL1: }  E[ F_{ik}| C_{i1},C_{i2},\ldots,C_{ik} ] = f_k
    \mbox{ with } F_{ik}=\frac{C_{i,k+1}}{C_{ik}}
  }{CL1: E[ F_ik| C_i1,C_i2,...,C_ik ] = f_k with F_ik=C_{i,k+1}/C_ik}
  
  \deqn{\mbox{CL2: }  Var( \frac{C_{i,k+1}}{C_{ik}} | C_{i1},C_{i2},
    \ldots,C_{ik} ) = \frac{\sigma_k^2}{w_{ik} C^\alpha_{ik}}
  }{CL2: Var( C_{i,k+1}/C_ik | C_i1, C_i2, ... ,C_ik ) = sigma_k^2/( w_ik C^alpha_ik)
  }
  \deqn{\mbox{CL3: }  \{ C_{i1},\ldots,C_{in}\}, \{
    C_{j1},\ldots,C_{jn}\},\mbox{ are independent for origin period } i
    \neq j 
  }{CL3: \{ C_i1, ... ,C_in\}, \{
    C_j1, ... ,C_jn\}, are independent for origin period i != j 
  }
  with \eqn{w_{ik} \in [0;1]}{w_{ik} in [0;1]}, \eqn{\alpha \in \{0,1,2\}}{alpha in \{0,1,2\}}.   
  If these assumptions hold, the Mack chain-ladder gives an
  unbiased estimator for IBNR (Incurred But Not Reported) claims.
  
  Here \eqn{w_{ik} are the \code{weights} from above.}
  
  The Mack chain-ladder model can be regarded as a special form of a 
  weighted linear regression
  through the origin for each development period:
  \code{lm(y ~ x  + 0, weights=weights/x^(2-alpha))}, 
  where \code{y} is the vector of claims at development period
  \eqn{k+1} and \code{x} is  the vector of claims at development period
  \eqn{k}. 

It is necessary, before actually applying the model, to check if the main assumptions behind the model (i.e. Calendar Year Effect and Correlation between subsequent Accident Years, see \code{\link{dfCorTest}}, \code{\link{cyEffTest}}) are verified.

}
\value{
  MackChainLadder returns a list with the following elements
  \item{call}{matched call} 
  \item{Triangle}{input triangle of cumulative claims}
  \item{FullTriangle}{forecasted full triangle}
  \item{Models}{linear regression models for each development period}
  \item{f}{chain-ladder age-to-age factors}
  \item{f.se}{standard errors of the chain-ladder age-to-age factors f
    (assumption CL1)}
  \item{F.se}{standard errors of the true chain-ladder age-to-age factors
    \eqn{F_{ik}} (square root of the variance in assumption CL2)}
  \item{sigma}{sigma parameter in CL2}
  \item{Mack.ProcessRisk}{variability in the projection of future losses
    not explained by the variability of the link ratio estimators (unexplained variation)}
  \item{Mack.ParameterRisk}{variability in the projection of future
    losses explained by the variability of the link-ratio estimators
    alone (explained variation)}
  \item{Mack.S.E}{total variability in the projection of future losses
    by the chain-ladder method; the square root of the mean square error
    of the chain-ladder estimate: \eqn{\mbox{Mack.S.E.}^2 = \mbox{Mack.ProcessRisk}^2 +
    \mbox{Mack.ParameterRisk}^2}{Mack.S.E.^2 = Mack.ProcessRisk^2 +
  Mack.ParameterRisk^2}}
  \item{Total.Mack.S.E}{total variability of projected loss for all
    origin years combined}
  \item{Total.ProcessRisk}{vector of process risk estimate of the total 
    of projected loss for all origin years combined by development period}
  \item{Total.ParameterRisk}{vector of parameter risk estimate of the total 
    of projected loss for all origin years combined by development period}
  \item{weights}{weights used}
  \item{alpha}{alphas used}
  \item{tail}{tail factor used. If tail was set to TRUE the output will
    include the linear model used to estimate the tail factor}
}
\references{
\cite{Thomas Mack. Distribution-free calculation of the standard error of chain ladder reserve estimates. \emph{Astin Bulletin}. Vol. 23. No 2. 1993. pp.213:225}

\cite{Thomas Mack. The standard error of chain ladder reserve estimates: Recursive calculation and inclusion of a tail factor. \emph{Astin Bulletin}. Vol. 29. No 2. 1999. pp.361:366}

\cite{Murphy, Daniel M. Unbiased Loss Development
Factors. \emph{Proceedings of the Casualty Actuarial Society Casualty
Actuarial Society} - Arlington, Virginia 1994: LXXXI 154-222}

\cite{Buchwalder, Bühlmann, Merz, and Wüthrich.
  The Mean Square Error of Prediction in the Chain Ladder Reserving Method
  (Mack and Murphy Revisited).
  \emph{Astin Bulletin} Vol. 36. 2006. pp.521:542
  }

}
\author{Markus Gesmann \email{markus.gesmann@gmail.com}}
\note{
  Additional references for further reading:
  
  \cite{England, PD and Verrall, RJ. Stochastic Claims Reserving in
  General Insurance (with discussion), \emph{British Actuarial Journal}
  8, III. 2002}

  \cite{Barnett and Zehnwirth. Best estimates for reserves. Proceedings
    of the CAS, LXXXVI I(167), November 2000.}  
  
% ~Make other sections like Warning with \section{Warning }{....} ~
}
\seealso{ See also \code{\link{qpaid}} for dealing with non-square triangles,  
  \code{\link{chainladder}} for the underlying chain-ladder method,  
  \code{\link{dfCorTest}} to check for Calendar Year Effect,
  \code{\link{cyEffTest}} to check for Development Factor Correlation,
  \code{\link{summary.MackChainLadder}},  
  \code{\link{quantile.MackChainLadder}},  
  \code{\link{plot.MackChainLadder}} and 
  \code{\link{residuals.MackChainLadder}} displaying results, 
  \code{\link{CDR.MackChainLadder}} for the one year claims development result.
}

\examples{
## See the Taylor/Ashe example in Mack's 1993 paper
GenIns
plot(GenIns)
plot(GenIns, lattice=TRUE)
GNI <- MackChainLadder(GenIns, est.sigma="Mack")
GNI$f
GNI$sigma^2
GNI # compare to table 2 and 3 in Mack's 1993 paper
plot(GNI)
plot(GNI, lattice=TRUE)

## Different weights
## Using alpha=0 will use straight average age-to-age factors 
MackChainLadder(GenIns, alpha=0)$f
# You get the same result via:
apply(GenIns[,-1]/GenIns[,-10],2, mean, na.rm=TRUE)

## Only use the last 5 diagonals, i.e. the last 5 calendar years
calPeriods <- (row(GenIns) + col(GenIns) - 1)
(weights <- ifelse(calPeriods <= 5, 0, ifelse(calPeriods > 10, NA, 1)))
MackChainLadder(GenIns, weights=weights, est.sigma = "Mack")

## Tail
## See the example in Mack's 1999 paper
Mortgage
m <- MackChainLadder(Mortgage)
round(summary(m)$Totals["CV(IBNR)",], 2) ## 26% in Table 6 of paper
plot(Mortgage)
# Specifying the tail and its associated uncertainty parameters
MRT <- MackChainLadder(Mortgage, tail=1.05, tail.sigma=71, tail.se=0.02, est.sigma="Mack")
MRT
plot(MRT, lattice=TRUE)
# Specify just the tail and the uncertainty parameters will be estimated
MRT <- MackChainLadder(Mortgage, tail=1.05)
MRT$f.se[9] # close to the 0.02 specified above
MRT$sigma[9] # less than the 71 specified above
# Note that the overall CV dropped slightly
round(summary(MRT)$Totals["CV(IBNR)",], 2) ## 24%
# tail parameter uncertainty equal to expected value 
MRT <- MackChainLadder(Mortgage, tail=1.05, tail.se = .05)
round(summary(MRT)$Totals["CV(IBNR)",], 2) ## 27%

## Parameter-risk (only) estimate of the total reserve = 3142387
tail(MRT$Total.ParameterRisk, 1) # located in last (ultimate) element
#  Parameter-risk (only) CV is about 19%
tail(MRT$Total.ParameterRisk, 1) / summary(MRT)$Totals["IBNR", ]

## Three terms in the parameter risk estimate
## First, the default (Mack) without the tail
m <- MackChainLadder(RAA, mse.method = "Mack")
summary(m)$Totals["Mack S.E.",]
## Then, with the third term
m <- MackChainLadder(RAA, mse.method = "Independence")
summary(m)$Totals["Mack S.E.",] ## Not significantly greater

## One year claims development results
M <- MackChainLadder(MW2014, est.sigma="Mack")
CDR(M)

## For more examples see:
\dontrun{
 demo(MackChainLadder)
}

}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ models }
