% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimationCKT.KendallReg.R
\name{CKT.kendallReg.fit}
\alias{CKT.kendallReg.fit}
\alias{CKT.kendallReg.predict}
\title{Fit Kendall's regression, a GLM-type model for conditional Kendall's tau}
\usage{
CKT.kendallReg.fit(
  observedX1,
  observedX2,
  observedZ,
  ZToEstimate,
  designMatrixZ = cbind(ZToEstimate, ZToEstimate^2, ZToEstimate^3),
  newZ = designMatrixZ,
  h_kernel,
  Lambda = identity,
  Lambda_inv = identity,
  lambda = NULL,
  Kfolds_lambda = 10,
  l_norm = 1,
  h_lambda = h_kernel,
  ...
)

CKT.kendallReg.predict(fit, newZ, lambda = NULL, Lambda_inv = identity)
}
\arguments{
\item{observedX1}{a vector of \code{n} observations of the first variable \eqn{X_1}.}

\item{observedX2}{a vector of \code{n} observations of the second variable \eqn{X_2}.}

\item{observedZ}{a vector of \code{n} observations of the conditioning variable,
or a matrix with \code{n} rows of observations of the conditioning vector
(if \eqn{Z} is multivariate).}

\item{ZToEstimate}{the intermediary dataset of observations of \eqn{Z}
at which the conditional Kendall's tau should be estimated.}

\item{designMatrixZ}{the transformation of the \code{ZToEstimate} that
will be used as predictors. By default, no transformation is applied.}

\item{newZ}{the new observations of the conditioning variable.}

\item{h_kernel}{bandwidth used for the first step of kernel smoothing.}

\item{Lambda}{the function to be applied on conditional Kendall's tau.
By default, the identity function is used.}

\item{Lambda_inv}{the functional inverse of \code{Lambda}.
By default, the identity function is used.}

\item{lambda}{the regularization parameter. If \code{NULL},
then it is chosen by K-fold cross validation.
Internally, cross-validation is performed by the function
\code{\link{CKT.KendallReg.LambdaCV}}.}

\item{Kfolds_lambda}{the number of folds used in the cross-validation
procedure to choose \code{lambda}.}

\item{l_norm}{type of norm used for selection of the optimal lambda by cross-validation.
\code{l_norm=1} corresponds to the sum of absolute values of differences
between predicted and estimated conditional Kendall's tau
while \code{l_norm=2} corresponds to the sum of squares of differences.}

\item{h_lambda}{the smoothing bandwidth used in the cross-validation
procedure to choose \code{lambda}.}

\item{...}{other arguments to be passed to \code{\link{CKT.kernel}}
for the first step (kernel-based) estimator of conditional Kendall's tau.}

\item{fit}{the fitted model, obtained by a call
to \code{CKT.kendallReg.fit}.}
}
\value{
The function \code{CKT.kendallReg.fit} returns
a list with the following components:
\itemize{
    \item \code{estimatedCKT}: the estimated CKT at the new data points \code{newZ}.

    \item \code{fit}: the fitted model, of S3 class glmnet
    (see \code{glmnet::\link[glmnet]{glmnet}} for more details).

    \item \code{lambda}: the value of the penalized parameter used.
    (i.e. either the one supplied by the user or
    the one determined by cross-validation)
}

\code{CKT.kendallReg.predict} returns
the predicted values of conditional Kendall's tau.
}
\description{
The function \code{CKT.kendallReg.fit} fits a regression-type model for the
conditional Kendall's tau between two variables \eqn{X_1} and \eqn{X_2}
conditionally to some predictors Z.
More precisely, it fits the model
\deqn{\Lambda(\tau_{X_1, X_2 | Z = z}) = \sum_{j=1}^{p'} \beta_j \psi_j(z),}
where \eqn{\tau_{X_1, X_2 | Z = z}} is the conditional Kendall's tau
between \eqn{X_1} and \eqn{X_2} conditionally to \eqn{Z=z},
\eqn{\Lambda} is a function from \eqn{]-1, 1]} to \eqn{R},
\eqn{(\beta_1, \dots, \beta_p)} are unknown coefficients to be estimated
and \eqn{\psi_1, \dots, \psi_{p'})} are a dictionary of functions.
To estimate \eqn{beta}, we used the penalized estimator which is defined
as the minimizer of the following criteria
\deqn{\frac{1}{2n'} \sum_{i=1}^{n'} [\Lambda(\hat\tau_{X_1, X_2 | Z = z_i})
- \sum_{j=1}^{p'} \beta_j \psi_j(z_i)]^2 + \lambda * |\beta|_1,}
where the \eqn{z_i} are a second sample (here denoted by \code{ZToEstimate}).

The function \code{CKT.kendallReg.predict} predicts
the conditional Kendall's tau between two variables
\eqn{X_1} and \eqn{X_2} given \eqn{Z=z} for some new
values of \eqn{z}.
}
\examples{
# We simulate from a conditional copula
set.seed(1)
N = 400
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = -0.9 + 1.8 * pnorm(Z, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1])
X2 = qnorm(simCopula[,2])

newZ = seq(2, 10, by = 0.1)
estimatedCKT_kendallReg <- CKT.kendallReg.fit(
   observedX1 = X1, observedX2 = X2, observedZ = Z,
   ZToEstimate = newZ, h_kernel = 0.07)

coef(estimatedCKT_kendallReg$fit,
     s = estimatedCKT_kendallReg$lambda)

# Comparison between true Kendall's tau (in black)
# and estimated Kendall's tau (in red)
trueConditionalTau = -0.9 + 1.8 * pnorm(newZ, mean = 5, sd = 2)
plot(newZ, trueConditionalTau , col="black",
   type = "l", ylim = c(-1, 1))
lines(newZ, estimatedCKT_kendallReg$estimatedCKT, col = "red")



}
\references{
Derumigny, A., & Fermanian, J. D. (2020).
On Kendall’s regression.
Journal of Multivariate Analysis, 178, 104610.
\doi{10.1016/j.jmva.2020.104610}
}
\seealso{
See also other estimators of conditional Kendall's tau:
\code{\link{CKT.fit.tree}}, \code{\link{CKT.fit.randomForest}},
\code{\link{CKT.fit.nNets}}, \code{\link{CKT.predict.kNN}},
\code{\link{CKT.kernel}}, \code{\link{CKT.fit.GLM}},
and the more general wrapper \code{\link{CKT.estimate}}.

See also the test of the simplifying assumption that a
conditional copula does not depend on the value of the
conditioning variable using the nullity of Kendall's regression
coefficients: \code{\link{simpA.kendallReg}}.
}
