\name{cov.sel}
\alias{cov.sel}
\title{ Model-Free Selection of Covariate Sets }
\description{
Dimension reduction of the covariate vector under unconfoundedness using model-free backward elimination algorithms, based on either marginal co-ordinate hypothesis testing, (MCH), (continuous covariates only) or kernel-based smoothing, (KS).
}
\usage{cov.sel(T, Y, X, type=c("dr", "np"), alg = 3,scope = NULL, alpha = 0.1,
		thru=0.5,thro=0.25,thrc=100,...)
}
\arguments{
  \item{T  }{A vector, containing \code{0} and \code{1}, indicating the binary treatment variable.}
  \item{Y  }{A vector of observed outcomes.}
  \item{X  }{A matrix or data frame containing columns of covariates. The covariates may be a mix of continuous, unordered discrete
(to be specified in the data frame using \code{factor}), and ordered discrete (to be specified in the data frame using \code{ordered}).}
  \item{type }{The type of method used. \code{"dr"} for MCH and \code{"np"} for KS. MCH is suitable in situations with only continuous covariates while KS can be used if discrete covariates are present.} 
   \item{alg  }{Specifying which algorithm to be use. \code{1} 
  indicates Algorithm A, \code{2} indicates Algorithm B and 
  \code{3} runs them both. See Details. \code{alg = 3} is default.}
  \item{scope  }{A character string giving the name of one (or 
  several) covariate(s) that must not be removed.}
\item{alpha  }{Stopping criterion for MCH: will stop removing covariates
   when the p-value for the next covariate to be removed is less 
   then \code{alpha}. The default is \code{alpha = 0.1}.}
  \item{thru }{Bandwidth threshold used for unordered discrete covariates if \code{type="np"}. Values in \eqn{[0,1]} are valid. \code{thru=0} removes all unordered discrete covariates and \code{thru=1} removes none of them. Default is \code{thru=0.5}. See Details.}
   \item{thro }{Bandwidth threshold used for ordered discrete covariates if \code{type="np"}.  Values in \eqn{[0,1]} are valid. \code{thro=0} removes all unordered discrete covariates and \code{thro=1} removes none of them. Default is \code{thro=0.25}. See Details.}
   \item{thrc }{Bandwidth threshold used for continuous covariates if \code{type="np"}. Non-negative values are valid. Default is \code{thr=100}. See Details.}
 \item{\dots  }{Additional arguments passed on to \code{dr}, \code{dr.step} or \code{npregbw}. If \code{type="dr"},  
\code{method}, can be set to \code{"sir"} or \code{"save"}, the first being 
   default, \code{trace=0} supresses the output generated by \code{dr.step}. If \code{type="np"}, \code{regtype} can be set to \code{"lc"} or \code{"ll"}, the first being default and \code{bwtype} can be set to \code{"fixed"}, \code{"generalized_nn"} or \code{"adaptive_nn"}, defaults to \code{"fixed"}. See \code{dr} and \code{npregbw} for usage of \code{na.action}.}
}
\details{
Performs model-free selection of covariates for situations where the parameter of interest is an average causal effect. This function is based on the framework of sufficient dimension reduction, that under unconfoundedness, reduces dimension of the covariate vector. A two-step procedure searching for a sufficient subset of the covariate vector is implemented in the form of algorithms. This function uses MCH (if \code{type="dr"}) or KS (if \code{type="np"}) in the form of two backward elimination algorithms, Algorithm A and Algorithm B proposed by de Luna, Waernbaum and Richardson (2011).

Algorithm A (\code{alg = 1}): First the covariates conditionally independent of the treatment, \code{T}, given the rest of the variables (\code{X.T}) are removed. Then the covariates conditionally independent of the potential outcomes (in each of the treatment groups) given the rest of the covariates are removed. This yields two subsets of covariates; \code{Q.1} and \code{Q.0} for the treatment and control group respectively.

Algorithm B (\code{alg = 2}): First the covariates conditionally independent of the potential outcome (in each of the treatment groups), given the rest of the covariates (\code{X.0} and \code{X.1}) are removed. Then the covariates conditionally independent of the treatment, \code{T}, given the rest of the covariates are removed. This yields two subsets of covariates; \code{Z.1} and \code{Z.0} for the treatment and control group respectively. 

\code{alg=3} runs both Algorithm A and B.

In KS the bandwidth range for unordered discrete covariates is [0, 1/#levels] while for ordered discrete covariates, no matter how many levels, the range is [0, 1]. For continuous covariates bandwidths ranges from 0 to infinity. Ordered discrete and continuous covariates are removed if their bandwidths exceed their respective thresholds. Unordered discrete covariates are removed if their bandwidths are larger than \code{thru} times the maximum bandwidth.

In case of MCH one can choose between sliced inverse regression, SIR, or sliced average variance estimation, SAVE. For KS the regression type can be set to local constant kernel or local linear and the bandwidth type can be set to fixed, generalized nearest neighbors or adaptive nearest neighbors. See \code{dr} and \code{npregbw} for details. Since \code{type="np"} results in a fully nonparametric covariate selection procedure this can be much slower than if \code{type="dr"}.}

\value{
 \code{cov.sel} returns a list with the following content:
  \item{X.T  }{The of covariates with minimum cardinality such that \eqn{P(}\code{T}|\code{X}\eqn{) = P(}\code{T}|\code{X.T}\eqn{)}.}
  \item{Q.0  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{Y.0}|\code{X.T}\eqn{) = P(}\code{Y.0}|\code{Q.0}\eqn{)}. Where \code{Y.0} is the response in the control group.}
  \item{Q.1  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{Y.1}|\code{X.T}\eqn{) = P(}\code{Y.1}|\code{Q.1}\eqn{)}. Where \code{Y.1} is the response in the treatment group.}
  \item{X.0  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{Y.0}|\code{X}\eqn{) = P(}\code{Y.0}|\code{X.0}\eqn{)}.}
  \item{X.1  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{Y.1}|\code{X}\eqn{) = P(}\code{Y.1}|\code{X.1}\eqn{)}.}
  \item{Z.0  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{T}|\code{X.0}\eqn{) = P(}\code{T}|\code{Z.0}\eqn{)}.}
  \item{Z.1  }{The set of covariates with minimum cardinality such that \eqn{P(}\code{T}|\code{X.1}\eqn{) = P(}\code{T}|\code{Z.1}\eqn{)}.}
  If \code{type="dr"}  the following \code{type}-specific content is returned:
  \item{          evectorsQ.0  }{The eigenvectors of the matrix whose columns span the reduced subspace \code{Q.0}.}
  \item{          evectorsQ.1  }{The eigenvectors of the matrix whose columns span the reduced subspace \code{Q.1}.}
  \item{          evectorsZ.0  }{The eigenvectors of the matrix whose columns span the reduced subspace \code{Z.0}.}
  \item{          evectorsZ.1  }{The eigenvectors of the matrix whose columns span the reduced subspace \code{Z.1}.}
  \item{          method  }{The method used, either \code{"sir"} or \code{"save"}.}
  If \code{type="np"} the following \code{type}-specific content is returned:
  \item{          bandwidthsQ.0  }{The selected bandwidths for the covariates in the reduced subspace \code{Q.0}.}
  \item{          bandwidthsQ.1  }{The selected bandwidths for the covariates in the reduced subspace \code{Q.1}.}
  \item{          bandwidthsZ.0  }{The selected bandwidths for the covariates in the reduced subspace \code{Z.0}.}
  \item{          bandwidthsZ.1  }{The selected bandwidths for the covariates in the reduced subspace \code{Z.1}.}
  \item{          regtype  }{The regression method used, either \code{"lc"} or \code{"ll"}.}
  \item{          bwtype  }{Type of bandwidth used, \code{"fixed"}, \code{"generalized_nn"} or \code{"adaptive_nn"}}
  \item{covar  }{Names of all covariates given as input \code{X}.}  

  
For marginal co-ordinate hypothesis test, \code{type="dr"}, as a side effect a data frame of labels, tests, and p.values 
  is printed.  
}
\references{ 
Cook, R. D. (2004). Testing Predictor
contributions in Sufficient Dimension Reduction. \emph{The Annals of statistics 32}. 1061-1092

de Luna, X., I. Waernbaum, and T. S. Richardson (2011). Covariate selection for the nonparametric estimation of an average treatment effect. \emph{Biometrika 98}. 861-875

Hall, P., Q. Li and J.S. Racine (2007). Nonparametric estimation of regression functions in the presence of irrelevant regressors. \emph{The Review of Economics and Statistics, 89}. 784-789

Li, L., R. D. Cook, and C. J. Nachtsheim (2005). Model-free
Variable Selection. \emph{Journal of the Royal
Statistical Society, Series B 67}. 285-299
}

\author{ Emma Persson, <emma.persson@stat.umu.se>, Jenny H?ggstr?m, <jenny.haggstrom@stat.umu.se> }
\note{ \code{cov.sel} calls the functions \code{dr}, 
\code{dr.step} and \code{npregbw} so the packages \code{dr} and \code{np} are required.
}
\seealso{\code{\link[dr]{dr}}, \code{\link[np]{np}}}
\examples{
## Marginal co-ordinate hypothesis test, continuous covariates only

data(datc)


##Algorithm A, keeping x6 and x7
  
ans <- cov.sel(T = datc$T, Y = datc$y, X = datc[,1:8], type="dr",
               alpha = 0.1, alg = 1, scope=c("x6","x7"))

summary(ans)

##Algorithm B, method "save"
  
ans <- cov.sel(T = datc$T, Y = datc$y, X = datc[,1:10], type="dr",
               alg = 2, method = "save", alpha = 0.3, na.action = "na.omit")

## Kernel-based smoothing, both categorical and continuous covariates

data(datfc)
##The example below with default setting takes about 9 minutes to run.
## ans <- cov.sel(T = datfc$T, Y = datfc$y, X = datfc[,1:8], type="np",
##               alpha = 0.1, alg = 3, scope=NULL, thru=0.5, thro=0.25, thrc=100)

## For illustration purposes we run Algorithm A using only the first 100 observations 
##and x1, x2, x3, x4 in datfc
ans <- cov.sel(T = datfc$T[1:100], Y = datfc$y[1:100], X = datfc[1:100,1:4], 
      type="np",alpha = 0.1, alg = 1, scope=NULL, thru=0.5, thro=0.25, thrc=100)

##The example below running Algorithm A, keeping x6 and x7 with regtype="ll"
##takes about 7 minutes to run.
##ans <- cov.sel(T = datfc$T, Y = datfc$y, X = datfc[,1:8], type="np",
##               alpha = 0.1, alg = 3, scope=c("x6","x7"), thru=0.5, thro=0.25, 
##               thrc=100, regtype="ll")

}
