#The main EM routine.
require(mvtnorm)
require(matrixcalc)

#' dcem_cluster (multivariate data): Part of DCEM package.
#'
#' Implements the Expectation Maximization algorithm for multivariate data. This function is called
#' by the dcem_train routine.
#'
#' @param data A matrix: The dataset provided by the user.
#'
#' @param meu (matrix): The matrix containing the initial meu(s).
#'
#' @param sigma (list): A list containing the initial covariance matrices.
#'
#' @param prior (vector): A vector containing the initial prior.
#'
#' @param num_clusters (numeric): The number of clusters specified by the user. Default value is 2.
#'
#' @param iteration_count (numeric): The number of iterations for which the algorithm should run, if the
#' convergence is not achieved then the algorithm stops. Default: 200.
#'
#' @param threshold (numeric): A small value to check for convergence (if the estimated meu are within this
#' specified threshold then the algorithm stops and exit).
#'
#' \strong{Note: Choosing a very small value (0.0000001) for threshold can increase the runtime substantially
#' and the algorithm may not converge. On the other hand, choosing a larger value (0.1)
#' can lead to sub-optimal clustering. Default: 0.00001}.
#'
#' @param num_data (numeric): The total number of observations in the data.
#'
#' @return
#'         A list of objects. This list contains parameters associated with the
#'         Gaussian(s) (posterior probabilities, meu, co-variance and prior)
#'
#'\enumerate{
#'         \item (1) Posterior Probabilities:  \strong{prob} :A matrix of
#'         posterior-probabilities.
#'
#'         \item (2) Meu: \strong{meu}: It is a matrix of meu(s). Each row in
#'         the matrix corresponds to one meu.
#'
#'         \item (3) Sigma: Co-variance matrices: \strong{sigma}
#'
#'         \item (4) prior: \strong{prior}: A vector of prior.
#'         }
#'
#' @usage
#' dcem_cluster_mv(data, meu, sigma, prior, num_clusters, iteration_count,
#' threshold, num_data)
#'
#' @references
#' Using data to build a better EM: EM* for big data.
#'
#' Hasan Kurban, Mark Jenne, Mehmet M. Dalkilic
#' (2016) <https://doi.org/10.1007/s41060-017-0062-1>.
#'
#' @author Parichit Sharma \email{parishar@iu.edu}, Hasan Kurban, Mark Jenne, Mehmet Dalkilic
#'
#' This work is partially supported by NCI Grant 1R01CA213466-01.

dcem_cluster_mv <-
  function(data,
           meu,
           sigma,
           prior,
           num_clusters,
           iteration_count,
           threshold,
           num_data)

  {
    counter = 1

    weights <- matrix(0,
                      nrow = num_clusters,
                      ncol = num_data,
                      byrow = TRUE)

    tolerance <- .Machine$double.eps

    # Repeat till convergence threshold or iteration which-ever is earlier.
    while (counter <= iteration_count) {
      old_meu <- meu
      weights <- matrix(0,
                        nrow = num_clusters,
                        ncol = num_data,
                        byrow = TRUE)

      # Expectation
      weights = expectation_mv(data,
                               weights,
                               meu,
                               sigma,
                               prior,
                               num_clusters,
                               tolerance)

      # Maximisation
      out = maximisation_mv(data, weights, meu, sigma, prior, num_clusters, num_data)
      meu = out$meu
      sigma = out$sigma
      prior = out$prior

      # Find the difference in the meu
      meu_diff <- sqrt(sum((meu - old_meu) ^ 2))

      # Check convergence
      if (!is.na(meu_diff) && round(meu_diff, 4) < threshold) {
        print((paste(
          "Convergence at iteration num_clustersber: ", counter
        )))
        break
      }

      # Check iterations
      else if (counter == iteration_count) {
        print("Iteration threshold crossed. Stoping the execution.")
        break
      }
      counter = counter + 1

    }

    output = list(
      prob = weights,
      'meu' = meu,
      'sigma' = sigma,
      'prior' = prior
    )
    return(output)

  }
