\name{lhs.design}
\alias{lhs.design}
\alias{lhs.augment}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Functions for accessing latin hypercube sampling designs from package lhs 
   or space-filling designs from package DiceDesign}
\description{
  Functions for comfortably accessing latin hypercube sampling designs from package lhs 
  or space-filling designs from package DiceDesign, 
  which are useful for quantitative factors with many possible levels. In particular, 
  they can be used in computer experiments. Most of the designs are random samples.
}
\usage{
lhs.design(nruns, nfactors, type="optimum", factor.names=NULL, seed=NULL, digits=NULL, 
         nlevels = nruns, randomize = FALSE, ...)
lhs.augment(lhs, m=1, type="optAugment", seed=NULL, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{nruns}{ number of runs in the latin hypercube sample; \cr
        for type \code{fact} (a full factorial with equally-space levels), 
        if \code{nlevels} is not separately specified, 
        this number is taken to be the common number of levels of all factors, 
        i.e. the resulting design will have \code{nruns^nfactors} runs;\cr
        alternatively, if \code{nlevels} is separately specified as a vector of different numbers of levels, 
        \code{nruns} can be missing or can be the correctly-specified number of runs.
        }
  \item{nfactors}{ number of factors in the latin hypercube sample }
  \item{type}{ character string indicating the type of design or augmentation method;
        defaults are \dQuote{optimum} for \code{lhs.design} and \dQuote{optAugment} 
        for \code{lhs.augment}.\cr\cr
        Function \code{lhs.design} calls \cr
        a function named typeLHS from package 
        \pkg{lhs} (types \code{genetic}, \code{improved}, \code{maximin}, \code{optimum}, \code{random}), \cr
        a function named typeDesign from package \pkg{DiceDesign} (types \code{dmax}, \code{strauss}, \code{fact})\cr
        or function \code{runif.faure} from package \pkg{DiceDesign} (type \code{faure}).\cr \cr
        Function \code{lhs.augment} calls function typeLHS from package \pkg{lhs}, \cr
        where possible choices for \code{type} are \code{augment}, \code{optSeeded}, or \code{optAugment}.\cr 
        see the respective functions from packages \pkg{\link[lhs:lhs-package]{lhs}} and 
        \pkg{\link[DiceDesign:DiceDesign-package]{DiceDesign}}. }
  \item{seed}{ seed for random number generation; latin hypercube samples from package 
        \pkg{lhs} are random samples. Specifying a seed makes the result reproducible. }
  \item{factor.names}{ list of scale end values for each factor; 
        names are used as variable names; if the list is not named, the variable names are X1, X2 and so forth;
        the original unit cube calculated by package \pkg{lhs} (scale ends 0 and 1 for each variable) 
        is rescaled to the values given in factor.names }
  \item{digits}{ digits to which the design columns are rounded; one single value 
        (the same for all factors) or a vector of length \code{nfactors};\cr
        note that the rounding is applied after generation of the design on the actual 
        data scale, i.e. the unit cube generated by the functions from packages \pkg{lhs} 
        or \pkg{DiceDesign} is NOT rounded}
  \item{nlevels}{used for type \code{fact} only; integer number or numeric vector of \code{nfactor} 
        integers; specifies the number of levels for each factor. If all factors have the same number 
        of levels, the number of levels can also be specified through \code{nruns}, which is 
        interpreted as the number of levels for type \code{fact}, if \code{nlevels} is 
        not separately specified}
  \item{randomize}{logical that prevents randomization per default. The option has an effect 
        for types \code{fact} and \code{faure} only. All other types are based on random 
        design generation anyway. 
        Note that preventing randomization is the default here, because these designs 
        are assumed to be used mostly for computer experimentation, where the systematics 
        of the non-randomized design may be beneficial. For hardware experimentation, 
        randomization should be set to \code{TRUE}!}
  \item{lhs}{design generated by function \code{lhs.design} (class \code{design}, 
        of type \code{lhs}}
  \item{m}{integer number of additional points to add to design \code{lhs} (note, however, 
        that \code{optSeeded} does not necessarily preserve all original runs!)}
  \item{\dots}{ additional arguments to the functions from packages \pkg{lhs} 
        or \pkg{DiceDesign}. 
        Refer to their documentation. \cr
        Functions for generating lhs designs: 
        \code{\link[lhs]{randomLHS}}, \code{\link[lhs]{geneticLHS}}, 
        \code{\link[lhs]{improvedLHS}}, \code{\link[lhs]{maximinLHS}}, 
        \code{\link[lhs]{optimumLHS}}, \code{\link[DiceDesign]{dmaxDesign}}, 
        \code{\link[DiceDesign]{straussDesign}}, \code{\link[DiceDesign]{runif.faure}}, 
        \code{\link[DiceDesign]{factDesign}}; \cr
        functions for augmenting lhs designs: 
        \code{\link[lhs]{augmentLHS}}, 
        \code{\link[lhs]{optSeededLHS}}, 
        \code{\link[lhs]{optAugmentLHS}})
        }
}
\details{
   Function \code{lhs.design} creates a latin hypercube sample, 
   function \code{lhs.augment} augments an existing latin hypercube sample (or in case of 
   type \code{optSeeded} takes the existing sample as the starting point but potentially modifies it). 
   In comparison to direct usage of package \pkg{lhs}, the functions add the possibility 
   of recoding lhs samples to a desired range, and they embed the lhs designs into 
   class \code{\link[DoE.base:class-design]{design}}.\cr
   Range coding is based on the recoding facility from package \pkg{rsm} and the 
   \code{factor.names} parameter used analogously to packages \pkg{DoE.base} and \pkg{FrF2}.
   
   The lhs designs are useful for quantitative factors, if it is considered desirable to uniformly 
   distribute design points over a hyperrectangular space. This is e.g. considered interesting 
   for computer experiments, where replications of the same settings are often useless.
   
   Supported design types are described in the documentation for 
     packages \code{\link[lhs:lhs-package]{lhs}} 
     and \code{\link[DiceDesign:DiceDesign-package]{DiceDesign}}.
   }
\value{
    Both functions return a data frame of S3 class \code{\link[DoE.base:class-design]{design}} 
    with attributes attached. 
    The data frame contains the experimental settings as recoded to the scale ends defined in factor.names (if given), 
    rounded to the number of digits given in \code{digits} (if given).
    The experimental factors in the matrix \code{desnum} attached as attribute \code{desnum} contain the 
    design in the unit cube (all experimental factors ranging from 0 to 1) as 
    returned by packages \pkg{lhs} or \pkg{DiceDesign}. \cr
    Function \code{lhs.augment} preserves additional variables (e.g. responses) that 
    have been added to the design \code{lhs} before augmenting. Note, however, that 
    the response data are NOT used in deciding about which points to augment the design with. 
    
    The attribute \code{run.order} is not very useful for most of these designs, as there is no standard order. 
    It therefore is present for formal reasons only and contains three identical columns of 1,2,...,nruns. 
    For designs created with \code{type=fact} or \code{type=faure}, the standard order is the order 
    in which package \pkg{DiceDesign} creates the design, and the actual run order may be different in case of 
    randomization.\cr
    In case of \code{lhs.augment}, if the design to be augmented had been reordered before, 
    the augmented design preserves this reorder and also the respective numbering of the design.

    The attribute \code{design.info} is a list of various design properties, with type resolving to \dQuote{lhs}. 
    In addition to the standard list elements (cf. \code{\link[DoE.base:class-design]{design}}), the \code{subtype} 
    element indicates the type of latin hypercube designs and possibly additional augmentations, the element 
    \code{quantitative} is a vector of \code{nfactor} logical TRUEs, 
    and the \code{digits} elements indicates the digits to which the data were rounded. \cr
    For designs created with package \pkg{DiceDesign}, special list elements from this package 
    are also added to \code{design.info}.\cr
    \code{randomize} is always TRUE for designs generated by random sampling, but may be FALSE for 
    designs created with \code{type=fact} or \code{type=faure}. \cr
    \code{replications} is always 1 and \code{repeat.only} is always FALSE; 
    these elements are only present to fulfill the formal requirements for class \code{design}.
    }
\author{ Ulrike Groemping }
\note{ This package is currently under intensive development. Substantial changes are to be expected in the near future.}
\references{ 
  Beachkofski, B., Grandhi, R. (2002) Improved Distributed Hypercube Sampling. American Institute of Aeronautics and Astronautics Paper 1274.
  
  Currin C., Mitchell T., Morris M. and Ylvisaker D. (1991) 
  Bayesian Prediction of Deterministic Functions With Applications to the Design and Analysis of Computer Experiments, 
  \emph{Journal of the American Statistical Association} \bold{86}, 953--963.

  Santner T.J., Williams B.J. and Notz W.I. (2003) The Design and Analysis of Computer Experiments, Springer, 121--161.
  
  Shewry, M. C. and Wynn and H. P. (1987) Maximum entropy sampling. \emph{Journal of Applied Statistics} \bold{14}, 165--170.
  
  Fang K.-T., Li R. and Sudjianto A. (2006) \emph{Design and Modeling for Computer Experiments}, Chapman & Hall.

  Stein, M. (1987) Large Sample Properties of Simulations Using Latin Hypercube Sampling. \emph{Technometrics} \bold{29}, 143--151. 
  
  Stocki, R. (2005) A method to improve design reliability using optimal Latin hypercube sampling. 
      \emph{Computer Assisted Mechanics and Engineering Sciences} \bold{12}, 87--105.
  
  
}
\examples{
   ## maximin design from package lhs
   plan <- lhs.design(20,7,"maximin",digits=2) 
   plan
   plot(plan)
   cor(plan)
   y <- rnorm(20)
   r.plan <- add.response(plan, y)
   
   ## augmenting the design with 10 additional points, default method
   plan2 <- lhs.augment(plan, m=10)
   plot(plan2)
   cor(plan2)
   
   ## purely random design (usually not ideal)
   plan3 <- lhs.design(20,4,"random",
          factor.names=list(c(15,25), c(10,90), c(0,120), c(12,24)), digits=2)
   plot(plan3)
   cor(plan3)
   
   ## optimum design from package lhs (default)
   plan4 <- lhs.design(20,4,"optimum",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   plot(plan4)
   cor(plan4)
   
   ## dmax design from package DiceDesign
   ## arguments range and niter_max are required
   ## ?dmaxDesign for more info
   plan5 <- lhs.design(20,4,"dmax",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2,
              range=0.2, niter_max=500)
   plot(plan5)
   cor(plan5)
   
   ## Strauss design from package DiceDesign
   ## argument RND is required
   ## ?straussDesign for more info
   plan6 <- lhs.design(20,4,"strauss",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2,
              RND = 0.2)
   plot(plan6)
   cor(plan6)
   
   ## full factorial design from package DiceDesign
   ## mini try-out version
   plan7 <- lhs.design(3,4,"fact",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   plot(plan7)
   cor(plan7)
   
   \dontrun{
   
   ## full factorial design from package DiceDesign
   ## not as many different levels as runs, but only a fixed set of levels
   ##    caution: too many levels can easily bring down the computer
   ##    above design with 7 distinct levels for each factor, 
   ##    implying 2401 runs 
   plan7 <- lhs.design(7,4,"fact",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   plot(plan7)
   cor(plan7)
   
   ## equivalent call
   plan7 <- lhs.design(,4,"fact",nlevels=7,
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   
   ## different number of levels for each factor
   plan8 <- lhs.design(,4,"fact",nlevels=c(5,6,5,7),
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   plot(plan8)
   cor(plan8)

   ## equivalent call (specifying nruns, not necessary but a good check)
   plan8 <- lhs.design(1050,4,"fact",nlevels=c(5,6,5,7),
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   }
   }
\seealso{ See Also \code{\link{compare}} for comparing optimality criteria for various 
       designs, \cr
    \code{\link[lhs]{lhs-package}} and 
    \code{\link[DiceDesign]{DiceDesign-package}} for the packages that do the calculations,\cr 
    \code{\link[FrF2]{FrF2}}, \code{\link[DoE.base]{oa.design}}, 
    \code{\link[DoE.base]{fac.design}}, \code{\link[FrF2]{pb}} for other possibilites of generating designs}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ array }
\keyword{ design }% __ONLY ONE__ keyword per line
