% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network.predictability.R
\name{network.predictability}
\alias{network.predictability}
\title{Predict New Data based on Network}
\usage{
network.predictability(network, original.data, newdata, ordinal.categories = 7)
}
\arguments{
\item{network}{Matrix or data frame.
A partial correlation network}

\item{original.data}{Matrix or data frame.
Must consist only of variables to be used to estimate the \code{network}.
See \strong{Examples}}

\item{newdata}{Matrix or data frame.
Must consist of the same variables in the same order as \code{original.data}.
See \strong{Examples}}

\item{ordinal.categories}{Numeric (length = 1).
\emph{Up to} the number of categories \emph{before} a variable is considered continuous.
Defaults to \code{7} categories before \code{8} is considered continuous}
}
\value{
Returns a list containing:

\item{predictions}{Predicted values of \code{newdata} based on the \code{network}}

\item{betas}{Beta coefficients derived from the \code{network}}

\item{results}{Performance metrics for each variable in \code{newdata}}
}
\description{
General function to compute a network's predictive power on new data,
following Haslbeck and Waldorp (2018) and Williams and Rodriguez (2022)

This implementation is different from the \code{predictability} in the \code{mgm} package
(Haslbeck), which is based on (regularized) regression. This implementation uses
the network directly, converting the partial correlations into an implied
precision (inverse covariance) matrix. See \strong{Details} for more information
}
\details{
This implementation of network predictability proceeds in several steps
with important assumptions:

1. Network was estimated using (partial) correlations (not regression like the
\code{mgm} package!)

2. Original data that was used to estimate the network in 1. is necessary to
apply the original scaling to the new data

3. (Linear) regression-like coefficients are obtained by reserve engineering the
inverse covariance matrix using the network's partial correlations (i.e.,
by setting the diagonal of the network to -1 and computing the inverse
of the opposite signed partial correlation matrix; see \code{EGAnet:::pcor2inv})

4. Predicted values are obtained by matrix multiplying the new data with these
coefficients

5. \strong{Dichotomous and polytomous} data are given categorical values based
on the \strong{original data's} thresholds and these thresholds are used to
convert the continuous predicted values into their corresponding categorical values

6. Evaluation metrics:

\itemize{

\item dichotomous --- \code{"Accuracy"} or the percent correctly predicted for the 0s and 1s
and \code{"Kappa"} or Cohen's Kappa (see cite)

\item polytomous --- \code{"Linear Kappa"} or linearly weighted Kappa and
\code{"Krippendorff's alpha"} (see cite)

\item continuous --- R-squared (\code{"R2"}) and root mean square error (\code{"RMSE"})

}
}
\examples{
# Load data
wmt <- wmt2[,7:24]

# Set seed (to reproduce results)
set.seed(42)

# Split data
training <- sample(
  1:nrow(wmt), round(nrow(wmt) * 0.80) # 80/20 split
)

# Set splits
wmt_train <- wmt[training,]
wmt_test <- wmt[-training,]

# EBICglasso (default for EGA functions)
glasso_network <- network.estimation(
  data = wmt_train, model = "glasso"
)

# Check predictability
network.predictability(
  network = glasso_network, original.data = wmt_train,
  newdata = wmt_test
)

}
\references{
\strong{Original Implementation of Node Predictability} \cr
Haslbeck, J. M., & Waldorp, L. J. (2018).
How well do network models predict observations? On the importance of predictability in network models.
\emph{Behavior Research Methods}, \emph{50}(2), 853–861.

\strong{Derivation of Regression Coefficients Used (Formula 3)} \cr
Williams, D. R., & Rodriguez, J. E. (2022).
Why overfitting is not (usually) a problem in partial correlation networks.
\emph{Psychological Methods}, \emph{27}(5), 822–840.

\strong{Cohen's Kappa} \cr
Cohen, J. (1960). A coefficient of agreement for nominal scales.
\emph{Educational and Psychological Measurement}, \emph{20}(1), 37-46.

Cohen, J. (1968). Weighted kappa: nominal scale agreement provision for scaled disagreement or partial credit.
\emph{Psychological Bulletin}, \emph{70}(4), 213-220.

\strong{Krippendorff's alpha} \cr
Krippendorff, K. (2013).
Content analysis: An introduction to its methodology (3rd ed.).
Thousand Oaks, CA: Sage.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
