% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eco.lsa.R
\name{eco.lsa}
\alias{eco.lsa}
\title{Local spatial analysis}
\usage{
eco.lsa(var, con, method = c("G*", "G", "I", "C"), zerocon = NA,
  nsim = 99, conditional = c("auto", "TRUE", "FALSE"),
  test = c("permutation", "bootstrap"), alternative = c("auto", "two.sided",
  "greater", "less"), adjust = "none", multi = c("matrix", "list"),
  pop = NULL)
}
\arguments{
\item{var}{Vector, matrix or data frame for the analysis. Multiple variables in columns.}

\item{con}{An object of class eco.weight obtained with the function \code{\link{eco.weight}},
a "listw" object, or a matrix, containing the weights for the analysis. 
If a matrix, an attribute "xy" with the projected coordinates is required.}

\item{method}{Method of analysis: "G" for Getis-Ord G, "G*" for Getis-Ord G*, 
"I" for local Moran's I or "C" for local Geary's C.}

\item{zerocon}{If zerocon = 0 the program assigns the value 0 to those individuals
with no connections; if zerocon = NA the program assigns NA. Default is NA.}

\item{nsim}{Number of Monte-Carlo simulations.}

\item{conditional}{Logical. Should be used a
conditional randomization? (Anselin 1998, Sokal and Thomson 2006). The option "auto"
sets conditional = TRUE for LISA methods and G, as suggested by Sokal (2008).}

\item{test}{If test = "bootstrap", for each individual test,
the program generates a bootstrap 
resampling and the associated confidence intervals of the null hypotesis.
If test = "permutation" (default) a permutation test is made and the P-value
is computed.}

\item{alternative}{The alternative hypothesis for "permutation" test.
 If "auto" is selected (default) the
program determines the alternative hypothesis in each individual test.
Other options are: "two.sided", "greater" and "less".}

\item{adjust}{Correction method of P-values for multiple tests, 
passed to \code{\link[stats]{p.adjust}}. Defalut is "none" (no correction).}

\item{multi}{multiple output format results. "list" for object with a list 
of individual test for each variable, or "matrix" for results as matrices
of multiples variables.}

\item{pop}{numeric factor with the population of each individual. Optional 
for multiple tests with multi = "matrix".}
}
\value{
For single test, the program returns an object of class "eco.lsa" 
with the following slots:

> OUT results - table with output results. 

--> If test =  "permutation": observed value of the statistic , null confidence 
interval and #rescaled observed value to [-1, 1] range, as in Sokal (2006)

--> If test =  "bootstrap": observed and expected value
of the statistic, alternative hypotesis, null confidence interval and 
rescaled observed value to [-1, 1] range, as in Sokal (2006)

> METHOD method (coefficent) used in the analysis

> TEST test method used (bootstrap, permutation)

> NSIM number of simulations

> PADJUST P-values adjust method for permutation tests

> COND conditional randomization (logical)

> XY input coordinates

For multiple test, if the parameter multi = "list", the program
returns a list of eco.lsa objects (one element for each variable).

For multiple test, if the parameter multi = "matrix", the program
returns an object of class "eco.multilsa" with the following slots:

> METHOD method used in the analysis

>  TEST test method used (bootstrap, permutation)

>  NSIM number of simulations

>  PADJUST P-values adjust method for permutation tests

>  COND conditional randomization (logical)

>  XY input coordinates

>  OBS observed value

>  EXP expected value

>  ALTER test alternative

>  PVAL pvalue for permutation test

>  LWR lower confidence interval bound of the null hypotesis

>  UPPR upper confidence interval bound of the null hypotesis

>  OBS.RES rescaled observed value to [-1, 1] range, as in Sokal (2006)

\strong{ACCESS TO THE SLOTS}
The content of the slots can be accessed 
with the corresponding accessors, using
the generic notation of EcoGenetics 
(<ecoslot.> + <name of the slot> + <name of the object>).
See help("EcoGenetics accessors") and the Examples
section below
}
\description{
Univariate and multivariable local spatial analysis. This program computes Getis-Ord G
and G*, and LISA's (local Moran and local Geary) statistics for the data Z,
with P-values or bootstrap confidence intervals.
}
\examples{

\dontrun{

data(eco.test)

#---------------------------------------------------------------------------#


#####################
# LOCAL MORAN'S I
#####################

DETAILED EXAMPLE

#-------------------------
# TESTING PHENOTYPIC DATA-
#-------------------------

set.seed(10)

# test for a single variable---------------------------------
#computing weights

con <- eco.weight(eco[["XY"]], method = "knearest",  k = 4, row.sd = TRUE) 
# row standardized weights = TRUE

# test for the first trait of the data frame P 
localmoran <- eco.lsa(eco[["P"]][, 1], con, method = "I", nsim = 99)     

# "rankplot" graph
eco.plotLocal(localmoran)

# test for several variables---------------------------------

# ordering the factor "pop" in increasing order and the object "eco"
# in relation to this ordered factor prior to the multivariate analysis.
# This step is important for "localplot" graphs.

eco <- eco[order(eco[["S"]][,1])]

#computing weights with the ordered object

con <- eco.weight(eco[["XY"]], method = "knearest",  k = 4, row.sd = TRUE) 
# row standardized weights = TRUE

all.traits <- eco.lsa(eco[["P"]], con, method = "I", nsim = 99)     

# Plot of the phenotypic spatial patterns

# "rasterplot" graph 
eco.plotLocal(all.traits)

# in grey: non significant results (P > 0.05)
# set significant = FALSE for showing significant and no significant results
eco.plotLocal(all.traits, significant = FALSE)

# single plots using "rankplot" graphs
all.single.traits <- eco.lsa(eco[["P"]],con, method = "I", nsim = 99, multi="list")
eco.plotLocal(all.single.traits)

# removing legends for a better visualization
eco.plotLocal(all.single.traits, legend = FALSE)
# - individual plots support ggplot2 sintax (plot equivalent to the previous):
eco.plotLocal(all.single.traits) + ggplot2::theme(legend.position="none")


#-------------------------
# TESTING GENOTYPIC DATA-
#-------------------------

# eco[["A"]] is a matrix with the genetic data of "eco"
# as frequencies for each allele in each individual.

head(eco[["A"]])      # head of the matrix - 40 alleles

# ordering the factor "pop" in increasing order and the object "eco"
# in relation to this ordered factor prior to the multivariate analysis.
# This step is important for "localplot" graphs

data(eco.test) # for security this resets the data (unordered)

eco <- eco[order(eco[["S"]][,1])] # ordering

# computing weights with the ordered object

con <- eco.weight(eco[["XY"]], method = "knearest",  k = 4, row.sd = TRUE) 
# row standardized weights = TRUE

# test for a single allele
localmoran.geno <-  eco.lsa(eco[["A"]][, 32], con, method = "I", nsim = 99)

# test for several alleles -  40 alleles (it runs in less than 1 min 
# for 99 simulations per allele;  999 simulations takes ~ 11 s per allele, 
# less than 8 min in total.) 
all.alleles <-  eco.lsa(eco[["A"]], con, method = "I", nsim = 99)

# plot all alleles to get an overview of the spatial patterns
eco.plotLocal(all.alleles)

# in grey: non significant results (P > 0.05)
# set significant = FALSE for showing significant and no significant results
eco.plotLocal(all.alleles, significant = FALSE)

# counting individuals with P < 0.05 for each allele (5 * 225 /100 ~  12 significant tests 
# by random)
signif <- apply(ecoslot.PVAL(all.alleles), 2, function(x) sum (x < 0.05))

# filtering alleles, loci with > 12 significant individual tests

A.local <- eco[["A"]][, signif > 12]     #filtered matrix

all.alleles.f <-  eco.lsa(eco[["A"]][, signif > 12] , con, method = "I", nsim = 99)


# Plot of the genotypic spatial patterns using "localplot" graphs

eco.plotLocal(all.alleles.f)


## using "rankplot" graphs

all.sf <- eco.lsa(A.local,  2, eco.lsa, con, method = "I", nsim = 99, multi = "list")
eco.plotLocal(all.sf, legend = FALSE) 


#####################
# GETIS-ORD'S G*
#####################

con<- eco.weight(eco[["XY"]], method = "knearest",  k = 4, self = TRUE) # self = TRUE for G*
getis.ak <- eco.lsa(eco[["P"]][, 1], con, method = "G*", nsim = 99, adjust = "none")
getis.ak

### to plot the results, the function "eco.lsa" calls "eco.rankplot"
### (see ?eco.rankplot) when test = "permutation" and "eco.forestplot" (see ?eco.forestplot)
###  when test = "bootstrap"

p <- eco.plotLocal(getis.ak)      # rankplot graph
p    #  points with colors of the color-scale:  
     #  points with P < 0.05. Yellow points : points with P > 0.05
p <- eco.plotLocal(getis.ak, significant = FALSE)  
p    # all points have a color of the color-scale 

#-----------------------
# ACCESSORS USE EXAMPLE
#-----------------------

# the slots are accessed with the generic format 
# (ecoslot. + name of the slot + name of the object). 
# See help("EcoGenetics accessors")

ecoslot.OUT(getis.ak)

## bootstrap example
getis.akb <- eco.lsa(eco[["P"]][, 1], con, method = "G*", nsim = 99, test = "bootstrap")
p <- eco.plotLocal(getis.akb)      # forestplot graph

p2 <- eco.plotLocal(getis.akb, interactivePlot = FALSE)  
p2 + ggplot2::theme_bw()   # the plot can be modified with ggplot2
                          # In this case, the background is modified  (white color)

#---------------------------------------------------------------------------#
 
#####################
# GETIS-ORD'S G
#####################

con <- eco.weight(eco[["XY"]], method = "knearest", k = 4) 
# self = FALSE for G
getis <- eco.lsa(eco[["P"]][, 1], con, method = "G", nsim = 99)
eco.plotLocal(getis)

#---------------------------------------------------------------------------#

#####################
# LOCAL GEARY'S C
#####################

con<- eco.weight(eco[["XY"]], method = "knearest",  k = 4, row.sd = TRUE) 
# row standardized weights = TRUE
localgeary <- eco.lsa(eco[["P"]][, 1], con, method = "C", nsim = 99, adjust = "none")
eco.plotLocal(localgeary)

}

}
\author{
Leandro Roser \email{leandroroser@ege.fcen.uba.ar}
}
\references{
Anselin L. 1995. Local indicators of spatial association-LISA. 
Geographical analysis. 27: 93-115.

Getis A., and J. Ord. 1992. The analysis of spatial association by
use of distance statistics. Geographical analysis, 24: 189-206. 

Ord J., and A. Getis. 1995. Local spatial autocorrelation statistics:
distributional issues and an application. Geographical analysis, 27: 286-306.

Sokal R., N. Oden and B. Thomson. 1998. Local spatial autocorrelation
in a biological model. Geographical Analysis, 30: 331-354.

Sokal R. and B. Thomson. 2006. Population structure inferred by local 
spatial autocorrelation: an example from an Amerindian tribal population. 
American journal of physical anthropology, 129: 121-131.
}

