#' @title Covariance Estimation
#'
#' @description When the variance of a derived statistic (e.g., a difference) is 
#'              required, the covariance between the two statistics must be 
#'              calculated. This function uses results generated by various
#'              functions (e.g., a \code{\link{lm.sdf}}) to find the covariance
#'              between two statistics.
#' @param varEstA a list of two \code{data.frame}s returned by a function after 
#'                the \code{returnVarEstInputs} argument was turned on.
#'                The statistic named in the \code{varA} argument must be
#'                present in each \code{data.frame}. 
#' @param varEstB a list of two \code{data.frame}s returned by a function after 
#'                the \code{returnVarEstInputs} argument was turned on.
#'                The statistic named in the \code{varA} argument must be
#'                present in each \code{data.frame}. When the same as 
#'                \code{varEstA}, the covariance is within one result.
#' @param varA a character that names the statistic in the \code{varEstA}
#'             argument for which a covariance is required.
#' @param varB a character that names the statistic in the \code{varEstB}
#'             argument for which a covariance is required.
#' @param jkSumMultiplier when the jackknife variance estimation method---or
#'                        balanced repeated replication (BRR) 
#'                        method---multiplies the final jackknife variance estimate by a value, 
#'                        set \code{jkSumMultiplier} to that value.
#'                        For an \code{edsurvey.data.frame}, or
#'                        a \code{light.edsurvey.data.frame},
#'                        the recommended value can be recovered with
#'                        \code{EdSurvey::getAttributes(}\code{myData,} \code{"jkSumMultiplier")}.
#'
#' @details
#' Note that these functions are not vectorized, so \code{varA} and 
#' \code{varB} must contain exactly one variable name.
#'
#' The method used to compute the covariance is in
#' the
#' \href{https://www.air.org/sites/default/files/EdSurvey-Statistics.pdf}{Statistics vignette}
#'  section \dQuote{Estimation of Covariances.}
#'
#' The method used to compute the degrees of freedom is in the
#' \href{https://www.air.org/sites/default/files/EdSurvey-Statistics.pdf}{Statistics vignette}
#' section \dQuote{Estimation of Degrees of Freedom.}
#'
#' @return
#' a numeric value; the jackknife covariance estimate
#'
#' @example man/examples/varEstToCov.R
#' @author Paul Bailey
#' @export
varEstToCov <- function(varEstA, varEstB=varEstA, varA, varB=varA, jkSumMultiplier) {
  if(! varA %in% unique(varEstA$JK$variable)) {
    stop(paste0("Could not find ", sQuote("varA"), " value of ", dQuote(varA) ," in varEstA$JK$variable. Existing values include ", paste(unique(varEstA$JK$variable), collapse=", "), "."))
  }
  if(! varB %in% unique(varEstB$JK$variable)) {
    stop(paste0("Could not find ", sQuote("varB"), " value of ", dQuote(varB) ," in varEstB$JK$variable. Existing values include ", paste(unique(varEstB$JK$variable), collapse=", "), "."))
  }
  if(!is.null(varEstA$PV) & !is.null(varEstB$PV)) {
    if(! varA %in% unique(varEstA$PV$variable)) {
      stop(paste0("Could not find ", sQuote("varA"), " value of ", dQuote(varA) ," in varEstA$PV$variable. Existing values include ", paste(unique(varEstA$PV$variable), collapse=", "), "."))
    }
    if(! varB %in% unique(varEstB$PV$variable)) {
      stop(paste0("Could not find ", sQuote("varB"), " value of ", dQuote(varB) ," in varEstB$PV$variable. Existing values include ", paste(unique(varEstB$PV$variable), collapse=", "), "."))
    }
  }
  mergeVar <- c("PV", "JKreplicate")
  if(any(!mergeVar %in% names(varEstA$JK))) {
    stop(paste0("Merge vars missing from varEstA$JK ", paste(mergeVar[!mergeVar %in% names(varEstA$JK)], collapse=", ")))
  }
  if(any(!mergeVar %in% names(varEstB$JK))) {
    stop(paste0("Merge vars missing from varEstB$JK ", paste(mergeVar[!mergeVar %in% names(varEstB$JK)], collapse=", ")))
  }
  JK <- merge(subset(varEstA$JK, variable==varA),
              subset(varEstB$JK, variable==varB),
              by=mergeVar, all.x=FALSE, all.y=FALSE,
              suffixes=c(".A", ".B"))
  if(nrow(JK) < 1) {
    stop("Could not find appropiriate values in JK results to calculate covariance.")
  }
  JK$cov <- jkSumMultiplier * JK$value.A * JK$value.B
  CovJK <- sum(JK$cov) / length(unique(JK$PV))
  # now get Imputation Covariance
  if(!is.null(varEstA$PV) & !is.null(varEstB$PV)) {
    mergeVar <- c("PV")
    if(sum(!mergeVar %in% names(varEstA$PV)) > 0) {
      stop(paste0("Merge vars missing from varEstA$PV ", paste(mergeVar[!mergeVar %in% names(varEstA$PV)], collapse=", ")))
    }
    if(sum(!mergeVar %in% names(varEstB$PV)) > 0) {
      stop(paste0("Merge vars missing from varEstB$PV ", paste(mergeVar[!mergeVar %in% names(varEstB$PV)], collapse=", ")))
    }
    PV <- merge(subset(varEstA$PV, variable==varA),
                subset(varEstB$PV, variable==varB),
                by=mergeVar, all.x=FALSE, all.y=FALSE,
                suffixes=c(".A", ".B"))
    if(nrow(PV) < 1) {
      stop("Could not find appropriate values in PV results to calculate covariance.")
    }
    PV$cov <- PV$value.A * PV$value.B
    M <- length(unique(PV$PV))
    if(M==1) {
      stop("Insufficient number of plausible values in PV results to calculate imputation variance.")
    }
    # see statistics vignette for source of the scaling factor
    CovImp <- (M+1)/(M*(M-1)) * sum(PV$cov)
  } else { #ends  if(!is.null(varEstA$PV) & !is.null(varEstB$PV))
    CovImp <- 0
  }
  return(CovJK + CovImp)
}
