#' Plot the sampled area
#'
#' @description With this function it is possible to plot the analyzed area. However you need to register a APIs.
#' If you are not willing to do this, you cannot plot and the function will throw a warning. See:
#' \url{https://www.rdocumentation.org/packages/ggmap/versions/3.0.0}.
#' @usage plot_distr(nearestCorner, grid, treelineDf, size = 12)
#' @param nearestCorner A list containing the longitude and the latitude (WGS 84) of the point which is used to load the map.
#' The values must be of the data type "numeric" and finite.
#' @param grid Data frame generated by the function \code{classify_above_treeline} and therefore containing: longitude, latitude
#' (WGS 84), growing season temperature, growing season length, and a boolean. Longitude and latitude must be of the data type
#' "numeric" and finite. For the boolean \code{TRUE}, \code{FALSE} and \code{NA} is allowed and
#' nothing else.
#' @param treelineDf A data frame containing line-shaped polygons. Each row containing: a identifier, a start latitude and
#' longitude, a end latitude and longitude (all WGS 84). All longitude and latitude parameters must be of the type "numeric"
#' and finite.
#' @param size Map zoom, for the "get_map" function of the "ggmap" library. One value, data type "integer", finite and in the range
#' from 3 to 21.
#' @return Nothing.
#' @author Livio Bätscher, Jurriaan M. de Vos
#' @examples \dontrun{
#' plot_distr(nearestCorner = pointsAboveTreeLine, grid = dfGrid, treelineDf = dfTreeline,
#'            size = 12)
#' }
#' @export

plot_distr <- function(nearestCorner, grid, treelineDf, size = 12) {
  #Error handling
  if (!is.list(nearestCorner)) {stop("nearestCorner must be a list")} else if (length(nearestCorner) != 2) {stop("nearestCorner must be of length 2")} else if (sum(!is.finite(unlist(nearestCorner))) != 0) {stop("nearestCorner elements must be numeric and finite")}
  if (!is.data.frame(grid)) {stop("grid must be a data frame")} else if (ncol(grid) < 3) {stop("grid needs to have at least three columns")} else if (sum(!is.finite(as.matrix(grid[,1:2]))) != 0) {stop("grid[,1:2] must contain only finite numeric elements")} else if (!is.logical(grid[,(ncol(grid))])) {stop("grid[,(ncol(grid))] must contain only logical constants")}
  if (!is.data.frame(treelineDf)) {stop("treelineDf must be a data frame")} else if (ncol(treelineDf) != 5) {stop("treelineDf needs to have five columns")} else if (sum(is.na(treelineDf[,1])) != 0) {stop("treelineDf[,1] can not contain NA")} else if (sum(!is.finite(as.matrix(treelineDf[,2:5]))) != 0) {stop("treelineDf[,2:5] must contain only finite numeric elements")}
  if (length(size) != 1) {stop("size must be of length 1")} else if (!is.finite(size) || size %% 1 != 0) {stop("size must be a finite integer")} else if (size > 21 || size < 3) {stop("size must be from 3 to 21")}

  #Change names to be sure that I can use them
  names(nearestCorner)[1] <- "lon"
  names(nearestCorner)[2] <- "lat"
  names(grid)[1] <- "longitude"
  names(grid)[2] <- "latitude"
  names(grid)[length(grid)] <- "aboveTreeline"
  names(treelineDf) <- c("id", "lat1", "lon1", "lat2", "lon2")

  #Check if the AIP is available
  if (!is.null(ggmap::showing_key()) && ggmap::showing_key() == TRUE) {

    #Load the map from Google
    map <- ggmap::get_map(location = c(lon = nearestCorner$lon, lat = nearestCorner$lat), zoom = size, maptype = "satellite")

    #Plot the the canton with my coordinates
    mapcoordinates <- ggmap::ggmap(map) + ggplot2::labs(x = "Longitude", y = "Latitude") + ggplot2::ggtitle("Classified area") +
      ggplot2::geom_point(data = grid, ggplot2::aes_(x =~ longitude, y =~ latitude, colour =~ aboveTreeline), size = 1) +
      ggplot2::geom_segment(data = treelineDf, ggplot2::aes_(x =~ lon1, y =~ lat1, xend =~ lon2, yend =~ lat2))

    plot(mapcoordinates) #Plot the map

  #Else we throw a warning
  } else {
    warning("Could not download a map. Plot is not generated. Please enable the APIs. See: https://www.rdocumentation.org/packages/ggmap/versions/3.0.0.")
  }
}
