% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/epinow.R
\name{epinow}
\alias{epinow}
\title{Real-time Rt Estimation, Forecasting and Reporting}
\usage{
epinow(
  reported_cases,
  samples = 1000,
  horizon = 7,
  generation_time,
  delays = list(),
  CrIs = c(0.2, 0.5, 0.9),
  return_output = FALSE,
  output = c("samples", "plots", "latest"),
  target_folder = NULL,
  target_date,
  forecast_args = NULL,
  logs = tempdir(),
  id = "epinow",
  verbose = FALSE,
  ...
)
}
\arguments{
\item{reported_cases}{A data frame of confirmed cases (confirm) by date (date). confirm must be integer and date must be
in date format.}

\item{samples}{Numeric, defaults to 1000. Number of samples post warmup.}

\item{horizon}{Numeric, defaults to 7. Number of days into the future to forecast.}

\item{generation_time}{A list containing the mean, standard deviation of the mean (mean_sd),
standard deviation (sd), standard deviation of the standard deviation and the maximum allowed value for the
generation time (assuming a gamma distribution).}

\item{delays}{A list of delays (i.e incubation period/reporting delay) between infection and report.
Each list entry must also be a list containing the mean, standard deviation of the mean (mean_sd),
standard deviation (sd), standard deviation of the standard deviation and the maximum allowed value for the
that delay (assuming a lognormal distribution with all parameters excepting the max allowed value
on the log scale). To use no delays set this to \code{list()}.}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{return_output}{Logical, defaults to FALSE. Should output be returned, this automatically updates to TRUE
if no directory for saving is specified.}

\item{output}{A character vector of optional output to return. Supported options are samples ("samples"),
plots ("plots"), the run time ("timing"), copying the dated folder into a latest folder (if \code{target_folder} is not null
\itemize{
\item set using "latest"), and the stan fit ("fit"). The default is to return samples and plots alongside summarised estimates
and summary statistics. This argument uses partial matching so for example passing "sam" will lead to samples
being reported.
}}

\item{target_folder}{Character string specifying where to save results (will create if not present).}

\item{target_date}{Date, defaults to maximum found in the data if not specified.}

\item{forecast_args}{A list of arguments to pass to \code{forecast_infections}. Unless at a minimum a \code{forecast_model} is passed
tin his list then \code{forecast_infections} will be bypassed.}

\item{logs}{Character path indicating the target folder in which to store log
information. Defaults to the temporary directory if not specified. Default logging
can be disabled if \code{logs} is set to NULL. If specifying a custom logging setup then
the code for \code{setup_default_logging} and the \code{setup_logging} function are a sensible
place to start.}

\item{id}{A character string used to assign logging information on error. Used by \code{regional_epinow}
to assign \code{epinow} errors to regions.}

\item{verbose}{Logical, defaults to \code{FALSE}. Should verbose debug progress messages be printed. Corresponds to the "DEBUG" level from
\code{futile.logger}. See \code{setup_logging} for more detailed logging options.}

\item{...}{Additional arguments passed to \code{estimate_infections}. See that functions documentation for options.}
}
\value{
A list of output from estimate_infections, forecast_infections,  report_cases, and report_summary.
}
\description{
This function wraps the functionality of \code{estimate_infections} and \code{forecast_infections} in order
to estimate Rt and cases by date of infection, forecast into these infections into the future. It also contains
additional functionality to convert forecasts to date of report and produce summary output useful for reporting
results and interpreting them.
}
\examples{
\donttest{
# construct example distributions
generation_time <- get_generation_time(disease = "SARS-CoV-2", source = "ganyani")
incubation_period <- get_incubation_period(disease = "SARS-CoV-2", source = "lauer")
reporting_delay <- bootstrapped_dist_fit(rlnorm(100, log(6), 1), max_value = 30)

# example case data
reported_cases <- EpiNow2::example_confirmed[1:40] 

# estimate Rt and nowcast/forecast cases by date of infection
out <- epinow(reported_cases = reported_cases, generation_time = generation_time,
              delays = list(incubation_period, reporting_delay),
              stan_args = list(cores = ifelse(interactive(), 4, 1),
              control = list(adapt_delta = 0.95)))
out

# optional forecasting using EpiSoon plug-in
if(requireNamespace("EpiSoon")){
   if(requireNamespace("forecastHybrid")){
   # report Rt along with forecasts
   out <- epinow(reported_cases = reported_cases, samples = 200,
                 generation_time = generation_time, 
                 delays = list(incubation_period, reporting_delay),
                 forecast_args = list(
                     forecast_model = function(y, ...){
                     EpiSoon::forecastHybrid_model(
                          y = y[max(1, length(y) - 21):length(y)],
                          model_params = list(models = "aefz", weights = "equal"),
                          forecast_params = list(PI.combination = "mean"), ...)}
                          ),
                 stan_args = list(warmup = 200, cores = ifelse(interactive(), 4, 1)))
    out
   }
}
}

}
