% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrapCIs.R
\name{bootstrapCI}
\alias{bootstrapCI}
\title{Function to compute bootstrap confidence intervals}
\usage{
bootstrapCI(
  object,
  which = NULL,
  resampling_fun_outer = NULL,
  resampling_fun_inner = NULL,
  B_outer = 100,
  B_inner = 25,
  type_inner = c("bootstrap", "kfold", "subsampling"),
  levels = c(0.05, 0.95),
  ...
)
}
\arguments{
\item{object}{a fitted model object of class \code{FDboost}, 
for which the confidence intervals should be computed.}

\item{which}{a subset of base-learners to take into account for 
computing confidence intervals.}

\item{resampling_fun_outer}{function for the outer resampling procedure.
\code{resampling_fun_outer} must be a function with arguments \code{object}
and \code{fun}, where \code{object} corresponds to the fitted 
\code{FDboost} object and \code{fun} is passed to the \code{fun}
argument of the resampling function (see examples).
If \code{NULL}, \code{\link{applyFolds}} is used with 100-fold boostrap.
Further arguments to \code{\link{applyFolds}} can be passed via \code{...}.
Although the function can be defined very flexible, it is recommended 
to use \code{applyFolds} and, in particular, not \code{cvrisk}, 
as in this case, weights of the inner and outer 
fold will interact, probably causing the inner 
resampling to crash. For bootstrapped confidence intervals
the outer function should usually be a bootstrap type of resampling.}

\item{resampling_fun_inner}{function for the inner resampling procudure,
which determines the optimal stopping iteration in each fold of the
outer resampling procedure. Should be a function with one argument
\code{object} for the fitted \code{FDboost} object. 
If \code{NULL}, \code{cvrisk} is used with 25-fold bootstrap.}

\item{B_outer}{Number of resampling folds in the outer loop.
Argument is overwritten, when a custom \code{resampling_fun_outer}
is supplied.}

\item{B_inner}{Number of resampling folds in the inner loop.
Argument is overwritten, when a custom \code{resampling_fun_inner}
is supplied.}

\item{type_inner}{character argument for specifying the cross-validation method for
the inner resampling level. Default is \code{"bootstrap"}. Currently  
bootstrap, k-fold cross-validation and subsampling are implemented.}

\item{levels}{the confidence levels required. If NULL, the 
raw results are returned.}

\item{...}{further arguments passed to \code{\link{applyFolds}} if
the default for \code{resampling_fun_outer} is used}
}
\value{
A list containing the elements \code{raw_results}, the 
\code{quantiles} and \code{mstops}. 
In \code{raw_results} and \code{quantiles}, each baselearner
selected with \code{which} in turn corresponds to a list
element. The quantiles are given as vector, matrix or list of
matrices depending on the nature of the effect. In case of functional
effects the list element in\code{quantiles} is a \code{length(levels)} times
\code{length(effect)} matrix, i.e. the rows correspond to the quantiles.
In case of coefficient surfaces, \code{quantiles} comprises a list of matrices,
where each list element corresponds to a quantile.
}
\description{
The model is fitted on bootstrapped samples of the data to compute bootstrapped 
coefficient estimates. To determine the optimal stopping iteration an inner bootstrap 
is run within each bootstrap fold. 
As estimation by boosting shrinks the coefficient estimates towards zero, 
to bootstrap confidence intervals are biased towards zero.
}
\note{
Note that parallelization can be achieved by defining
the \code{resampling_fun_outer} or \code{_inner} accordingly.
See, e.g., \code{\link{cvrisk}} on how to parallelize resampling
functions or the examples below. Also note that by defining
a custum inner or outer resampling function the respective
argument \code{B_inner} or \code{B_outer} is ignored.
For models with complex baselearners, e.g., created by combining
several baselearners with the Kronecker or row-wise tensor product,
it is also recommended to use \code{levels = NULL} in order to
let the function return the raw results and then manually compute
confidence intervals.
If a baselearner is not selected in any fold, the function
treats its effect as constantly zero.
}
\examples{
if(require(refund)){
#########
# model with linear functional effect, use bsignal()
# Y(t) = f(t) + \int X1(s)\beta(s,t)ds + eps
set.seed(2121)
data1 <- pffrSim(scenario = "ff", n = 40)
data1$X1 <- scale(data1$X1, scale = FALSE)
dat_list <- as.list(data1)
dat_list$t <- attr(data1, "yindex")
dat_list$s <- attr(data1, "xindex")

## model fit by FDboost 
m1 <- FDboost(Y ~ 1 + bsignal(x = X1, s = s, knots = 8, df = 3), 
              timeformula = ~ bbs(t, knots = 8), data = dat_list)

}
              
\dontrun{             
# a short toy example with to few folds  
# and up to 200 boosting iterations 
bootCIs <- bootstrapCI(m1[200], B_inner = 2, B_outer = 5) 

# look at stopping iterations
bootCIs$mstops

# plot bootstrapped coefficient estimates
plot(bootCIs, ask = FALSE)
}

## now speed things up by defining the inner resampling
## function with parallelization based on mclapply (does not work on Windows)

my_inner_fun <- function(object){ 
cvrisk(object, folds = cvLong(id = object$id, weights = 
model.weights(object), 
B = 10 # 10-fold for inner resampling
), mc.cores = 10) # use ten cores
}

\dontrun{
bootCIs <- bootstrapCI(m1, resampling_fun_inner = my_inner_fun)
}

## We can also use the ... argument to parallelize the applyFolds
## function in the outer resampling 

\dontrun{
bootCIs <- bootstrapCI(m1, mc.cores = 30)
}

## Now let's parallelize the outer resampling and use 
## crossvalidation instead of bootstrap for the inner resampling

my_inner_fun <- function(object){ 
cvrisk(object, folds = cvLong(id = object$id, weights = 
model.weights(object), type = "kfold", # use CV
B = 10, # 10-fold for inner resampling
),
mc.cores = 10) # use ten cores
}

# use applyFolds for outer function to avoid messing up weights
my_outer_fun <- function(object, fun){
applyFolds(object = object,
folds = cv(rep(1, length(unique(object$id))), 
type = "bootstrap", B = 100), fun = fun,
mc.cores = 10) # parallelize on 10 cores
}

######## Example for scalar-on-function-regression with bsignal() 
data("fuelSubset", package = "FDboost")

## center the functional covariates per observed wavelength
fuelSubset$UVVIS <- scale(fuelSubset$UVVIS, scale = FALSE)
fuelSubset$NIR <- scale(fuelSubset$NIR, scale = FALSE)

## to make mboost:::df2lambda() happy (all design matrix entries < 10)
## reduce range of argvals to [0,1] to get smaller integration weights
fuelSubset$uvvis.lambda <- with(fuelSubset, (uvvis.lambda - min(uvvis.lambda)) /
(max(uvvis.lambda) - min(uvvis.lambda) ))
fuelSubset$nir.lambda <- with(fuelSubset, (nir.lambda - min(nir.lambda)) /
(max(nir.lambda) - min(nir.lambda) ))

## model fit with scalar response and two functional linear effects 
## include no intercept as all base-learners are centered around 0    

mod2 <- FDboost(heatan ~ bsignal(UVVIS, uvvis.lambda, knots = 40, df = 4, check.ident = FALSE) 
               + bsignal(NIR, nir.lambda, knots = 40, df=4, check.ident = FALSE), 
               timeformula = NULL, data = fuelSubset) 


\dontrun{
# takes some time, because of defaults: B_outer = 100, B_inner = 25
bootCIs <- bootstrapCI(mod2)
}

## run with a larger number of outer bootstrap samples
## and only 10-fold for validation of each outer fold
## WARNING: This may take very long!
\dontrun{
bootCIs <- bootstrapCI(mod2, B_outer = 1000, B_inner = 10)
}

}
\author{
David Ruegamer, Sarah Brockhaus
}
