#' Model-Assisted module - Generate population data for MA module.
#' 
#' Generates population data for generating model-assisted estimation.  Plots
#' that are totally nonsampled are excluded from estimation dataset. Next, an
#' adjustment factor is calculated by strata (if MAmethod="PS") or by
#' estimation unit to adjust for nonsampled (nonresponse) conditions that have
#' proportion less than 1. Attributes adjusted to a per-acre value are summed
#' by plot, divided by the adjustment factor, and averaged by stratum and/or
#' estimation unit. Note: population data must be generated by MA method.
#' 
#' If variables are NULL, then it will prompt user to input variables.
#' 
#' Necessary variables:\cr 
#' \tabular{llll}{ 
#' \tab \bold{Data} \tab \bold{Variable} \tab \bold{Description}\cr 
#' \tab tree \tab tuniqueid 
#' \tab Unique identifier for each plot, to link to pltassgn (e.g. PLT_CN).\cr 
#' \tab \tab CONDID \tab Unique identifier of each condition on plot, to link to
#' cond.  Set CONDID=1, if only 1 condition per plot.\cr 
#' \tab \tab TPA_UNADJ \tab Number of trees per acre each sample tree represents 
#' (e.g. DESIGNCD=1: TPA_UNADJ=6.018046 for trees on subplot; 74.965282 for 
#' trees on microplot).\cr 
#' \tab cond \tab cuniqueid \tab Unique identifier for each plot, to link to 
#' pltassgn (e.g. PLT_CN).\cr 
#' \tab \tab CONDID \tab Unique identfier of each condition on plot. Set 
#' CONDID=1, if only 1 condition per plot.\cr 
#' \tab \tab CONDPROP_UNADJ \tab Unadjusted proportion of condition on
#' each plot.  Set CONDPROP_UNADJ=1, if only 1 condition per plot.\cr 
#' \tab \tab COND_STATUS_CD \tab Status of each forested condition on plot 
#' (i.e. accessible forest, nonforest, water, etc.)\cr 
#' \tab \tab NF_COND_STATUS_CD \tab If ACI=TRUE. Status of each nonforest 
#' condition on plot (i.e. accessible nonforest, nonsampled nonforest)\cr 
#' \tab \tab SITECLCD \tab If landarea=TIMBERLAND. Measure of site 
#' productivity.\cr 
#' \tab \tab RESERVCD \tab If landarea=TIMBERLAND. Reserved status.\cr 
#' \tab \tab SUBPROP_UNADJ \tab Unadjusted proportion of subplot conditions 
#' on each plot. Set SUBPROP_UNADJ=1, if only 1 condition per subplot.\cr 
#' \tab \tab MICRPROP_UNADJ \tab If microplot tree attributes. Unadjusted 
#' proportion of microplot conditions on each plot. Set MICRPROP_UNADJ=1, 
#' if only 1 condition per microplot.\cr 
#' \tab \tab MACRPROP_UNADJ \tab If macroplot tree attributes. Unadjusted 
#' proportion of macroplot conditions on each plot. Set MACRPROP_UNADJ=1, 
#' if only 1 condition per macroplot.\cr \tab pltassgn \tab puniqueid 
#' \tab Unique identifier for each plot, to link to cond (e.g. CN).\cr
#' \tab \tab STATECD \tab Identifies state each plot is located in.\cr 
#' \tab \tab INVYR \tab Identifies inventory year of each plot.\cr 
#' \tab \tab PLOT_STATUS_CD \tab Status of each plot (i.e. sampled, 
#' nonsampled). If not included, all plots are assumed as sampled.\cr }
#' 
#' For available reference tables: sort(unique(FIESTAutils::ref_codes$VARIABLE)) \cr
#' 
#' @param popType String. Type of evaluation(s) to include in population data.
#' Note: currently only c('CURR', 'VOL', 'LULC') are available. See details
#' below for descriptions of each.
#' @param popTabs List of population tables the user would like returned.
#'  See help(popTables) for a list of options.
#' @param popTabIDs List of unique IDs corresponding to the population tables
#' that the user has requested. See help(popTableIDs) for a list of
#' options.
#' @param popFilter List of population filters. See help(popFilters) for a 
#' list of options. 
#' @param pltassgn DF/DT, Optional. R object, sf R object, comma-delimited
#' file(.csv), layer or spatial layer in dsn, or shapefile(.shp). Plot-level
#' assignment of estimation unit and/or strata, with one record for each plot.
#' @param pltassgnid String.
#' @param dsn String. Name of database where tree, cond, and plot-level tables
#' reside.  The dsn varies by driver. See gdal OGR vector formats
#' (https://www.gdal.org/ogr_formats.html).
#' @param pjoinid String. Join variable in plot to match pltassgnid. Does not
#' need to be uniqueid. If using most current XY coordinates for plot
#' assignments, use identifier for plot (e.g., PLOT_ID).
#' @param areawt String. Name of variable for summarizing area weights (e.g.,
#' CONDPROP_UNADJ).
#' @param adjplot Logical. If TRUE, adjusts for nonresponse at plot-level.
#' @param unitvar String. Name of the estimation unit variable in unitarea and
#' cond or pltassgn data frame with estimation unit assignment for each plot
#' (e.g., 'ESTN_UNIT'). Optional if only one estimation unit.
#' @param unitarea Numeric or DF. Total area by estimation unit. If only 1
#' estimation unit, include number of total acreage for the area of interest or
#' a data frame with area and estimation unit. If more than one estimation
#' unit, provide a data frame of total area by estimation unit, including
#' unitvar and areavar.
#' @param areavar String. Name of area variable in unitarea. Default="ACRES".
#' @param unitzonal DF/DT. Table with zonal auxiliary information by estimation
#' unit. For continuous data, means by estimation unit; for categorical data,
#' proportion of class by estimation unit.
#' @param prednames String vector. Name(s) of predictor variables to include in
#' model.
#' @param predfac String vector. Name(s) of prednames that are factors (i.e.,
#' categorical). Names will change in output depending on number of categories.
#' @param standardize Logical. If TRUE, predictors are standardized. 
#' @param savedata Logical. If TRUE, saves table(s) to outfolder.
#' @param saveobj Logical. If TRUE, saves returned list object to outfolder.
#' @param objnm String. Name of *.rds object.
#' @param unit_opts List. See help(unit_options()) for a list of options.
#' @param savedata_opts List. See help(savedata_options()) for a list
#' of options. Only used when savedata = TRUE.  
#' @param MAdata List. Data output from FIESTA::MAdata().
#' @param pltdat R List object. Output data list components from
#' FIESTA::spGetPlots().
#' @param auxdat List. Auxiliary data output from FIESTA::spGetAuxiliary().
#' @param gui Logical. If gui, user is prompted for parameters.
#' @param ... For extendibility.
#' @return A list with population data for Green-Book estimates.
#' 
#' \item{condx}{ Data frame. Condition-level data including plot-level
#' assignment of estimation unit and stratum (if strata=TRUE) and adjusted
#' condition proportion. } 
#' \item{pltcondx}{ Data frame. Condition-level data, merged with plot data. } 
#' \item{cuniqueid}{ String. Unique identifier of plot in condx and pltcondx. } 
#' \item{condid}{ String. Unique identifier of condition in condx and pltcondx. } 
#' \item{treex}{ Data frame. If esttype='TREE', tree-level data, including 
#' sample adjustment factor. }
#' \item{tuniqueid}{ String. If esttype='TREE', unique identifier of plot in
#' treex. } 
#' \item{ACI.filter}{ String. If ACI=FALSE, ACI.filter="COND_STATUS_CD
#' == 1" . } 
#' \item{unitarea}{ String. Returned table of area by estimation unit. } 
#' \item{unitvar}{ String. Variable name for estimation unit. }
#' \item{expcondtab}{ String. If ACI=FALSE, ACI.filter="COND_STATUS_CD == 1". } 
#' \item{plotsampcnt}{ Data frame. Number of plots by PLOT_STATUS_CD. } 
#' \item{condsampcnt}{ Data frame. Number of conditions by COND_STATUS_CD. } 
#' \item{states}{ String. State names in dataset. } 
#' \item{invyrs}{ String. Range of inventory years in dataset. }
#' 
#' \tabular{lll}{ \tab \bold{Variable} \tab \bold{Description}\cr 
#' \tab unitvar \tab estimation unit \cr 
#' \tab n.total \tab number of plots for estimation unit \cr 
#' \tab CONDPROP_UNADJ_SUM \tab summed condition proportion by strata and 
#' estimation unit \cr 
#' \tab CONDPROP_ADJFAC \tab adjusted condition proportion by strata after 
#' nonsampled plots removed \cr 
#' \tab AREA_USED \tab total area of estimation unit \cr 
#' \tab expfac \tab strata-level expansion factor after nonsampled plots and 
#' conditions removed (AREA_USED/n.strata) \cr 
#' \tab EXPNS \tab strata-level area expansions (expfac * strwt)\cr }
#' 
#' Table(s) are also written to outfolder.
#' @note
#' 
#' ADJUSTMENT FACTOR:\cr The adjustment factor is necessary to account for
#' nonsampled conditions. It is calculated for each estimation unit by strata.
#' by summing the unadjusted proportions of the subplot, microplot, and
#' macroplot (i.e. *PROP_UNADJ) and dividing by the number of plots in the
#' strata/estimation unit).
#' 
#' An adjustment factor is determined for each tree based on the size of the
#' plot it was measured on. This is identified using TPA_UNADJ as follows:
#' 
#' \tabular{llr}{ \tab \bold{PLOT SIZE} \tab \bold{TPA_UNADJ} \cr 
#' \tab SUBPLOT \tab 6.018046 \cr 
#' \tab MICROPLOT \tab 74.965282 \cr 
#' \tab MACROPLOT \tab 0.999188 \cr }
#' 
#' If ACI=FALSE, only nonsampled forest conditions are accounted for in the
#' adjustment factor. \cr 
#' If ACI=TRUE, the nonsampled nonforest conditions are
#' removed as well and accounted for in adjustment factor.  This is if you are
#' interested in estimates for all lands or nonforest lands in the
#' All-Condition-Inventory.
#' 
#' unitcombine:\cr If TRUE and less than 2 plots in any one estimation unit,
#' all estimation units with 10 or less plots are combined. The current method
#' for combining is to group the estimation unit with less than 10 plots with
#' the estimation unit following in consecutive order (numeric or
#' alphabetical), restrained by survey unit (UNITCD) if included in dataset,
#' and continuing until the number of plots equals 10. If there are no
#' estimation units following in order, it is combined with the estimation unit
#' previous in order.
#' 
#' stratcombine:\cr If TRUE and less than 2 plots in any one strata class
#' within an esimation unit, all strata classes with 2 or less plots are
#' combined. The current method for combining is to group the strata with less
#' than 2 plots with the strata class following in consecutive order (numeric
#' or alphabetical), restrained by estimation unit (if unitcombine=FALSE), and
#' continuing until the number of plots equals 10. If there are no strata
#' classes following in order, it is combined with the estimation unit previous
#' in order.
#' @author Tracey S. Frescino, Paul L. Patterson
#' @references Scott, Charles T.; Bechtold, William A.; Reams, Gregory A.;
#' Smith, William D.; Westfall, James A.; Hansen, Mark H.; Moisen, Gretchen G.
#' 2005. Sample-based estimators used by the Forest Inventory and Analysis
#' national information management system. Gen. Tech. Rep. SRS-80. Asheville,
#' NC: U.S. Department of Agriculture, Forest Service, Southern Research
#' Station, p.53-77.
#' @keywords data
#' @examples 
#' \donttest{
#' # NOTE: FIA data objects used in these examples are stored in `FIESTA`, but
#' # can be generated for populations of interest by the user with functions in
#' # `FIESTA` such as `spGetPlots()`, `spGetAuxiliary()`, etc. For more
#' # information, see `FIESTA`'s extensive vignettes.
#'  
#' # Population data for counties in Wyoming
#' modMApop(popTabs = list(tree = FIESTA::WYtree,
#'                         cond = FIESTA::WYcond),
#'          pltassgn = FIESTA::WYpltassgn,
#'          pltassgnid = "CN",
#'          unitarea = FIESTA::WYunitarea,
#'          unitvar = "ESTN_UNIT",
#'          unitzonal = FIESTA::WYunitzonal,
#'          prednames = c("dem", "tcc", "tpi", "tnt"),
#'          predfac = "tnt")
#'          
#' # Adding seedling data as well
#' modMApop(popTabs = list(tree = FIESTA::WYtree,
#'                         cond = FIESTA::WYcond,
#'                         seed = FIESTA::WYseed),
#'          pltassgn = FIESTA::WYpltassgn,
#'          pltassgnid = "CN",
#'          unitarea = FIESTA::WYunitarea,
#'          unitvar = "ESTN_UNIT",
#'          unitzonal = FIESTA::WYunitzonal,
#'          prednames = c("dem", "tcc", "tpi", "tnt"),
#'          predfac = "tnt")
#' }
#' @export modMApop
modMApop <- function(popType="VOL",
                     popTabs = popTables(),
                     popTabIDs = popTableIDs(), 
                     popFilter = popFilters(),
                     pltassgn = NULL,
                     pltassgnid = "PLT_CN", 
                     dsn = NULL, 
                     pjoinid = "CN", 
                     areawt = "CONDPROP_UNADJ", 
                     adjplot = TRUE, 
                     unitvar = NULL, 
                     unitarea = NULL, 
                     areavar = "ACRES", 
                     unitzonal = NULL, 
                     prednames = NULL, 
                     predfac = NULL, 
                     standardize = TRUE,
                     savedata = FALSE, 
                     saveobj = FALSE, 
                     objnm = "MApopdat", 
                     unit_opts = NULL, 
                     savedata_opts = NULL, 
                     MAdata = NULL, 
                     pltdat = NULL, 
                     auxdat = NULL, 
                     gui = FALSE, 
                     ...){

  ##################################################################################
  ## DESCRIPTION:
  ## Generates population data 'on-the-fly', including strata weights, number
  ## of plots by strata and estimation unit, strata-level expansion factors,
  ## and sample-based area adjustment factors.
  ## - checks input parameters and data tables, including removing nonsampled
  ##   plots and conditions (see check.popdata for details).
  ## - checks auxiliary data (i.e., stratification data).
  ## - calculates adjustment factors for nonresponse and appends an adjustment
  ##   variable to condition and tree data.
  ##################################################################################

  ## CHECK GUI - IF NO ARGUMENTS SPECIFIED, ASSUME GUI=TRUE
  if (nargs() == 0) gui <- TRUE

  ## If gui.. set variables to NULL
  if (gui) {
    areavar=cuniqueid=ACI=tuniqueid=savedata=unitvar <- NULL
  }
  
  ## Set parameters
  adjtree <- FALSE
  popType <- "VOL"
  nonsamp.pfilter=nonsamp.cfilter <- NULL
  returnlst <- list()

  ## Set global variables
  ONEUNIT=n.total=expcondtab=bndx <- NULL
  strata <- FALSE
  adj <- ifelse(adjplot, "plot", "none")
  
  
  ##################################################################
  ## CHECK PARAMETER NAMES
  ##################################################################
  
  ## Check input parameters
  input.params <- names(as.list(match.call()))[-1]
  formallst <- names(formals(modMApop)) 
  if (!all(input.params %in% formallst)) {
    miss <- input.params[!input.params %in% formallst]
    stop("invalid parameter: ", toString(miss))
  }
 
  ## Check parameter lists
  pcheck.params(input.params, unit_opts=unit_opts, savedata_opts=savedata_opts)
  
  ## Set unit defaults
  unit_defaults_list <- formals(unit_options)[-length(formals(unit_options))]
  
  for (i in 1:length(unit_defaults_list)) {
    assign(names(unit_defaults_list)[[i]], unit_defaults_list[[i]])
  }
  
  ## Set user-supplied unit values
  if (length(unit_opts) > 0) {
    for (i in 1:length(unit_opts)) {
      if (names(unit_opts)[[i]] %in% names(unit_defaults_list)) {
        assign(names(unit_opts)[[i]], unit_opts[[i]])
      } else {
        stop(paste("Invalid parameter: ", names(unit_opts)[[i]]))
      }
    }
  }
  
  if (unit.action == "combine") {
    stop("must combine units before calculating population data")
  }

  ## Set popFilters defaults
  popFilters_defaults_list <- formals(popFilters)[-length(formals(popFilters))]
  
  for (i in 1:length(popFilters_defaults_list)) {
    assign(names(popFilters_defaults_list)[[i]], popFilters_defaults_list[[i]])
  }
  
  ## Set user-supplied popFilters values
  if (length(popFilter) > 0) {
    for (i in 1:length(popFilter)) {
      if (names(popFilter)[[i]] %in% names(popFilters_defaults_list)) {
        assign(names(popFilter)[[i]], popFilter[[i]])
      } else {
        stop(paste("Invalid parameter: ", names(popFilter)[[i]]))
      }
    }
  }

  ## Set savedata defaults
  savedata_defaults_list <- formals(savedata_options)[-length(formals(savedata_options))]
  
  for (i in 1:length(savedata_defaults_list)) {
    assign(names(savedata_defaults_list)[[i]], savedata_defaults_list[[i]])
  }
  
  ## Set user-supplied savedata values
  if (length(savedata_opts) > 0) {
    if (!savedata) {
      message("savedata=FALSE with savedata parameters... no data are saved")
    }
    for (i in 1:length(savedata_opts)) {
      if (names(savedata_opts)[[i]] %in% names(savedata_defaults_list)) {
        assign(names(savedata_opts)[[i]], savedata_opts[[i]])
      } else {
        stop(paste("Invalid parameter: ", names(savedata_opts)[[i]]))
      }
    }
  }
  
  ## Set strata defaults
  strata_defaults_list <- formals(strata_options)[-length(formals(strata_options))]

  for (i in 1:length(strata_defaults_list)) {
    assign(names(strata_defaults_list)[[i]], strata_defaults_list[[i]])
  }
    
  ## Set popTables defaults
  popTables_defaults_list <- formals(popTables)[-length(formals(popTables))]
  
  for (i in 1:length(popTables_defaults_list)) {
    assign(names(popTables_defaults_list)[[i]], popTables_defaults_list[[i]])
  }
    
  ## Set popTabIDs defaults
  popTableIDs_defaults_list <- formals(popTableIDs)[-length(formals(popTableIDs))]
  
  for (i in 1:length(popTableIDs_defaults_list)) {
    if (names(popTableIDs_defaults_list)[[i]] == "cond") {
      assign("cuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "plt") {
      assign("puniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "tree") {
      assign("tuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "seed") {
      assign("suniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "vsubpspp") {
      assign("vsppuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "vsubpstr") {
      assign("vstruniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "invsubp") {
      assign("invuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "subplot") {
      assign("subpuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "subp_cond") {
      assign("subcuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "cond_dwm_calc") {
      assign("dwmuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "grm") {
      assign("grmuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "plot_pplot") {
      assign("pplotuniqueid", popTableIDs_defaults_list[[i]])
    }
    if (names(popTableIDs_defaults_list)[[i]] == "cond_pcond") {
      assign("pconduniqueid", popTableIDs_defaults_list[[i]])
    }
  }
  

  ##################################################################
  ## CHECK PARAMETER INPUTS
  ##################################################################
  
  ## Check savedata 
  savedata <- pcheck.logical(savedata, varnm="savedata", 
		title="Save data tables?", first="YES", gui=gui, stopifnull=TRUE)

  ## Check saveobj 
  saveobj <- pcheck.logical(saveobj, varnm="saveobj", 
		title="Save SApopdat object?", first="YES", gui=gui, stopifnull=TRUE)
  

  ## Check output
  ########################################################
  if (savedata) {
    outlst <- pcheck.output(outfolder=outfolder, out_dsn=out_dsn, 
            out_fmt=out_fmt, outfn.pre=outfn.pre, outfn.date=outfn.date, 
            overwrite_dsn=overwrite_dsn, overwrite_layer=overwrite_layer,
            add_layer=add_layer, append_layer=append_layer, gui=gui)
    outfolder <- outlst$outfolder
    out_dsn <- outlst$out_dsn
    out_fmt <- outlst$out_fmt
    overwrite_layer <- outlst$overwrite_layer
    append_layer <- outlst$append_layer
    outfn.date <- outlst$outfn.date
    outfn.pre <- outlst$outfn.pre
  } 

  if (saveobj) {
    outobj_fmtlst <- c('rds', 'rda', 'llo')
    outobj_fmt <- pcheck.varchar(var2check=outobj_fmt, varnm="outobj_fmt", gui=gui,
		checklst=outobj_fmtlst, caption="outobj_fmt", multiple=FALSE, stopifnull=TRUE)

    if (is.null(objnm)) {
      objnm <- "MApopdat"
    }
    #if (append_layer) overwrite_layer <- FALSE
    if (append_layer) message("currently cannot append to object lists")
    objfn <- getoutfn(outfn=objnm, ext=outobj_fmt, outfolder=outfolder, 
		overwrite=overwrite_layer, outfn.pre=outfn.pre, outfn.date=outfn.date)
  }


  ## Check popType
  ########################################################
  #evalTyplst <- c("ALL", "CURR", "VOL", "LULC", "P2VEG", "INV", "GRM", "DWM")
  DWM_types <- c("CWD", "FWD_SM", "FWD_LG", "DUFF")
  evalTyplst <- c("ALL", "CURR", "VOL", "LULC", "P2VEG", "INV", "DWM", "CHNG", "GRM")
  popType <- pcheck.varchar(var2check=popType, varnm="popType", gui=gui,
		checklst=evalTyplst, caption="popType", multiple=FALSE, stopifnull=TRUE)
  popevalid <- as.character(evalid)
  if (!is.null(evalid)) {
    substr(popevalid, nchar(popevalid)-1, nchar(popevalid)) <- 
		FIESTA::ref_popType[FIESTA::ref_popType$popType %in% popType, "EVAL_TYP_CD"]
  } 

 
  ###################################################################################
  ## Load data
  ###################################################################################
  if (!is.null(MAdata)) {
    list.items <- c("tabs", "unitarea", "unitvar", "unitzonal")
    MAdata <- pcheck.object(MAdata, "MAdata", list.items=list.items)
    #bnd <- MAdata$bnd
    popTabs <- MAdata$tabs
    popTabIDs <- MAdata$tabIDs
    pltassgn <- MAdata$pltassgn
    pltassgnid <- MAdata$pltassgnid
    unitarea <- MAdata$unitarea
    areavar <- MAdata$areavar
    unitzonal <- MAdata$unitzonal
    puniqueid <- MAdata$puniqueid
    pjoinid <- MAdata$pjoinid

    if (is.null(unitvar)) {
      unitvar <- MAdata$unitvar
      unitvar2 <- MAdata$unitvar2
    } 
    if (is.null(npixelvar)) {
      npixelvar <- MAdata$npixelvar
    }
    if (is.null(prednames)) {
      prednames <- MAdata$prednames
    } else {
      if (!all(prednames %in% MAdata$prednames)) {
        stop("invalid prednames: ", toString(prednames[!prednames %in% MAdata$prednames]))
      }
    }
    if (is.null(predfac)) {
      predfac <- MAdata$predfac
    }
    predfac <- predfac[predfac %in% prednames]
    
  } else {
    if (!is.null(pltdat)) {
      popTabs <- pltdat$tabs
      popTabIDs <- pltdat$tabIDs
      pjoinid <- pltdat$pjoinid
    }
    if (!is.null(auxdat)) {
      list.items <- c("pltassgn", "unitzonal", "unitvar", "prednames", "unitarea")
      auxdat <- pcheck.object(auxdat, "auxdat", list.items=list.items)
      pltassgn <- auxdat$pltassgn
      pltassgnid <- auxdat$pltassgnid
      unitzonal <- auxdat$unitzonal
      unitvar <- auxdat$unitvar
      unitvar2 <- auxdat$unitvar2
      unitarea <- auxdat$unitarea
      areavar <- auxdat$areavar

      if (is.null(npixelvar)) {
        npixelvar <- auxdat$npixelvar
      }
      if (is.null(prednames)) {
        prednames <- auxdat$prednames
      } else {
        if (!all(prednames %in% auxdat$prednames))
          stop("invalid prednames: ", toString(prednames[!prednames %in% auxdat$prednames]))
      }
      if (is.null(predfac)) {
        predfac <- auxdat$predfac
      }
      predfac <- predfac[predfac %in% prednames]
    }
  } 

  ## Set user-supplied popTable values 
  popTables_defaults_list <- formals(popTables)[-length(formals(popTables))]
  if (length(popTabs) > 0) {
    for (i in 1:length(popTabs)) {
      if (names(popTabs)[[i]] %in% names(popTables_defaults_list)) {
        assign(names(popTabs)[[i]], popTabs[[i]])
      } else {
        stop(paste("Invalid parameter: ", names(popTabs)[[i]]))
      }
    }
  } else {
    stop("need to include popTabs")
  }

  list.items <- {}
  if (popType == "LULC") {
    list.items <- c(list.items, "lulcx")
  }
  if (popType == "P2VEG") {
    list.items <- c(list.items, "vsubpspp", "vsubpstr", "subplot", "subp_cond")
  }
  popTabs <- pcheck.object(popTabs, "popTabs", list.items=list.items)

 
  ## Set user-supplied popTabIDs values
  ### Check for invalid parameters first
  popTableIDs_defaults_list <- formals(popTableIDs)[-length(formals(popTableIDs))]
  for (i in 1:length(popTabIDs)) {
    if (!(names(popTabIDs)[[i]] %in% names(popTableIDs_defaults_list))) {
      stop(paste("Invalid parameter: ", names(popTabIDs)[[i]]))
    }
  }
  ### Then actually set the values
  for (nm in names(popTabs)) {
    if (!any(names(popTabIDs) == nm)) {
      popTabIDs[[nm]] <- popTableIDs_defaults_list[[nm]]
    }
  }

  ###################################################################################
  ## CHECK PLOT PARAMETERS AND DATA
  ## Generate table of sampled/nonsampled plots and conditions
  ## Remove nonsampled plots (if nonsamp.pfilter != "NONE")
  ## Applies plot filters
  ###################################################################################
  pltcheck <- check.popdataPLT(dsn=dsn, tabs=popTabs, tabIDs=popTabIDs, 
      pltassgn=pltassgn, pltassgnid=pltassgnid, pjoinid=pjoinid, 
      module="MA", popType=popType, popevalid=popevalid, adj=adj, ACI=ACI, 
      evalid=evalid, measCur=measCur, measEndyr=measEndyr, 
      measEndyr.filter=measEndyr.filter, invyrs=invyrs, intensity=intensity,
      nonsamp.pfilter=nonsamp.pfilter, unitarea=unitarea, areavar=areavar, 
      unitvar=unitvar, unitvar2=unitvar2, areaunits=areaunits, 
      unit.action=unit.action, prednames=prednames, predfac=predfac)
  if (is.null(pltcheck)) return(NULL)
  pltassgnx <- pltcheck$pltassgnx
  pltassgnid <- pltcheck$pltassgnid
  pfromqry <- pltcheck$pfromqry
  palias <- pltcheck$palias
  pjoinid <- pltcheck$pjoinid
  whereqry <- pltcheck$whereqry
  ACI <- pltcheck$ACI
  pltx <- pltcheck$pltx
  puniqueid <- pltcheck$puniqueid
  unitvar <- pltcheck$unitvar
  unitvar2 <- pltcheck$unitvar2
  unitarea <- pltcheck$unitarea
  areavar <- pltcheck$areavar
  areaunits <- pltcheck$areaunits
  unit.action <- pltcheck$unit.action
  prednames <- pltcheck$prednames
  predfac <- pltcheck$predfac
  P2POINTCNT <- pltcheck$P2POINTCNT 
  plotsampcnt <- pltcheck$plotsampcnt
  states <- pltcheck$states
  invyrs <- pltcheck$invyrs
  dbconn <- pltcheck$dbconn

  if (ACI) {
    nfplotsampcnt <- pltcheck$nfplotsampcnt
  }

  if (popType %in% c("ALL", "AREA", "VOL")) {
 
    ###################################################################################
    ## Check parameters and data for popType AREA/VOL
    ###################################################################################
    popcheck <- check.popdataVOL(gui=gui, 
               tabs=popTabs, tabIDs=popTabIDs, pltassgnx=pltassgnx, 
               pfromqry=pfromqry, palias=palias, pjoinid=pjoinid, whereqry=whereqry, 
               adj=adj, ACI=ACI, pltx=pltx, puniqueid=puniqueid, dsn=dsn, dbconn=dbconn,
               condid="CONDID", nonsamp.cfilter=nonsamp.cfilter)
    if (is.null(popcheck)) return(NULL)
    condx <- popcheck$condx
    pltcondx <- popcheck$pltcondx
    treef <- popcheck$treef
    seedf <- popcheck$seedf
    cuniqueid <- popcheck$cuniqueid
    condid <- popcheck$condid
    tuniqueid <- popcheck$tuniqueid
    ACI.filter <- popcheck$ACI.filter
    condsampcnt <- popcheck$condsampcnt
    areawt <- popcheck$areawt
    tpropvars <- popcheck$tpropvars
  }

  ###################################################################################
  ## Check auxiliary data
  ###################################################################################
  auxlut <- unitzonal
  makedummy <- TRUE
  auxdat <- check.auxiliary(module = "MA", 
                  pltx = pltassgnx, puniqueid = pltassgnid, 
                  unitvar = unitvar, unitvar2 = unitvar2, 
                  unitarea = unitarea, 
                  areavar = areavar, 
                  minplotnum.unit = minplotnum.unit, 
                  unit.action = unit.action, 
                  auxlut = auxlut, 
                  prednames = prednames, 
                  predfac = predfac, 
                  makedummy = makedummy, 
                  npixelvar = npixelvar, 
                  standardize = standardize,
                  auxtext = "unitlut", removetext = "unitarea", )
  pltassgnx <- setDT(auxdat$pltx)
  unitarea <- auxdat$unitarea
  unitvar <- auxdat$unitvar
  unitvars <- auxdat$unitvars
  unitlut <- auxdat$auxlut
  prednames <- auxdat$prednames
  predfac <- auxdat$predfac
  npixels <- auxdat$npixels
  unitNA <- auxdat$unitNA
  if (is.null(key(pltassgnx))) setkeyv(pltassgnx, pltassgnid)


  ## Change names based on data.frame names
  unitlutcols <- which(names(unitlut) %in% prednames)
  pltassgnxcols <- which(names(pltassgnx) %in% prednames)
  dfnames <- colnames(data.frame(unitlut[, unitlutcols, with=FALSE]))
  names(unitlut)[unitlutcols] <- dfnames
  names(pltassgnx)[pltassgnxcols] <- dfnames
  prednames <- dfnames


  ###################################################################################
  ## GET ADJUSTMENT FACTORS BY STRATA AND/OR ESTIMATION UNIT FOR NONSAMPLED CONDITIONS
  ## Calculates adjustment factors for area and trees by strata (and estimation unit)
  ##		to account for nonsampled plots and conditions.
  ## Creates an adjusted condition proportion by merging strata-level adjustment
  ##		factors to cond and dividing CONDPROP_UNADJ by adjustment factor.
  ###################################################################################
  ## Returns:
  ##  1. Summed proportions (*PROP_UNADJ_SUM) and adjustment factors (*PROP_ADJFAC)  
  ##     by strata and estunit (*PROP_UNADJ_SUM / n.strata)
  ##  2. Adjusted condition proportion (CONDPROP_ADJ) appended to condx
  ###################################################################################
  ## Merge plot strata info to condx
  if (is.null(key(condx))) setkeyv(condx, c(cuniqueid, condid))
  condx <- condx[pltassgnx[,c(pltassgnid, unitvar, prednames), with=FALSE]]

  ## If more than one unitvar, 
  ## split the concatenated unitvar variable to keep original columns
  if (!is.null(unitvar2)) {
    condx[, (unitvars) := tstrsplit(get(unitvar), "-", fixed=TRUE)]
  }

  if (adj == "none") {
    setkeyv(condx, c(cuniqueid, condid))
  } else {
    if (popType %in% c("ALL", "VOL", "CURR")) {
      adjfacdata <- getadjfactorVOL(adj=adj, 
                        condx = condx, 
                        treex = treef, 
                        seedx = seedf, 
                        cuniqueid = cuniqueid, 
                        condid = condid,
                        unitlut = unitlut, 
                        unitvars = unitvar,
                        unitarea = unitarea,
                        areavar = areavar, 
                        areawt = areawt,
                        tpropvars = tpropvars
                        )
      condx <- adjfacdata$condx
      treef <- adjfacdata$treex
      seedf <- adjfacdata$seedx
      varadjlst <- adjfacdata$varadjlst
      areawtnm <- adjfacdata$areawtnm
      stratalut <- adjfacdata$unitlut
      expcondtab <- adjfacdata$expcondtab
    }
  }

 
  ###################################################################################
  ## Return population data objects
  ###################################################################################
  estvar.area <- ifelse(adj == "none", "CONDPROP_UNADJ", "CONDPROP_ADJ")
  returnlst$popType <- popType
  if (!is.null(bndx)) {
    returnlst$bndx <- bndx
  }
  if (is.null(key(unitarea))) {
     setkeyv(unitarea, unitvar)
  }
  returnlst <- append(returnlst, list(condx=condx, pltcondx=pltcondx, 
             cuniqueid=cuniqueid, condid=condid, ACI.filter=ACI.filter,
             unitarea=unitarea, areavar=areavar, areaunits=areaunits,
             unitvar=unitvar, unitvars=unitvars, unitlut=unitlut, 
             plotsampcnt=plotsampcnt, condsampcnt=condsampcnt, 
             npixels=npixels, npixelvar=npixelvar,
             states=states, invyrs=invyrs, estvar.area=estvar.area, adj=adj))

  if (!is.null(treef)) {
    returnlst$treex <- treef
    returnlst$tuniqueid <- tuniqueid
    returnlst$adjtree <- adjtree
  }
  if (!is.null(seedf)) {
    returnlst$seedx <- seedf
  }
  returnlst$prednames <- prednames
  returnlst$predfac <- predfac


  ###################################################################################
  ## Save population data objects
  ###################################################################################

  if (saveobj) {
    objfn <- getoutfn(outfn=objnm, outfolder=outfolder, 
		      overwrite=overwrite_layer, outfn.date=outfn.date, ext="rda")
    save(returnlst, file=objfn)
    message("saving object to: ", objfn)
  } 

  if (savedata) {
    datExportData(condx, 
        savedata_opts=list(outfolder=outfolder, 
                            out_fmt=out_fmt, 
                            out_dsn=out_dsn, 
                            out_layer="condx",
                            outfn.pre=outfn.pre, 
                            outfn.date=outfn.date, 
                            overwrite_layer=overwrite_layer,
                            append_layer=append_layer,
                            add_layer=TRUE))
    datExportData(pltcondx, 
        savedata_opts=list(outfolder=outfolder, 
                            out_fmt=out_fmt, 
                            out_dsn=out_dsn, 
                            out_layer="pltcondx",
                            outfn.pre=outfn.pre, 
                            outfn.date=outfn.date, 
                            overwrite_layer=overwrite_layer,
                            append_layer=append_layer,
                            add_layer=TRUE))

    if (!is.null(treef)) {
      datExportData(treef, 
          savedata_opts=list(outfolder=outfolder, 
                              out_fmt=out_fmt, 
                              out_dsn=out_dsn, 
                              out_layer="treex",
                              outfn.pre=outfn.pre, 
                              outfn.date=outfn.date, 
                              overwrite_layer=overwrite_layer,
                              append_layer=append_layer,
                              add_layer=TRUE))
    }
    if (!is.null(seedf)) {
      datExportData(seedf, 
          savedata_opts=list(outfolder=outfolder, 
                              out_fmt=out_fmt, 
                              out_dsn=out_dsn, 
                              out_layer="seedx",
                              outfn.pre=outfn.pre, 
                              outfn.date=outfn.date, 
                              overwrite_layer=overwrite_layer,
                              append_layer=append_layer,
                              add_layer=TRUE))
    }

    datExportData(pltassgnx, 
         savedata_opts=list(outfolder=outfolder, 
                              out_fmt=out_fmt, 
                              out_dsn=out_dsn, 
                              out_layer="pltassgn",
                              outfn.pre=outfn.pre, 
                              outfn.date=outfn.date, 
                              overwrite_layer=overwrite_layer,
                              append_layer=append_layer,
                              add_layer=TRUE))
    datExportData(unitarea, 
         savedata_opts=list(outfolder=outfolder, 
                              out_fmt=out_fmt, 
                              out_dsn=out_dsn, 
                              out_layer="unitarea",
                              outfn.pre=outfn.pre, 
                              outfn.date=outfn.date, 
                              overwrite_layer=overwrite_layer,
                              append_layer=append_layer,
                              add_layer=TRUE))
  }

  return(returnlst)
}

