% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/link2tidymodels.R
\name{krr_reg}
\alias{krr_reg}
\title{Kernel Ridge Regression}
\usage{
krr_reg(
  mode = "regression",
  kernel = NULL,
  opt = NULL,
  eps = NULL,
  n_threads = NULL,
  m = NULL,
  rho = NULL,
  penalty = NULL,
  fastcv = NULL
)
}
\arguments{
\item{mode}{A single string; only `"regression"` is supported.}

\item{kernel}{Kernel matrix \eqn{K} has two kinds of Kernel ("gaussian", "laplace").}

\item{opt}{Method for constructing or approximating :
 \describe{
  \item{\code{"exact"}}{Construct the full kernel matrix
  \eqn{K \in \mathbb{R}^{n\times n}} using design matrix \eqn{X}.}
  \item{\code{"nystrom"}}{Construct a low-rank approximation of
      the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
      using the Nyström approximation.}
  \item{\code{"pivoted"}}{Construct a low-rank approximation of
      the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
      using Pivoted Cholesky decomposition.}
\item{\code{"rff"}}{Use Random Fourier Features to construct a feature map
  \eqn{Z \in \mathbb{R}^{n \times m}} (with \eqn{m} random features) so that
  \eqn{K \approx Z Z^\top}. Here, \eqn{m} is the number of features.}
 }}

\item{eps}{Tolerance parameter used only in \code{"pivoted"}
for stopping criterion of the Pivoted Cholesky decomposition.}

\item{n_threads}{Number of parallel threads. It is applied only for
\code{opt = "nystrom"} or \code{opt = "rff"}, and for the
 Laplace kernel (\code{kernel = "laplace"}).}

\item{m}{Approximation rank(number of random features) used for the low-rank kernel approximation.}

\item{rho}{Scaling parameter of the kernel(\eqn{\rho}).}

\item{penalty}{Regularization parameter.}

\item{fastcv}{If \code{TRUE}, accelerated cross-validation is
performed via sequential testing (early stopping) as implemented in the \pkg{CVST} package.}
}
\value{
A parsnip model specification of class \code{"krr_reg"}.
}
\description{
`krr_reg()` defines a Kernel Ridge Regression (KRR) model specification
for use with the tidymodels ecosystem via \pkg{parsnip}. This spec can be
paired with the \code{"fastkrr"} engine implemented in this package to fit
exact or kernel approximation (Nyström, Pivoted Cholesky, Random Fourier Features) within
\pkg{recipes}/\pkg{workflows} pipelines.
}
\examples{
\donttest{
if (all(vapply(
  c("parsnip","stats","modeldata"),
  requireNamespace, quietly = TRUE, FUN.VALUE = logical(1)
))) {
library(tidymodels)
library(parsnip)
library(stats)
library(modeldata)

# Data analysis
data(ames)
ames = ames \%>\% mutate(Sale_Price = log10(Sale_Price))

set.seed(502)
ames_split = initial_split(ames, prop = 0.80, strata = Sale_Price)
ames_train = training(ames_split) # dim (2342, 74)
ames_test  = testing(ames_split) # dim (588, 74)

# Model spec
krr_spec = krr_reg(kernel = "gaussian", opt = "exact",
                   m = 50, eps = 1e-6, n_threads = 4,
                   rho = 1, penalty = tune()) \%>\%
 set_engine("fastkrr") \%>\%
 set_mode("regression")

# Define rec
rec = recipe(Sale_Price ~ Longitude + Latitude, data = ames_train)

# workflow
wf = workflow() \%>\%
  add_recipe(rec) \%>\%
  add_model(krr_spec)

# Define hyper-parameter grid
param_grid = grid_regular(
  dials::penalty(range = c(-10, -3)),
  levels = 5
)

# CV setting
set.seed(123)
cv_folds = vfold_cv(ames_train, v = 5, strata = Sale_Price)

# Tuning
tune_results = tune_grid(
  wf,
  resamples = cv_folds,
  grid = param_grid,
  metrics = metric_set(rmse),
  control = control_grid(verbose = TRUE, save_pred = TRUE)
)

# Result check
collect_metrics(tune_results)

# Select best parameter
best_params = select_best(tune_results, metric = "rmse")

# Finalized model spec using best parameter
final_spec = finalize_model(krr_spec, best_params)
final_wf <- workflow() \%>\%
  add_recipe(rec) \%>\%
  add_model(final_spec)

# Finalized fitting using best parameter
final_fit = final_wf \%>\% fit(data = ames_train)

# Prediction
predict(final_fit, new_data = ames_test)
print(best_params)

}}

}
