% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/attrMort.R
\name{attrMort}
\alias{attrMort}
\title{Influenza- and temperature-attributable mortality for a FluMoDL object}
\usage{
attrMort(m, par = c("H1", "H3", "B", "temp", "RSV"), sel = "week",
  from = NULL, to = NULL, temprange = "cold", ci = TRUE,
  nsim = 5000, mcsamples = FALSE, progress = TRUE, blup = FALSE)
}
\arguments{
\item{m}{An object of class \code{FluMoDL}.}

\item{par}{A character vector indicating which exposures to calculate the
attributable mortality for. Defaults to \code{c("H1","H3","B","temp","RSV")}, which
indicates all three influenza proxies, temperature and RSV (if it exists in the model).}

\item{sel}{For which time period(s) to calculate attributable mortality. This can be
one of several choices. For \code{sel="week"} (the default) and \code{sel="season"} attributable
mortality is calculated for each week or each season respectively. One can also
provide to \code{sel} a list of index vectors (integer or logical) corresponding to
particular rows of \code{m$data}, or a matrix of logical index vectors, or a single
index vector. Note that the index vectors should point to \emph{consecutive rows}
of \code{m$data}.}

\item{from}{Week (integer, in YYYYWW format) or season to start from, in case
\code{sel="week"} or \code{sel="season"} respectively.}

\item{to}{Week (integer, in YYYYWW format) or season to end with, in case
\code{sel="week"} or \code{sel="season"} respectively.}

\item{temprange}{In case temperature-attributable mortality is calculated (argument
\code{par} includes "temp"), this argument specifies the temperature range or interest.
This can be one of several choices.

If \code{temprange="cold"} (the default) mortality
attributable to cold temperatures is calculated, i.e. temperatures below the MMP
(minimum mortality point). If \code{temprange="heat"} mortality attributable to hot
temperatures is calculated, i.e. those above the MMP. If \code{temperature="all"} the
entire range of temperatures is used, i.e. any temperature other than the MMP.

Alternatively one can provide a numeric vector of length two, indicating a specific
temperature range; this can also be provided as a \emph{character} vector of length two,
where one of the elements can be the word "MMP", which will be replaced with the MMP
temperature.}

\item{ci}{If \code{TRUE}, empirical (Monte Carlo) 95%% Confidence Intervals are returned
for all attributable mortality estimates.}

\item{nsim}{Number of Monte Carlo simulations to run per attributable mortality estimate.
Defaults to 5000. Increase if higher precision is required (and you don't mind the wait).}

\item{mcsamples}{If \code{TRUE}, return all Monte Carlo simulation samples in the output.
See below.}

\item{progress}{If \code{TRUE}, a progress bar appears if Monte Carlo simulations are
run and if there are more than three time periods selected in argument \code{sel}. Set to
\code{FALSE} to suppress the progress bar.}

\item{blup}{If \code{FALSE} (the default), the model coefficients stored in \code{m$model}
are used for the calculation of attributable mortality. If \code{TRUE}, the coefficients
\link[=blup.FluMoDL]{stored in the FluMoDL object} are used; if \code{blup=TRUE} but
\code{blup(m)} is \code{NULL}, a warning is generated. Alternatively, \code{blup} can
be another object of class \code{\link{summary.FluMoDL}}, whose coefficients are used for the
calculation.}
}
\value{
If \code{mcsamples=FALSE} (the default), a data.frame is returned with columns named
  'FluH1', 'FluH3', 'FluB' and 'Temp' (and/or 'RSV'), depending on the argument \code{par},
  and also 'FluH1.lo', 'FluH1.hi', 'FluH3.lo', ..., if \code{ci=TRUE}. Each row in the output
  corresponds to a selection
  made in argument \code{sel}, for example if \code{sel="week"} (the default) rows correspond to
  each week available in the data. If all influenza types/subtypes are selected in \code{par}, a
  column named 'AllFlu' is also calculated automatically, with the mortality (and 95%% CI)
  attributable to all influenza types/subtypes.

  If \code{mcsamples=TRUE}, a list is returned with elements 'result' and 'mcsamples'. The
  first contains the data.frame with point estimates of influenza- and/or temperature-attributable
  mortality, as before (no 95%% CI is provided, even if \code{ci=TRUE}). The 'mcsamples'
  element contains a list of the Monte Carlo simulation samples for each parameter in \code{par}.
}
\description{
This function uses an object of class \code{FluMoDL} to calculate mortality
attributed to influenza and/or temperature.
}
\details{
All attributable mortalities are calculated using the "backward" perspective, meaning
  the mortality at any given day that is attributable to exposures up to 30 days previously
  (=the maximum lag).

  Confidence intervals (when \code{ci=TRUE}) are obtained empirically through Monte Carlo
  simulations; this can take quite some time if lots of CIs need to be calculated (for example
  if \code{sel=TRUE}). For this reason, a progress bar is shown by default in this case
  (which can be suppressed by \code{progress=FALSE}).

  Temperature-attributable mortalities are by default calculated for cold temperatures, i.e.
  temperatures lower than the minimum mortality point (MMP). Note, however, that the adjustment
  in the FluMoDL is made for the entire range of daily mean temperatures, not just for cold.
  Therefore mortality attributable to any range of temperatures can be calculated, e.g. for
  heat, extreme cold, extreme heat, etc. See argument \code{temprange} above for details.
}
\examples{
data(greece) # Use example surveillance data from Greece
m <- with(greece, fitFluMoDL(deaths = daily$deaths,
    temp = daily$temp, dates = daily$date,
    proxyH1 = weekly$ILI * weekly$ppH1,
    proxyH3 = weekly$ILI * weekly$ppH3,
    proxyB = weekly$ILI * weekly$ppB,
    yearweek = weekly$yearweek))

\donttest{
# Calculate influenza-attributable estimates by season, until 2016-17:
attr1 <- attrMort(m, par=c("H1","H3","B"), sel="season", to=2016)
attr1

# Calculate influenza-attributable estimates by week, only point
#    estimates, for the 2014-15 season:
attr2 <- attrMort(m, par=c("H1","H3","B"), sel="week",
    from=201440, to=201520, ci=FALSE)
attr2
}

# Calculate mortality attributable to temperatures below 5 celsius, for
#    the period of January 2017:
attr3 <- attrMort(m, par="temp",
    sel=with(m$data, which(dates>="2017-1-1" & dates<="2017-1-31")),
    temprange=c(5,-20))

\donttest{
# Calculate attributable mortalities for the entire 2017-18 season, and
#    return the Monte Carlo simulation samples in the output
attr4 <- attrMort(m, sel="season", from=2017, to=2017, mcsamples=TRUE)
}

}
\references{
\itemize{
 \item Lytras T, Pantavou K, Mouratidou E, Tsiodras S. Mortality attributable to seasonal influenza
  in Greece, 2013 to 2017: variation by type/subtype and age, and a possible harvesting effect.
  \href{https://www.eurosurveillance.org/content/10.2807/1560-7917.ES.2019.24.14.1800118}{Euro Surveill.}
  2019;24(14):pii=1800118 (\href{https://www.ncbi.nlm.nih.gov/pubmed/30968823}{PubMed})

\item Gasparrini A, Leone M. Attributable risk from distributed lag models.
\href{https://bmcmedresmethodol.biomedcentral.com/articles/10.1186/1471-2288-14-55}{BMC Med Res Methodol} 2014;14:55.
}
}
