#' @export
#' @title Some Examples of a Time Circle Model with a Consumer and a Type of Bank
#' @aliases gemIntertemporal_TimeCircle_Bank_1_2
#' @description Some examples of a time circle model with a consumer and a type of bank.
#' These models can be used to solve some intertemporal savings problems.
#'
#' In these example, an np-period-lived consumer gets some payoff (or cash, exhaustible resource etc.) in each period.
#' In each period the consumer can use payoff for consumption, save payoff into bank or get a loan from the bank.
#' The interest rate is given.
#' The consumer has a CES intertemporal utility function and attempts to maximize intertemporal utility by saving and borrowing.
#' @param ... arguments to be passed to the function sdm2.
#' @examples
#' \donttest{
#' #### an example with a 5-period-lived consumer (see Zhang, 2008, section 1.3)
#' np <- 5 # the number of internal periods
#'
#' interest.rate <- 0.1
#' zeta <- (1+interest.rate)^np # the ratio of repayments to loans
#' S <- matrix(NA, np, np + 1)
#' S[1:np, np + 1] <- 100/(1:np)
#'
#' B <- matrix(0, np, np + 1)
#' B[1:np, 1:np] <- diag(np)[, c(2:np, 1)]
#' B[1, np] <- 1 / zeta
#'
#' dstl.bank <- list()
#' for (k in 1:np) {
#'   dstl.bank[[k]] <- node_new(
#'     "output",
#'     type = "Leontief",
#'     a = 1/(1+interest.rate),
#'     paste0("payoff", k)
#'   )
#' }
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CES",
#'   es = 0.5, alpha = 1,
#'   beta = prop.table(1:np),
#'   paste0("payoff", 1:np)
#' )
#'
#' ge <- sdm2(
#'   A = c(dstl.bank, dst.consumer),
#'   B = B,
#'   S0Exg = S,
#'   names.commodity = paste0("payoff", 1:np),
#'   names.agent = c(paste0("bank", 1:np), "consumer"),
#'   numeraire = "payoff1",
#'   ts = TRUE,
#'   policy = makePolicyMeanValue(30)
#' )
#'
#' ge$p
#' ge$z
#' ge$D
#' ge$S
#' ge$DV
#' ge$SV
#' growth_rate(ge$p)
#'
#' ####
#' dst.consumer$es <- 0
#'
#' ge <- sdm2(
#'   A = c(dstl.bank, dst.consumer),
#'   B = B,
#'   S0Exg = S,
#'   names.commodity = paste0("payoff", 1:np),
#'   names.agent = c(paste0("bank", 1:np), "consumer"),
#'   numeraire = "payoff1",
#'   ts = TRUE,
#'   policy = makePolicyMeanValue(30)
#' )
#'
#' ge$p
#' ge$z
#' ge$D
#' ge$S
#' ge$DV
#' ge$SV
#' growth_rate(ge$p)
#' }

gemIntertemporal_TimeCircle_Bank_1_2 <- function(...) sdm2(...)
