# SVN revision:   
# Date of last change: 2014-04-09
# Last changed by: ccampbell
# 
# Original author: ccampbell
# Copyright Activinsights, Cambridge UK 2013-2017
###############################################################################

context("create training fit")

###############################################################################

test_that("createGENEAmodel", {
    
    p1 <- read.csv(file = file.path(system.file(package = "GENEAclassify"), 
            "testdata", "trainingData2.csv"))
    
    f1 <- createGENEAmodel(data = p1, features = c("UpDown.mean", "UpDown.median", "UpDown.var", "UpDown.sd", 
        "UpDown.mad", "Degrees.mean", "Degrees.median", "Degrees.var", 
        "Degrees.sd", "Degrees.mad", "Magnitude.sum", "Magnitude.mean"), plot = FALSE, verbose = FALSE)
    
    expect_that(f1, is_a("rpart"))
    
    expect_that(f1, is_a("GENEA"))
    
    expect_that(features(f1), equals(c("Magnitude.sum", "Degrees.mad", "UpDown.mad", "UpDown.var", 
            "UpDown.sd", "UpDown.mean", "UpDown.median")))
    
    expect_that(levels(f1), equals(c("active", "sleep")))
    
    f2 <- createGENEAFit(data = p1, features = 1:5, plot = FALSE, verbose = FALSE)
    
    expect_that(features(f2), equals(c("UpDown.mad", "UpDown.var", "UpDown.sd", "UpDown.mean", "UpDown.median")))
    
    expect_that(levels(f2), equals(c("active", "sleep")))
    
    # 1.1 Normal Use for File

    p2 <- read.csv(file.path(system.file(package = "GENEAclassify"), "testdata", "trainingData9.csv"))
    
    ### use createGENEAFit to create a training fit

    # no outputs to directory, only GENEA rpart object created in session
    
    set.seed(354435)
    
    f3 <- createGENEAFit(data = p2, plot = FALSE, verbose = FALSE)
    
    t3 <- structure(list(frame = structure(list(var = structure(c(3L, 2L, 
            5L, 2L, 1L, 1L, 1L, 2L, 1L, 5L, 1L, 2L, 5L, 2L, 1L, 1L, 1L, 1L, 
            4L, 1L, 2L, 1L, 1L), .Label = c("<leaf>", "Degrees.sd", "Principal.Frequency.mad", 
            "Segment.Duration", "UpDown.sd"), class = "factor"), n = c(482L, 
            382L, 147L, 128L, 61L, 67L, 19L, 235L, 135L, 100L, 11L, 89L, 
            68L, 57L, 27L, 30L, 11L, 21L, 100L, 36L, 64L, 13L, 51L), wt = c(482, 
            382, 147, 128, 61, 67, 19, 235, 135, 100, 11, 89, 68, 57, 27, 
            30, 11, 21, 100, 36, 64, 13, 51), dev = c(367, 269, 87, 68, 17, 
            35, 7, 168, 75, 72, 2, 61, 45, 34, 13, 17, 5, 10, 44, 4, 28, 
            0, 15), yval = c(2, 2, 3, 3, 3, 2, 6, 2, 2, 8, 5, 8, 1, 1, 1, 
            8, 9, 9, 4, 4, 7, 4, 7), complexity = c(0.147138964577657, 0.0476839237057221, 
            0.0381471389645777, 0.0381471389645777, 0, 0, 0.01, 0.0476839237057221, 
            0.00953678474114441, 0.0245231607629428, 0.01, 0.0163487738419619, 
            0.0163487738419619, 0.0108991825613079, 0.00817438692098093, 
            0.00544959128065395, 0.01, 0.00817438692098093, 0.0340599455040872, 
            0, 0.0340599455040872, 0.01, 0.00817438692098093), ncompete = c(3L, 
            3L, 3L, 3L, 0L, 0L, 0L, 3L, 0L, 3L, 0L, 3L, 3L, 3L, 0L, 0L, 0L, 
            0L, 3L, 0L, 3L, 0L, 0L), nsurrogate = c(3L, 3L, 1L, 3L, 0L, 0L, 
            0L, 3L, 0L, 1L, 0L, 1L, 0L, 3L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 0L, 
            0L), yval2 = structure(c(2, 2, 3, 3, 3, 2, 6, 2, 2, 8, 5, 8, 
            1, 1, 1, 8, 9, 9, 4, 4, 7, 4, 7, 37, 37, 0, 0, 0, 0, 0, 37, 14, 
            23, 0, 23, 23, 23, 14, 9, 0, 0, 0, 0, 0, 0, 0, 115, 113, 46, 
            45, 13, 32, 1, 67, 60, 7, 0, 7, 7, 4, 0, 4, 3, 0, 2, 0, 2, 0, 
            2, 69, 69, 60, 60, 44, 16, 0, 9, 8, 1, 0, 1, 1, 1, 1, 0, 0, 0, 
            0, 0, 0, 0, 0, 57, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 
            0, 1, 0, 56, 32, 24, 13, 11, 13, 13, 0, 0, 0, 0, 0, 13, 0, 13, 
            9, 4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 32, 31, 26, 14, 4, 10, 
            12, 5, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 50, 10, 8, 
            2, 0, 2, 6, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 40, 4, 36, 0, 36, 
            55, 54, 1, 1, 0, 1, 0, 53, 25, 28, 0, 28, 18, 17, 4, 13, 1, 10, 
            1, 0, 1, 0, 1, 54, 54, 6, 6, 0, 6, 0, 48, 21, 27, 2, 25, 14, 
            8, 4, 4, 6, 11, 0, 0, 0, 0, 0, 0.0767634854771784, 0.0968586387434555, 
            0, 0, 0, 0, 0, 0.157446808510638, 0.103703703703704, 0.23, 0, 
            0.258426966292135, 0.338235294117647, 0.403508771929825, 0.518518518518518, 
            0.3, 0, 0, 0, 0, 0, 0, 0, 0.238589211618257, 0.295811518324607, 
            0.312925170068027, 0.3515625, 0.213114754098361, 0.477611940298507, 
            0.0526315789473684, 0.285106382978723, 0.444444444444444, 0.07, 
            0, 0.0786516853932584, 0.102941176470588, 0.0701754385964912, 
            0, 0.133333333333333, 0.272727272727273, 0, 0.02, 0, 0.03125, 
            0, 0.0392156862745098, 0.143153526970954, 0.180628272251309, 
            0.408163265306122, 0.46875, 0.721311475409836, 0.238805970149254, 
            0, 0.0382978723404255, 0.0592592592592593, 0.01, 0, 0.0112359550561798, 
            0.0147058823529412, 0.0175438596491228, 0.037037037037037, 0, 
            0, 0, 0, 0, 0, 0, 0, 0.118257261410788, 0.00261780104712042, 
            0, 0, 0, 0, 0, 0.00425531914893617, 0, 0.01, 0, 0.0112359550561798, 
            0.0147058823529412, 0, 0, 0, 0.0909090909090909, 0, 0.56, 0.888888888888889, 
            0.375, 1, 0.215686274509804, 0.0269709543568465, 0.0340314136125654, 
            0, 0, 0, 0, 0, 0.0553191489361702, 0, 0.13, 0.818181818181818, 
            0.0449438202247191, 0.0588235294117647, 0.0701754385964912, 0.148148148148148, 
            0, 0, 0, 0, 0, 0, 0, 0, 0.0663900414937759, 0.081151832460733, 
            0.17687074829932, 0.109375, 0.0655737704918033, 0.149253731343284, 
            0.631578947368421, 0.0212765957446809, 0.037037037037037, 0, 
            0, 0, 0, 0, 0, 0, 0, 0, 0.01, 0, 0.015625, 0, 0.0196078431372549, 
            0.103734439834025, 0.0261780104712042, 0.054421768707483, 0.015625, 
            0, 0.0298507462686567, 0.315789473684211, 0.00851063829787234, 
            0.0148148148148148, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.4, 0.111111111111111, 
            0.5625, 0, 0.705882352941177, 0.114107883817427, 0.141361256544503, 
            0.00680272108843537, 0.0078125, 0, 0.0149253731343284, 0, 0.225531914893617, 
            0.185185185185185, 0.28, 0, 0.314606741573034, 0.264705882352941, 
            0.298245614035088, 0.148148148148148, 0.433333333333333, 0.0909090909090909, 
            0.476190476190476, 0.01, 0, 0.015625, 0, 0.0196078431372549, 
            0.112033195020747, 0.141361256544503, 0.0408163265306122, 0.046875, 
            0, 0.0895522388059702, 0, 0.204255319148936, 0.155555555555556, 
            0.27, 0.181818181818182, 0.280898876404494, 0.205882352941176, 
            0.140350877192982, 0.148148148148148, 0.133333333333333, 0.545454545454545, 
            0.523809523809524, 0, 0, 0, 0, 0, 1, 0.79253112033195, 0.304979253112033, 
            0.265560165975104, 0.12655601659751, 0.139004149377593, 0.0394190871369295, 
            0.487551867219917, 0.280082987551867, 0.20746887966805, 0.0228215767634855, 
            0.184647302904564, 0.141078838174274, 0.118257261410788, 0.0560165975103734, 
            0.0622406639004149, 0.0228215767634855, 0.0435684647302905, 0.20746887966805, 
            0.0746887966804979, 0.132780082987552, 0.0269709543568465, 0.105809128630705
            ), .Dim = c(23L, 20L), .Dimnames = list(NULL, c("", "", "", "", 
            "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "nodeprob"
            )))), .Names = c("var", "n", "wt", "dev", "yval", "complexity", 
            "ncompete", "nsurrogate", "yval2"), row.names = c(1L, 2L, 4L, 
            8L, 16L, 17L, 9L, 5L, 10L, 11L, 22L, 23L, 46L, 92L, 184L, 185L, 
            93L, 47L, 3L, 6L, 7L, 14L, 15L), class = "data.frame"), where = structure(c(6L, 
            9L, 6L, 15L, 6L, 7L, 9L, 5L, 9L, 5L, 6L, 6L, 6L, 9L, 20L, 23L, 
            20L, 23L, 23L, 20L, 20L, 20L, 23L, 20L, 20L, 20L, 20L, 23L, 20L, 
            20L, 23L, 17L, 20L, 20L, 20L, 20L, 20L, 23L, 23L, 20L, 23L, 20L, 
            20L, 20L, 23L, 20L, 20L, 20L, 23L, 20L, 20L, 20L, 20L, 20L, 22L, 
            22L, 22L, 22L, 22L, 22L, 22L, 22L, 20L, 22L, 22L, 22L, 20L, 20L, 
            20L, 22L, 22L, 6L, 20L, 23L, 23L, 23L, 23L, 23L, 23L, 7L, 9L, 
            23L, 20L, 23L, 7L, 23L, 23L, 7L, 23L, 23L, 23L, 23L, 23L, 23L, 
            23L, 23L, 6L, 20L, 23L, 23L, 23L, 23L, 23L, 23L, 7L, 9L, 23L, 
            20L, 23L, 7L, 23L, 23L, 7L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
            23L, 9L, 9L, 16L, 17L, 17L, 6L, 6L, 9L, 9L, 9L, 18L, 9L, 16L, 
            17L, 9L, 17L, 15L, 9L, 9L, 9L, 11L, 9L, 6L, 18L, 15L, 17L, 9L, 
            9L, 11L, 9L, 18L, 9L, 18L, 16L, 9L, 15L, 18L, 16L, 9L, 18L, 9L, 
            17L, 9L, 18L, 18L, 6L, 18L, 9L, 18L, 18L, 15L, 11L, 11L, 11L, 
            11L, 11L, 11L, 15L, 15L, 11L, 11L, 15L, 11L, 6L, 7L, 9L, 5L, 
            6L, 7L, 5L, 7L, 7L, 6L, 23L, 6L, 7L, 6L, 7L, 5L, 7L, 7L, 6L, 
            9L, 7L, 9L, 7L, 5L, 6L, 6L, 6L, 9L, 7L, 9L, 6L, 5L, 5L, 5L, 5L, 
            5L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 9L, 
            6L, 6L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 9L, 6L, 5L, 5L, 
            5L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 6L, 9L, 6L, 5L, 5L, 5L, 6L, 6L, 
            6L, 9L, 5L, 5L, 6L, 6L, 9L, 15L, 9L, 17L, 9L, 9L, 5L, 6L, 9L, 
            6L, 9L, 9L, 5L, 6L, 5L, 9L, 6L, 9L, 17L, 23L, 17L, 5L, 9L, 9L, 
            6L, 9L, 6L, 9L, 5L, 6L, 5L, 9L, 9L, 9L, 9L, 6L, 6L, 6L, 6L, 6L, 
            6L, 6L, 6L, 9L, 9L, 6L, 9L, 7L, 16L, 16L, 9L, 6L, 6L, 5L, 5L, 
            6L, 6L, 6L, 9L, 6L, 9L, 6L, 9L, 9L, 9L, 9L, 9L, 5L, 9L, 9L, 9L, 
            9L, 9L, 9L, 6L, 9L, 9L, 9L, 9L, 6L, 9L, 6L, 9L, 9L, 5L, 9L, 5L, 
            6L, 9L, 23L, 9L, 9L, 9L, 6L, 9L, 9L, 9L, 5L, 9L, 9L, 9L, 5L, 
            9L, 6L, 16L, 16L, 9L, 9L, 9L, 9L, 6L, 9L, 9L, 6L, 9L, 9L, 6L, 
            16L, 17L, 9L, 9L, 16L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 16L, 15L, 
            9L, 15L, 9L, 9L, 16L, 16L, 9L, 16L, 15L, 16L, 9L, 9L, 16L, 16L, 
            9L, 9L, 9L, 16L, 9L, 9L, 6L, 18L, 9L, 15L, 18L, 18L, 16L, 18L, 
            18L, 18L, 16L, 9L, 23L, 9L, 18L, 18L, 16L, 18L, 18L, 9L, 15L, 
            15L, 9L, 9L, 15L, 15L, 16L, 15L, 15L, 16L, 16L, 9L, 9L, 9L, 9L, 
            9L, 15L, 9L, 16L, 9L, 16L, 9L, 16L, 9L, 9L, 15L, 9L, 16L, 15L, 
            15L, 15L, 15L, 16L, 15L, 16L, 15L, 9L), .Names = c("1", "2", 
            "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14", 
            "15", "16", "17", "18", "19", "20", "21", "22", "23", "24", "25", 
            "26", "27", "28", "29", "30", "31", "32", "33", "34", "35", "36", 
            "37", "38", "39", "40", "41", "42", "43", "44", "45", "46", "47", 
            "48", "49", "50", "51", "52", "53", "54", "55", "56", "57", "58", 
            "59", "60", "61", "62", "63", "64", "65", "66", "67", "68", "69", 
            "70", "71", "72", "73", "74", "75", "76", "77", "78", "79", "80", 
            "81", "82", "83", "84", "85", "86", "87", "88", "89", "90", "91", 
            "92", "93", "94", "95", "96", "97", "98", "99", "100", "101", 
            "102", "103", "104", "105", "106", "107", "108", "109", "110", 
            "111", "112", "113", "114", "115", "116", "117", "118", "119", 
            "120", "121", "122", "123", "124", "125", "126", "127", "128", 
            "129", "130", "131", "132", "133", "134", "135", "136", "137", 
            "138", "139", "140", "141", "142", "143", "144", "145", "146", 
            "147", "148", "149", "150", "151", "152", "153", "154", "155", 
            "156", "157", "158", "159", "160", "161", "162", "163", "164", 
            "165", "166", "167", "168", "169", "170", "171", "172", "173", 
            "174", "175", "176", "177", "178", "179", "180", "181", "182", 
            "183", "184", "185", "186", "187", "188", "189", "190", "191", 
            "192", "193", "194", "195", "196", "197", "198", "199", "200", 
            "201", "202", "203", "204", "205", "206", "207", "208", "209", 
            "210", "211", "212", "213", "214", "215", "216", "217", "218", 
            "219", "220", "221", "222", "223", "224", "225", "226", "227", 
            "228", "229", "230", "231", "232", "233", "234", "235", "236", 
            "237", "238", "239", "240", "241", "242", "243", "244", "245", 
            "246", "247", "248", "249", "250", "251", "252", "253", "254", 
            "255", "256", "257", "258", "259", "260", "261", "262", "263", 
            "264", "265", "266", "267", "268", "269", "270", "271", "272", 
            "273", "274", "275", "276", "277", "278", "279", "280", "281", 
            "282", "283", "284", "285", "286", "287", "288", "289", "290", 
            "291", "292", "293", "294", "295", "296", "297", "298", "299", 
            "300", "301", "302", "303", "304", "305", "306", "307", "308", 
            "309", "310", "311", "312", "313", "314", "315", "316", "317", 
            "318", "319", "320", "321", "322", "323", "324", "325", "326", 
            "327", "328", "329", "330", "331", "332", "333", "334", "335", 
            "336", "337", "338", "339", "340", "341", "342", "343", "344", 
            "345", "346", "347", "348", "349", "350", "351", "352", "353", 
            "354", "355", "356", "357", "358", "359", "360", "361", "362", 
            "363", "364", "365", "366", "367", "368", "369", "370", "371", 
            "372", "373", "374", "375", "376", "377", "378", "379", "380", 
            "381", "382", "383", "384", "385", "386", "387", "388", "389", 
            "390", "391", "392", "393", "394", "395", "396", "397", "398", 
            "399", "400", "401", "402", "403", "404", "405", "406", "407", 
            "408", "409", "410", "411", "412", "413", "414", "415", "416", 
            "417", "418", "419", "420", "421", "422", "423", "424", "425", 
            "426", "427", "428", "429", "430", "431", "432", "433", "434", 
            "435", "436", "437", "438", "439", "440", "441", "442", "443", 
            "444", "445", "446", "447", "448", "449", "450", "451", "452", 
            "453", "454", "455", "456", "457", "458", "459", "460", "461", 
            "462", "463", "464", "465", "466", "467", "468", "469", "470", 
            "471", "472", "473", "474", "475", "476", "477", "478", "479", 
            "480", "481", "482")), terms = Activity ~ Segment.Duration + 
                Principal.Frequency.mad + UpDown.sd + Degrees.sd, cptable = structure(c(0.147138964577657, 
            0.0476839237057221, 0.0381471389645777, 0.0340599455040872, 0.0245231607629428, 
            0.0163487738419619, 0.0108991825613079, 0.01, 0, 1, 3, 5, 7, 
            8, 10, 11, 1, 0.852861035422343, 0.757493188010899, 0.681198910081744, 
            0.613079019073569, 0.588555858310627, 0.555858310626703, 0.544959128065395, 
            1, 0.861035422343324, 0.809264305177112, 0.798365122615804, 0.724795640326975, 
            0.694822888283379, 0.697547683923706, 0.711171662125341, 0.0254971928581653, 
            0.028425492758394, 0.0290920620693169, 0.0292062049778416, 0.0297493931104612, 
            0.0298602497235428, 0.0298527688932101, 0.0298075861027944), .Dim = c(8L, 
            5L), .Dimnames = list(c("1", "2", "3", "4", "5", "6", "7", "8"
            ), c("CP", "nsplit", "rel error", "xerror", "xstd"))), method = "class", 
                parms = structure(list(prior = structure(c(0.0767634854771784, 
                0.238589211618257, 0.143153526970954, 0.118257261410788, 
                0.0269709543568465, 0.0663900414937759, 0.103734439834025, 
                0.114107883817427, 0.112033195020747), .Dim = 9L, .Dimnames = list(
                    c("1", "2", "3", "4", "5", "6", "7", "8", "9"))), loss = structure(c(0, 
                1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
                1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 
                1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 
                1, 1, 1, 0), .Dim = c(9L, 9L)), split = 1), .Names = c("prior", 
                "loss", "split")), control = structure(list(minsplit = 20L, 
                    minbucket = 7, cp = 0.01, maxcompete = 4L, maxsurrogate = 5L, 
                    usesurrogate = 2L, surrogatestyle = 0L, maxdepth = 30L, 
                    xval = 10L), .Names = c("minsplit", "minbucket", "cp", 
                "maxcompete", "maxsurrogate", "usesurrogate", "surrogatestyle", 
                "maxdepth", "xval")), functions = structure(list(summary = function (yval, 
                    dev, wt, ylevel, digits) 
                {
                    nclass <- (ncol(yval) - 2L)/2L
                    group <- yval[, 1L]
                    counts <- yval[, 1L + (1L:nclass)]
                    yprob <- yval[, 1L + nclass + 1L:nclass]
                    nodeprob <- yval[, 2L * nclass + 2L]
                    if (!is.null(ylevel)) 
                        group <- ylevel[group]
                    temp1 <- formatg(counts, format = "%5g")
                    temp2 <- formatg(yprob, format = "%5.3f")
                    if (nclass > 1) {
                        temp1 <- apply(matrix(temp1, ncol = nclass), 1L, 
                            paste, collapse = " ")
                        temp2 <- apply(matrix(temp2, ncol = nclass), 1L, 
                            paste, collapse = " ")
                    }
                    dev <- dev/(wt[1L] * nodeprob)
                    paste0("  predicted class=", format(group, justify = "left"), 
                        "  expected loss=", formatg(dev, digits), "  P(node) =", 
                        formatg(nodeprob, digits), "\n", "    class counts: ", 
                        temp1, "\n", "   probabilities: ", temp2)
                }, print = function (yval, ylevel, digits) 
                {
                    temp <- if (is.null(ylevel)) 
                        as.character(yval[, 1L])
                    else ylevel[yval[, 1L]]
                    nclass <- (ncol(yval) - 2L)/2L
                    yprob <- if (nclass < 5L) 
                        format(yval[, 1L + nclass + 1L:nclass], digits = digits, 
                            nsmall = digits)
                    else formatg(yval[, 1L + nclass + 1L:nclass], digits = 2L)
                    if (!is.matrix(yprob)) 
                        yprob <- matrix(yprob, nrow = 1L)
                    temp <- paste0(temp, " (", yprob[, 1L])
                    for (i in 2L:ncol(yprob)) temp <- paste(temp, yprob[, 
                        i], sep = " ")
                    temp <- paste0(temp, ")")
                    temp
                }, text = function (yval, dev, wt, ylevel, digits, n, use.n) 
                {
                    nclass <- (ncol(yval) - 2L)/2L
                    group <- yval[, 1L]
                    counts <- yval[, 1L + (1L:nclass)]
                    if (!is.null(ylevel)) 
                        group <- ylevel[group]
                    temp1 <- formatg(counts, digits)
                    if (nclass > 1L) 
                        temp1 <- apply(matrix(temp1, ncol = nclass), 1L, 
                            paste, collapse = "/")
                    if (use.n) 
                        paste0(format(group, justify = "left"), "\n", temp1)
                    else format(group, justify = "left")
                }), .Names = c("summary", "print", "text")), numresp = 11L, 
                splits = structure(c(482, 482, 482, 482, 0, 0, 0, 382, 382, 
                382, 382, 0, 0, 0, 147, 147, 147, 147, 0, 128, 128, 128, 
                128, 0, 0, 0, 235, 235, 235, 235, 0, 0, 0, 100, 100, 100, 
                100, 0, 89, 89, 89, 89, 0, 68, 68, 68, 68, 57, 57, 57, 57, 
                0, 0, 0, 100, 100, 100, 100, 0, 64, 64, 64, 64, 0, -1, -1, 
                1, -1, -1, 1, 1, -1, -1, 1, -1, -1, 1, -1, 1, 1, -1, -1, 
                1, -1, 1, -1, -1, -1, 1, -1, -1, -1, 1, -1, -1, -1, -1, 1, 
                -1, 1, -1, 1, -1, 1, 1, -1, -1, 1, 1, 1, -1, 1, 1, 1, -1, 
                1, 1, -1, 1, -1, -1, -1, 1, -1, -1, -1, -1, -1, 48.4995333579545, 
                42.7082362361848, 29.3533707812309, 28.9957999383241, 0.937759336099585, 
                0.898340248962656, 0.883817427385892, 24.0833129865514, 22.1769019766741, 
                6.52402918002679, 4.15295937508751, 0.863874345549738, 0.696335078534031, 
                0.617801047120419, 11.1531227622628, 7.43566673981421, 4.96996320652154, 
                3.30136054421767, 0.931972789115646, 10.1825720271593, 5.36979166666669, 
                2.94826056618821, 1.54626881270902, 0.796875, 0.6328125, 
                0.5546875, 11.4471079590229, 8.71866717800648, 3.81372418969156, 
                2.97691696084976, 0.68936170212766, 0.651063829787234, 0.582978723404255, 
                7.63536261491319, 5.55617165443253, 3.62242424242426, 2.61546019198195, 
                0.92, 4.40748434205143, 3.44593608638552, 2.64290578335522, 
                1.92376196421141, 0.808988764044944, 3.91359414579229, 2.60252100840336, 
                2.17851307189542, 1.3250773993808, 2.4214424951267, 2.20486842105262, 
                1.63655761024182, 1.20823798627001, 0.614035087719298, 0.614035087719298, 
                0.578947368421053, 10.8126388888889, 8.855, 5.73909090909091, 
                1.86947368421053, 0.65, 11.5582107843137, 1.70420321637427, 
                1.66365740740741, 1.34375, 0.84375, 13.0835, 179, 1.95, 11.642, 
                259, 1.565, 2.5665, 21.8435, 19.053, 1.149, 345, 12.5795, 
                1.149, 20.5, 1.8405, 11.3, 2.113, 145.5, 1.4865, 8.4885, 
                11.7655, 1.112, 99.5, 6.0705, 0.408, 27, 59.308, 26.6805, 
                0.8155, 169, 31.542, 87.5, 0.334, 45.573, 103.5455, 0.2595, 
                99, 472.5, 103.5455, 20.907, 0.2595, 38.5, 135.5, 20.8655, 
                0.2595, 80.5895, 39.5, 80.5895, 0.2595, 54.5, 40.179, 63.5, 
                0.2595, 38.7555, 810, 0.5415, 20.9045, 0.426, 2.075, 0.5275, 
                0.443, 20.8305, 366, 0.429, 0, 0, 0, 0, 0.7, 0.51, 0.44, 
                0, 0, 0, 0, 0.646258503401361, 0.210884353741497, 0.00680272108843537, 
                0, 0, 0, 0, 0.473684210526316, 0, 0, 0, 0, 0.573770491803279, 
                0.229508196721311, 0.0655737704918033, 0, 0, 0, 0, 0.27, 
                0.18, 0.02, 0, 0, 0, 0, 0.272727272727273, 0, 0, 0, 0, 0.19047619047619, 
                0, 0, 0, 0, 0, 0, 0, 0, 0.185185185185185, 0.185185185185185, 
                0.111111111111111, 0, 0, 0, 0, 0.0277777777777778, 0, 0, 
                0, 0, 0.230769230769231), .Dim = c(64L, 5L), .Dimnames = list(
                    c("Principal.Frequency.mad", "Segment.Duration", "UpDown.sd", 
                    "Degrees.sd", "Segment.Duration", "UpDown.sd", "Degrees.sd", 
                    "Degrees.sd", "UpDown.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "UpDown.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "UpDown.sd", "Degrees.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "Degrees.sd", "Degrees.sd", "UpDown.sd", 
                    "Principal.Frequency.mad", "Segment.Duration", "UpDown.sd", 
                    "Principal.Frequency.mad", "Segment.Duration", "Degrees.sd", 
                    "UpDown.sd", "Principal.Frequency.mad", "Segment.Duration", 
                    "UpDown.sd", "Segment.Duration", "Principal.Frequency.mad", 
                    "UpDown.sd", "Degrees.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "Segment.Duration", "Degrees.sd", 
                    "UpDown.sd", "Principal.Frequency.mad", "Segment.Duration", 
                    "Segment.Duration", "UpDown.sd", "Principal.Frequency.mad", 
                    "Degrees.sd", "Segment.Duration", "Degrees.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "UpDown.sd", "Segment.Duration", 
                    "Principal.Frequency.mad", "UpDown.sd", "Segment.Duration", 
                    "Degrees.sd", "Principal.Frequency.mad", "UpDown.sd", 
                    "Degrees.sd", "Degrees.sd", "UpDown.sd", "Principal.Frequency.mad", 
                    "Segment.Duration", "UpDown.sd"), c("count", "ncat", 
                    "improve", "index", "adj"))), variable.importance = structure(c(91.0233345025179, 
                74.8703944298319, 56.5926684330715, 51.0246411005211), .Names = c("Degrees.sd", 
                "UpDown.sd", "Principal.Frequency.mad", "Segment.Duration"
                )), y = c(9L, 9L, 9L, 5L, 6L, 6L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 5L, 5L, 5L, 5L, 5L, 
                5L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
                6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
                6L, 6L, 6L, 6L, 6L, 6L, 6L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 1L, 1L, 
                1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
                1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
                1L, 1L, 1L, 1L, 1L), ordered = structure(c(FALSE, FALSE, 
                FALSE, FALSE), .Names = c("Segment.Duration", "Principal.Frequency.mad", 
                "UpDown.sd", "Degrees.sd"))), .Names = c("frame", "where", 
            "terms", "cptable", "method", "parms", "control", "functions", 
            "numresp", "splits", "variable.importance", "y", "ordered"), xlevels = structure(list(), .Names = character(0)), ylevels = c("active", 
            "cycle", "drive", "nonwear", "run", "sit", "sleep", "swim", "walk"
            ), class = c("rpart", "GENEA"), features = c("Principal.Frequency.mad", 
            "Segment.Duration", "Degrees.sd", "UpDown.sd"))
    
    f3$call <- NULL
    
    expect_that(f3, equals(t3))
    
    # 2 Update Model
    
    set.seed(178735)
    
    f4 <- createGENEAFit(data = p2, 
        features = c("UpDown.mean", "Step.mad", "Light.mean", 
            "Magnitude.mean", "UpDown.median", 
            "Degrees.sd", "Degrees.mad", 
            "Temp.mean", "UpDown.sd"), 
        plot = FALSE, verbose = FALSE)
    
    t4 <- structure(list(frame = structure(list(var = structure(c(2L, 6L, 
            3L, 1L, 5L, 1L, 4L, 4L, 1L, 1L, 1L, 5L, 4L, 1L, 4L, 1L, 1L, 6L, 
            1L, 1L, 5L, 6L, 1L, 1L, 4L, 4L, 1L, 7L, 1L, 1L, 1L), .Label = c("<leaf>", 
            "Degrees.mad", "Degrees.sd", "Light.mean", "Magnitude.mean", 
            "Temp.mean", "UpDown.mean"), class = "factor"), n = c(482L, 321L, 
            123L, 40L, 83L, 32L, 51L, 25L, 11L, 14L, 26L, 198L, 158L, 15L, 
            143L, 127L, 16L, 40L, 10L, 30L, 161L, 67L, 58L, 9L, 94L, 44L, 
            20L, 24L, 7L, 17L, 50L), wt = c(482, 321, 123, 40, 83, 32, 51, 
            25, 11, 14, 26, 198, 158, 15, 143, 127, 16, 40, 10, 30, 161, 
            67, 58, 9, 94, 44, 20, 24, 7, 17, 50), dev = c(367, 214, 84, 
            10, 47, 5, 19, 15, 2, 4, 4, 100, 61, 2, 46, 31, 6, 10, 1, 0, 
            106, 14, 5, 2, 46, 24, 0, 10, 1, 3, 2), yval = c(2, 2, 9, 3, 
            1, 1, 9, 9, 1, 9, 9, 2, 2, 3, 2, 2, 9, 8, 5, 8, 4, 4, 4, 6, 7, 
            3, 3, 6, 2, 6, 7), complexity = c(0.128065395095368, 0.0817438692098093, 
            0.0735694822888283, 0.00272479564032698, 0.0626702997275204, 
            0.00272479564032698, 0.0122615803814714, 0.0122615803814714, 
            0.01, 0.01, 0.01, 0.0790190735694823, 0.0354223433242507, 0.01, 
            0.0245231607629428, 0.00817438692098093, 0.01, 0.0245231607629428, 
            0.01, 0.01, 0.125340599455041, 0.0190735694822888, 0, 0.01, 0.0544959128065395, 
            0.0381471389645777, 0.01, 0.0163487738419619, 0.01, 0.01, 0.01
            ), ncompete = c(4L, 4L, 4L, 0L, 4L, 0L, 4L, 4L, 0L, 0L, 0L, 4L, 
            4L, 0L, 4L, 0L, 0L, 4L, 0L, 0L, 4L, 4L, 0L, 0L, 4L, 4L, 0L, 4L, 
            0L, 0L, 0L), nsurrogate = c(5L, 5L, 5L, 0L, 5L, 0L, 5L, 5L, 0L, 
            0L, 0L, 5L, 0L, 0L, 2L, 0L, 0L, 4L, 0L, 0L, 5L, 1L, 0L, 0L, 5L, 
            5L, 0L, 5L, 0L, 0L, 0L), yval2 = structure(c(2, 2, 9, 3, 1, 1, 
            9, 9, 1, 9, 9, 2, 2, 3, 2, 2, 9, 8, 5, 8, 4, 4, 4, 6, 7, 3, 3, 
            6, 2, 6, 7, 37, 37, 36, 0, 36, 27, 9, 9, 9, 0, 0, 1, 1, 0, 1, 
            1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 115, 107, 9, 
            4, 5, 0, 5, 1, 1, 0, 4, 98, 97, 0, 97, 96, 1, 1, 1, 0, 8, 0, 
            0, 0, 8, 8, 0, 8, 6, 2, 0, 69, 47, 34, 30, 4, 0, 4, 4, 0, 4, 
            0, 13, 13, 13, 0, 0, 0, 0, 0, 0, 22, 2, 2, 0, 20, 20, 20, 0, 
            0, 0, 0, 57, 2, 1, 0, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 
            0, 0, 0, 55, 53, 53, 0, 2, 2, 0, 2, 1, 1, 0, 13, 13, 4, 0, 4, 
            4, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 
            0, 0, 0, 0, 0, 32, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 7, 1, 6, 
            2, 4, 0, 0, 0, 25, 9, 2, 7, 16, 14, 0, 14, 0, 14, 2, 50, 0, 0, 
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 2, 0, 
            2, 48, 0, 0, 0, 0, 0, 48, 55, 54, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
            54, 24, 0, 24, 23, 1, 30, 0, 30, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
            0, 54, 54, 39, 6, 33, 1, 32, 10, 0, 10, 22, 15, 15, 1, 14, 4, 
            10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0767634854771784, 
            0.115264797507788, 0.292682926829268, 0, 0.433734939759036, 0.84375, 
            0.176470588235294, 0.36, 0.818181818181818, 0, 0, 0.00505050505050505, 
            0.00632911392405063, 0, 0.00699300699300699, 0.0078740157480315, 
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.238589211618257, 
            0.333333333333333, 0.0731707317073171, 0.1, 0.0602409638554217, 
            0, 0.0980392156862745, 0.04, 0.0909090909090909, 0, 0.153846153846154, 
            0.494949494949495, 0.613924050632911, 0, 0.678321678321678, 0.755905511811024, 
            0.0625, 0.025, 0.1, 0, 0.0496894409937888, 0, 0, 0, 0.0851063829787234, 
            0.181818181818182, 0, 0.333333333333333, 0.857142857142857, 0.117647058823529, 
            0, 0.143153526970954, 0.146417445482866, 0.276422764227642, 0.75, 
            0.0481927710843374, 0, 0.0784313725490196, 0.16, 0, 0.285714285714286, 
            0, 0.0656565656565657, 0.0822784810126582, 0.866666666666667, 
            0, 0, 0, 0, 0, 0, 0.136645962732919, 0.0298507462686567, 0.0344827586206897, 
            0, 0.212765957446809, 0.454545454545455, 1, 0, 0, 0, 0, 0.118257261410788, 
            0.00623052959501558, 0.00813008130081301, 0, 0.0120481927710843, 
            0, 0.0196078431372549, 0.04, 0.0909090909090909, 0, 0, 0.00505050505050505, 
            0.00632911392405063, 0, 0.00699300699300699, 0.0078740157480315, 
            0, 0, 0, 0, 0.341614906832298, 0.791044776119403, 0.913793103448276, 
            0, 0.0212765957446809, 0.0454545454545455, 0, 0.0833333333333333, 
            0.142857142857143, 0.0588235294117647, 0, 0.0269709543568465, 
            0.0404984423676013, 0.032520325203252, 0, 0.0481927710843374, 
            0.125, 0, 0, 0, 0, 0, 0.0454545454545455, 0, 0, 0, 0, 0, 0.225, 
            0.9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0663900414937759, 
            0.0218068535825545, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0353535353535354, 
            0.0443037974683544, 0.0666666666666667, 0.041958041958042, 0.015748031496063, 
            0.25, 0, 0, 0, 0.15527950310559, 0.134328358208955, 0.0344827586206897, 
            0.777777777777778, 0.170212765957447, 0.318181818181818, 0, 0.583333333333333, 
            0, 0.823529411764706, 0.04, 0.103734439834025, 0, 0, 0, 0, 0, 
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.31055900621118, 0.0298507462686567, 
            0, 0.222222222222222, 0.51063829787234, 0, 0, 0, 0, 0, 0.96, 
            0.114107883817427, 0.168224299065421, 0, 0, 0, 0, 0, 0, 0, 0, 
            0, 0.272727272727273, 0.151898734177215, 0, 0.167832167832168, 
            0.181102362204724, 0.0625, 0.75, 0, 1, 0.0062111801242236, 0.0149253731343284, 
            0.0172413793103448, 0, 0, 0, 0, 0, 0, 0, 0, 0.112033195020747, 
            0.168224299065421, 0.317073170731707, 0.15, 0.397590361445783, 
            0.03125, 0.627450980392157, 0.4, 0, 0.714285714285714, 0.846153846153846, 
            0.0757575757575758, 0.0949367088607595, 0.0666666666666667, 0.0979020979020979, 
            0.031496062992126, 0.625, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
            0, 0, 1, 0.66597510373444, 0.255186721991701, 0.0829875518672199, 
            0.172199170124481, 0.0663900414937759, 0.105809128630705, 0.0518672199170125, 
            0.0228215767634855, 0.029045643153527, 0.0539419087136929, 0.410788381742739, 
            0.327800829875519, 0.0311203319502075, 0.296680497925311, 0.263485477178423, 
            0.033195020746888, 0.0829875518672199, 0.020746887966805, 0.0622406639004149, 
            0.33402489626556, 0.139004149377593, 0.120331950207469, 0.0186721991701245, 
            0.195020746887967, 0.0912863070539419, 0.04149377593361, 0.0497925311203319, 
            0.0145228215767635, 0.0352697095435685, 0.103734439834025), .Dim = c(31L, 
            20L), .Dimnames = list(NULL, c("", "", "", "", "", "", "", "", 
            "", "", "", "", "", "", "", "", "", "", "", "nodeprob")))), .Names = c("var", 
            "n", "wt", "dev", "yval", "complexity", "ncompete", "nsurrogate", 
            "yval2"), row.names = c(1L, 2L, 4L, 8L, 9L, 18L, 19L, 38L, 76L, 
            77L, 39L, 5L, 10L, 20L, 21L, 42L, 43L, 11L, 22L, 23L, 3L, 6L, 
            12L, 13L, 7L, 14L, 28L, 29L, 58L, 59L, 15L), class = "data.frame"), 
                where = structure(c(4L, 6L, 4L, 6L, 23L, 23L, 14L, 14L, 14L, 
                14L, 14L, 14L, 14L, 14L, 23L, 23L, 23L, 23L, 30L, 23L, 23L, 
                23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 29L, 23L, 
                9L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 16L, 23L, 
                23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
                23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
                23L, 23L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 
                31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 
                31L, 24L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 
                31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 31L, 
                31L, 31L, 24L, 31L, 16L, 17L, 16L, 14L, 11L, 4L, 4L, 11L, 
                10L, 10L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 17L, 
                16L, 16L, 4L, 11L, 11L, 10L, 10L, 10L, 11L, 11L, 11L, 11L, 
                11L, 11L, 10L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 11L, 
                11L, 4L, 10L, 10L, 10L, 10L, 11L, 19L, 19L, 19L, 19L, 19L, 
                19L, 19L, 19L, 19L, 6L, 6L, 6L, 30L, 24L, 16L, 30L, 30L, 
                30L, 30L, 30L, 30L, 24L, 24L, 16L, 24L, 24L, 24L, 24L, 30L, 
                30L, 30L, 17L, 31L, 30L, 17L, 31L, 17L, 30L, 30L, 17L, 30L, 
                14L, 4L, 27L, 27L, 27L, 27L, 27L, 27L, 27L, 27L, 4L, 4L, 
                4L, 4L, 4L, 27L, 4L, 4L, 27L, 4L, 27L, 14L, 4L, 4L, 27L, 
                4L, 4L, 4L, 27L, 14L, 14L, 27L, 14L, 27L, 14L, 4L, 27L, 27L, 
                4L, 4L, 27L, 4L, 4L, 27L, 27L, 4L, 4L, 10L, 4L, 23L, 23L, 
                4L, 4L, 4L, 4L, 10L, 4L, 4L, 4L, 4L, 10L, 10L, 16L, 16L, 
                16L, 16L, 29L, 16L, 19L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
                16L, 16L, 16L, 29L, 16L, 16L, 16L, 16L, 16L, 11L, 4L, 9L, 
                4L, 4L, 4L, 11L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
                16L, 16L, 16L, 16L, 16L, 16L, 30L, 30L, 16L, 16L, 16L, 16L, 
                16L, 17L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 29L, 
                16L, 11L, 11L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
                16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 29L, 16L, 16L, 16L, 
                16L, 29L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
                16L, 29L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
                16L, 16L, 16L, 16L, 16L, 23L, 16L, 16L, 20L, 20L, 20L, 16L, 
                20L, 20L, 16L, 20L, 16L, 20L, 20L, 20L, 16L, 16L, 16L, 16L, 
                20L, 20L, 20L, 20L, 20L, 20L, 16L, 20L, 20L, 20L, 16L, 20L, 
                20L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 20L, 20L, 20L, 20L, 
                17L, 20L, 16L, 20L, 16L, 16L, 16L, 20L, 20L, 20L, 16L, 20L, 
                6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 16L, 6L, 9L, 
                6L, 9L, 6L, 6L, 9L, 9L, 6L, 9L, 9L, 6L, 6L, 9L, 6L, 6L, 6L, 
                6L, 6L, 6L, 6L, 6L, 9L, 6L, 9L), .Names = c("1", "2", "3", 
                "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14", 
                "15", "16", "17", "18", "19", "20", "21", "22", "23", "24", 
                "25", "26", "27", "28", "29", "30", "31", "32", "33", "34", 
                "35", "36", "37", "38", "39", "40", "41", "42", "43", "44", 
                "45", "46", "47", "48", "49", "50", "51", "52", "53", "54", 
                "55", "56", "57", "58", "59", "60", "61", "62", "63", "64", 
                "65", "66", "67", "68", "69", "70", "71", "72", "73", "74", 
                "75", "76", "77", "78", "79", "80", "81", "82", "83", "84", 
                "85", "86", "87", "88", "89", "90", "91", "92", "93", "94", 
                "95", "96", "97", "98", "99", "100", "101", "102", "103", 
                "104", "105", "106", "107", "108", "109", "110", "111", "112", 
                "113", "114", "115", "116", "117", "118", "119", "120", "121", 
                "122", "123", "124", "125", "126", "127", "128", "129", "130", 
                "131", "132", "133", "134", "135", "136", "137", "138", "139", 
                "140", "141", "142", "143", "144", "145", "146", "147", "148", 
                "149", "150", "151", "152", "153", "154", "155", "156", "157", 
                "158", "159", "160", "161", "162", "163", "164", "165", "166", 
                "167", "168", "169", "170", "171", "172", "173", "174", "175", 
                "176", "177", "178", "179", "180", "181", "182", "183", "184", 
                "185", "186", "187", "188", "189", "190", "191", "192", "193", 
                "194", "195", "196", "197", "198", "199", "200", "201", "202", 
                "203", "204", "205", "206", "207", "208", "209", "210", "211", 
                "212", "213", "214", "215", "216", "217", "218", "219", "220", 
                "221", "222", "223", "224", "225", "226", "227", "228", "229", 
                "230", "231", "232", "233", "234", "235", "236", "237", "238", 
                "239", "240", "241", "242", "243", "244", "245", "246", "247", 
                "248", "249", "250", "251", "252", "253", "254", "255", "256", 
                "257", "258", "259", "260", "261", "262", "263", "264", "265", 
                "266", "267", "268", "269", "270", "271", "272", "273", "274", 
                "275", "276", "277", "278", "279", "280", "281", "282", "283", 
                "284", "285", "286", "287", "288", "289", "290", "291", "292", 
                "293", "294", "295", "296", "297", "298", "299", "300", "301", 
                "302", "303", "304", "305", "306", "307", "308", "309", "310", 
                "311", "312", "313", "314", "315", "316", "317", "318", "319", 
                "320", "321", "322", "323", "324", "325", "326", "327", "328", 
                "329", "330", "331", "332", "333", "334", "335", "336", "337", 
                "338", "339", "340", "341", "342", "343", "344", "345", "346", 
                "347", "348", "349", "350", "351", "352", "353", "354", "355", 
                "356", "357", "358", "359", "360", "361", "362", "363", "364", 
                "365", "366", "367", "368", "369", "370", "371", "372", "373", 
                "374", "375", "376", "377", "378", "379", "380", "381", "382", 
                "383", "384", "385", "386", "387", "388", "389", "390", "391", 
                "392", "393", "394", "395", "396", "397", "398", "399", "400", 
                "401", "402", "403", "404", "405", "406", "407", "408", "409", 
                "410", "411", "412", "413", "414", "415", "416", "417", "418", 
                "419", "420", "421", "422", "423", "424", "425", "426", "427", 
                "428", "429", "430", "431", "432", "433", "434", "435", "436", 
                "437", "438", "439", "440", "441", "442", "443", "444", "445", 
                "446", "447", "448", "449", "450", "451", "452", "453", "454", 
                "455", "456", "457", "458", "459", "460", "461", "462", "463", 
                "464", "465", "466", "467", "468", "469", "470", "471", "472", 
                "473", "474", "475", "476", "477", "478", "479", "480", "481", 
                "482")), terms = Activity ~ UpDown.mean + Step.mad + Light.mean + 
                    Magnitude.mean + UpDown.median + Degrees.sd + Degrees.mad + 
                    Temp.mean + UpDown.sd, cptable = structure(c(0.128065395095368, 
                0.125340599455041, 0.0817438692098093, 0.0790190735694823, 
                0.0735694822888283, 0.0626702997275204, 0.0544959128065395, 
                0.0381471389645777, 0.0354223433242507, 0.0245231607629428, 
                0.0190735694822888, 0.0163487738419619, 0.0122615803814714, 
                0.01, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 11, 12, 13, 15, 1, 0.871934604904632, 
                0.746594005449591, 0.664850136239782, 0.5858310626703, 0.512261580381471, 
                0.449591280653951, 0.395095367847411, 0.356948228882834, 
                0.321525885558583, 0.272479564032698, 0.253405994550409, 
                0.237057220708447, 0.212534059945504, 1, 0.866485013623978, 
                0.76566757493188, 0.689373297002725, 0.653950953678474, 0.602179836512262, 
                0.547683923705722, 0.509536784741144, 0.482288828337875, 
                0.427792915531335, 0.392370572207084, 0.359673024523161, 
                0.337874659400545, 0.329700272479564, 0.0254971928581653, 
                0.0283430054730088, 0.0294958968133454, 0.0298736589351208, 
                0.0299105032701856, 0.0298075861027944, 0.0294958968133454, 
                0.0291502462515817, 0.0288367936338873, 0.0280350879786121, 
                0.0273810446099536, 0.026676656975738, 0.0261494628686749, 
                0.0259392074667368), .Dim = c(14L, 5L), .Dimnames = list(
                    c("1", "2", "3", "4", "5", "6", "7", "8", "9", "10", 
                    "11", "12", "13", "14"), c("CP", "nsplit", "rel error", 
                    "xerror", "xstd"))), method = "class", parms = structure(list(
                    prior = structure(c(0.0767634854771784, 0.238589211618257, 
                    0.143153526970954, 0.118257261410788, 0.0269709543568465, 
                    0.0663900414937759, 0.103734439834025, 0.114107883817427, 
                    0.112033195020747), .Dim = 9L, .Dimnames = list(c("1", 
                    "2", "3", "4", "5", "6", "7", "8", "9"))), loss = structure(c(0, 
                    1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
                    1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 
                    1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 
                    1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 
                    1, 1, 1, 1, 1, 1, 1, 0), .Dim = c(9L, 9L)), split = 1), .Names = c("prior", 
                "loss", "split")), control = structure(list(minsplit = 20L, 
                    minbucket = 7, cp = 0.01, maxcompete = 4L, maxsurrogate = 5L, 
                    usesurrogate = 2L, surrogatestyle = 0L, maxdepth = 30L, 
                    xval = 10L), .Names = c("minsplit", "minbucket", "cp", 
                "maxcompete", "maxsurrogate", "usesurrogate", "surrogatestyle", 
                "maxdepth", "xval")), functions = structure(list(summary = function (yval, 
                    dev, wt, ylevel, digits) 
                {
                    nclass <- (ncol(yval) - 2L)/2L
                    group <- yval[, 1L]
                    counts <- yval[, 1L + (1L:nclass)]
                    yprob <- yval[, 1L + nclass + 1L:nclass]
                    nodeprob <- yval[, 2L * nclass + 2L]
                    if (!is.null(ylevel)) 
                        group <- ylevel[group]
                    temp1 <- formatg(counts, format = "%5g")
                    temp2 <- formatg(yprob, format = "%5.3f")
                    if (nclass > 1) {
                        temp1 <- apply(matrix(temp1, ncol = nclass), 1L, 
                            paste, collapse = " ")
                        temp2 <- apply(matrix(temp2, ncol = nclass), 1L, 
                            paste, collapse = " ")
                    }
                    dev <- dev/(wt[1L] * nodeprob)
                    paste0("  predicted class=", format(group, justify = "left"), 
                        "  expected loss=", formatg(dev, digits), "  P(node) =", 
                        formatg(nodeprob, digits), "\n", "    class counts: ", 
                        temp1, "\n", "   probabilities: ", temp2)
                }, print = function (yval, ylevel, digits) 
                {
                    temp <- if (is.null(ylevel)) 
                        as.character(yval[, 1L])
                    else ylevel[yval[, 1L]]
                    nclass <- (ncol(yval) - 2L)/2L
                    yprob <- if (nclass < 5L) 
                        format(yval[, 1L + nclass + 1L:nclass], digits = digits, 
                            nsmall = digits)
                    else formatg(yval[, 1L + nclass + 1L:nclass], digits = 2L)
                    if (!is.matrix(yprob)) 
                        yprob <- matrix(yprob, nrow = 1L)
                    temp <- paste0(temp, " (", yprob[, 1L])
                    for (i in 2L:ncol(yprob)) temp <- paste(temp, yprob[, 
                        i], sep = " ")
                    temp <- paste0(temp, ")")
                    temp
                }, text = function (yval, dev, wt, ylevel, digits, n, use.n) 
                {
                    nclass <- (ncol(yval) - 2L)/2L
                    group <- yval[, 1L]
                    counts <- yval[, 1L + (1L:nclass)]
                    if (!is.null(ylevel)) 
                        group <- ylevel[group]
                    temp1 <- formatg(counts, digits)
                    if (nclass > 1L) 
                        temp1 <- apply(matrix(temp1, ncol = nclass), 1L, 
                            paste, collapse = "/")
                    if (use.n) 
                        paste0(format(group, justify = "left"), "\n", temp1)
                    else format(group, justify = "left")
                }), .Names = c("summary", "print", "text")), numresp = 11L, 
                splits = structure(c(482, 482, 482, 482, 482, 0, 0, 0, 0, 
                0, 321, 321, 321, 321, 321, 0, 0, 0, 0, 0, 123, 123, 123, 
                123, 123, 0, 0, 0, 0, 0, 83, 83, 83, 83, 83, 0, 0, 0, 0, 
                0, 51, 51, 51, 51, 51, 0, 0, 0, 0, 0, 25, 25, 25, 25, 25, 
                0, 0, 0, 0, 0, 198, 198, 198, 198, 198, 0, 0, 0, 0, 0, 158, 
                158, 158, 158, 158, 143, 143, 143, 143, 143, 0, 0, 40, 40, 
                40, 40, 40, 0, 0, 0, 0, 161, 161, 161, 161, 161, 0, 0, 0, 
                0, 0, 67, 67, 67, 67, 67, 0, 94, 94, 94, 94, 94, 0, 0, 0, 
                0, 0, 44, 44, 44, 44, 44, 0, 0, 0, 0, 0, 24, 24, 24, 24, 
                24, 0, 0, 0, 0, 0, 1, 1, -1, 1, -1, 1, 1, 1, -1, 1, -1, -1, 
                -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, 1, 1, -1, -1, -1, 
                -1, 1, 1, 1, -1, 1, -1, -1, -1, 1, -1, 1, 1, -1, 1, -1, -1, 
                1, -1, -1, 1, 1, 1, 1, 1, 1, -1, -1, -1, -1, 1, -1, 1, -1, 
                -1, -1, 1, -1, -1, -1, -1, -1, 1, 1, -1, -1, -1, -1, 1, -1, 
                -1, -1, -1, -1, 1, -1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, 1, 
                -1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, 1, -1, 
                1, 1, 1, -1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, 1, 1, 
                -1, -1, -1, -1, 1, 1, -1, -1, 1, 1, 1, 40.3977754210497, 
                37.9905728165165, 36.6447026875567, 29.3533707812309, 28.9957999383241, 
                0.883817427385892, 0.879668049792531, 0.786307053941909, 
                0.746887966804979, 0.738589211618257, 33.313552668237, 25.854191768364, 
                25.1176375478047, 21.4395502385576, 19.2574358828128, 0.728971962616822, 
                0.713395638629283, 0.685358255451713, 0.669781931464174, 
                0.641744548286604, 20.1361739641493, 19.7080117798082, 19.4660753880266, 
                17.3735620190427, 16.5732182451695, 0.926829268292683, 0.886178861788618, 
                0.853658536585366, 0.772357723577236, 0.715447154471545, 
                16.3689611386723, 11.1743127502559, 6.64404283801875, 6.36078560818618, 
                5.78584177981769, 0.72289156626506, 0.698795180722892, 0.686746987951807, 
                0.63855421686747, 0.626506024096386, 4.70057315233786, 3.80130189727713, 
                3.65424836601307, 3.57040998217469, 2.12593295382669, 0.686274509803922, 
                0.666666666666667, 0.647058823529412, 0.607843137254902, 
                0.607843137254902, 7.87116883116883, 3.24, 2.9374358974359, 
                2.62333333333334, 2.40363636363637, 0.76, 0.76, 0.76, 0.72, 
                0.72, 24.6737693389592, 23.1536709122916, 16.7494410909904, 
                15.647723132969, 15.5942902532474, 0.843434343434343, 0.828282828282828, 
                0.823232323232323, 0.803030303030303, 0.803030303030303, 
                16.8488802336904, 9.01977717622006, 6.86464451806778, 6.56426484907497, 
                6.381275492429, 12.8189733494852, 10.5371457489879, 7.6231246590289, 
                7.11000625390868, 5.5482176360225, 0.909090909090909, 0.902097902097902, 
                13.65, 13.5145161290323, 10.2462382445141, 5.2625, 5.06636828644501, 
                0.975, 0.925, 0.875, 0.775, 33.8727270216908, 33.0379419567925, 
                20.8053004236927, 13.1359851988899, 11.7028985507246, 0.782608695652174, 
                0.708074534161491, 0.658385093167702, 0.658385093167702, 
                0.658385093167702, 11.2064390690227, 3.61422616646498, 2.86575188573263, 
                2.79031764255645, 2.23134328358209, 0.895522388059702, 29.0381431334623, 
                19.9458305337813, 13.2695572748764, 10.6789007092199, 8.28723404255319, 
                0.776595744680851, 0.765957446808511, 0.75531914893617, 0.74468085106383, 
                0.680851063829787, 15.9090909090909, 15.3521550912855, 4.8171368861024, 
                4.02296254470167, 4.02296254470167, 0.977272727272727, 0.75, 
                0.727272727272727, 0.704545454545455, 0.636363636363636, 
                6.10924369747899, 5.08888888888889, 5.08888888888889, 2.3781512605042, 
                1.87394957983193, 0.958333333333333, 0.833333333333333, 0.75, 
                0.75, 0.75, 3.3355, 159, 25.35, 1.95, 11.642, 3.8275, 4.3645, 
                2.5, 30.85, 3.336, 25.15, 1.1405, 127, 15.1965, 52.477, -40.9975, 
                -41.552, 108, 0.9985, 5.0075, 21.865, 11.49, 1.109, 19.053, 
                7.413, 12.2315, 8.9515, 40.401, 210, -15.3205, 1.109, 54.5, 
                -60.5, 103.827, -10.5925, -7.2315, 52.2615, -21.1935, 49.173, 
                66.1215, 34.5, 24.45, 41.7425, 24.463, 1.015, 33.772, 24.463, 
                -4.768, -41.2085, 22.25, 15.5, 1.0175, 29.088, 178.2825, 
                -10.5925, -10.5925, 178.2825, 1.0175, -3.6425, 29.088, 1.1405, 
                32.2465, 56.1975, 158.5, 26.6805, 46.3315, 68.115, 48.5565, 
                6572.3655, 25.35, 158.5, 32.2465, 63.0725, 15.1025, 26.6805, 
                10.5, 32.2465, 26.6805, 63.0725, 16.4075, 121.0595, 0.965, 
                27.2, 1.8745, 42.2475, 733.1455, 80.897, 1.8745, 42.2475, 
                1637.161, 80.897, 1.0045, 27.7, 147, 0.5415, 0.3705, 26.15, 
                0.5415, -79.486, -80.839, 0.426, 28.4, 0.3705, -1.3435, 1.255, 
                0.489, 0.961, 0.5, 25.75, 2.224, 0.833, 10.086, 30.2, 0.3705, 
                1.077, 2.6735, 10.086, 147, 26.8, 6.199, 4.8485, 3.7915, 
                26.8, 6.199, 1.0365, 2.224, 2.064, -4.014, 3.01, 29.35, 2.224, 
                1.016, -5.5845, 27.35, 1.056, 2.224, 16.62, 0, 0, 0, 0, 0, 
                0.652173913043478, 0.639751552795031, 0.360248447204969, 
                0.242236024844721, 0.217391304347826, 0, 0, 0, 0, 0, 0.292682926829268, 
                0.252032520325203, 0.178861788617886, 0.138211382113821, 
                0.0650406504065041, 0, 0, 0, 0, 0, 0.775, 0.65, 0.55, 0.3, 
                0.125, 0, 0, 0, 0, 0, 0.28125, 0.21875, 0.1875, 0.0625, 0.03125, 
                0, 0, 0, 0, 0, 0.36, 0.32, 0.28, 0.2, 0.2, 0, 0, 0, 0, 0, 
                0.454545454545455, 0.454545454545455, 0.454545454545455, 
                0.363636363636364, 0.363636363636364, 0, 0, 0, 0, 0, 0.225, 
                0.15, 0.125, 0.025, 0.025, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
                0.1875, 0.125, 0, 0, 0, 0, 0, 0.9, 0.7, 0.5, 0.1, 0, 0, 0, 
                0, 0, 0.477611940298507, 0.298507462686567, 0.17910447761194, 
                0.17910447761194, 0.17910447761194, 0, 0, 0, 0, 0, 0.222222222222222, 
                0, 0, 0, 0, 0, 0.522727272727273, 0.5, 0.477272727272727, 
                0.454545454545455, 0.318181818181818, 0, 0, 0, 0, 0, 0.95, 
                0.45, 0.4, 0.35, 0.2, 0, 0, 0, 0, 0, 0.857142857142857, 0.428571428571429, 
                0.142857142857143, 0.142857142857143, 0.142857142857143), .Dim = c(137L, 
                5L), .Dimnames = list(c("Degrees.mad", "Light.mean", "Temp.mean", 
                "UpDown.sd", "Degrees.sd", "UpDown.sd", "Degrees.sd", "Light.mean", 
                "Temp.mean", "Step.mad", "Temp.mean", "Magnitude.mean", "Light.mean", 
                "Degrees.mad", "Degrees.sd", "UpDown.median", "UpDown.mean", 
                "Light.mean", "Magnitude.mean", "Degrees.sd", "Degrees.sd", 
                "Degrees.mad", "Magnitude.mean", "UpDown.sd", "Step.mad", 
                "Degrees.mad", "UpDown.sd", "Step.mad", "Light.mean", "UpDown.mean", 
                "Magnitude.mean", "Light.mean", "UpDown.median", "Degrees.sd", 
                "UpDown.mean", "UpDown.median", "Degrees.mad", "UpDown.mean", 
                "UpDown.sd", "Degrees.sd", "Light.mean", "Temp.mean", "Degrees.sd", 
                "Degrees.mad", "Magnitude.mean", "Degrees.sd", "Degrees.mad", 
                "UpDown.median", "UpDown.mean", "Temp.mean", "Light.mean", 
                "Magnitude.mean", "UpDown.sd", "Step.mad", "UpDown.mean", 
                "UpDown.mean", "Step.mad", "Magnitude.mean", "UpDown.median", 
                "UpDown.sd", "Magnitude.mean", "Degrees.mad", "Degrees.sd", 
                "Light.mean", "UpDown.sd", "Degrees.mad", "Degrees.sd", "UpDown.sd", 
                "Step.mad", "Temp.mean", "Light.mean", "Degrees.mad", "Degrees.sd", 
                "UpDown.mean", "UpDown.sd", "Light.mean", "Degrees.mad", 
                "UpDown.sd", "Degrees.sd", "UpDown.mean", "Degrees.sd", "Magnitude.mean", 
                "Temp.mean", "Magnitude.mean", "UpDown.sd", "Step.mad", "Degrees.sd", 
                "Magnitude.mean", "UpDown.sd", "Step.mad", "Degrees.sd", 
                "Magnitude.mean", "Temp.mean", "Light.mean", "Degrees.sd", 
                "Degrees.mad", "Temp.mean", "Degrees.sd", "UpDown.mean", 
                "UpDown.median", "UpDown.sd", "Temp.mean", "Degrees.mad", 
                "UpDown.mean", "UpDown.median", "UpDown.sd", "Magnitude.mean", 
                "Light.mean", "Temp.mean", "Degrees.mad", "UpDown.sd", "UpDown.mean", 
                "Temp.mean", "Degrees.mad", "UpDown.sd", "Degrees.sd", "UpDown.mean", 
                "Light.mean", "Temp.mean", "Degrees.sd", "UpDown.mean", "UpDown.median", 
                "Temp.mean", "Degrees.sd", "Magnitude.mean", "Degrees.mad", 
                "UpDown.mean", "UpDown.mean", "UpDown.median", "Temp.mean", 
                "Degrees.mad", "Magnitude.mean", "UpDown.median", "Temp.mean", 
                "Magnitude.mean", "Degrees.mad", "UpDown.sd"), c("count", 
                "ncat", "improve", "index", "adj"))), variable.importance = structure(c(118.601679035572, 
                113.739439279905, 106.711650595967, 88.2903166027879, 87.5998044869168, 
                76.7580327475607, 43.097447380255, 30.8766690194563, 29.8357337886118
                ), .Names = c("Temp.mean", "Light.mean", "Magnitude.mean", 
                "Degrees.sd", "Degrees.mad", "UpDown.sd", "UpDown.mean", 
                "Step.mad", "UpDown.median")), y = c(9L, 9L, 9L, 5L, 6L, 
                6L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
                4L, 4L, 4L, 4L, 4L, 4L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
                7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
                9L, 9L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 
                6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
                6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
                2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
                8L, 8L, 8L, 8L, 8L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
                1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
                1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), ordered = structure(c(FALSE, 
                FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE), .Names = c("UpDown.mean", 
                "Step.mad", "Light.mean", "Magnitude.mean", "UpDown.median", 
                "Degrees.sd", "Degrees.mad", "Temp.mean", "UpDown.sd"))), .Names = c("frame", 
            "where", "terms", "cptable", "method", "parms", "control", "functions", 
            "numresp", "splits", "variable.importance", "y", "ordered"), xlevels = structure(list(), .Names = character(0)), ylevels = c("active", 
            "cycle", "drive", "nonwear", "run", "sit", "sleep", "swim", "walk"
            ), class = c("rpart", "GENEA"), features = c("Degrees.mad", "UpDown.sd", 
            "Temp.mean", "UpDown.median", "Degrees.sd", "Magnitude.mean", 
            "Light.mean", "UpDown.mean"))
    
    f4$call <- NULL
    
    expect_that(f4, equals(t4))
    
    # 3.2 Specify Features not in data is error
    
    expect_error(createGENEAFit(data = p2, features = "not.found", 
            plot = FALSE, verbose = FALSE), 
        regexp = "features not found in data")
    
    # 3.3 Specify Features, specify model features
    
    f5 <- createGENEAFit(data = p2, features = c("Degrees.mad", 
            "Degrees.mean", "Degrees.median", "Degrees.sd", 
            "Degrees.var", "Light.max", "Light.mean", "Magnitude.mean",  
            "Principal.Frequency.mad", "Principal.Frequency.median", 
            "Principal.Frequency.ratio"),
        plot = FALSE, verbose = FALSE)
    
    expect_that(features(f5), equals(c("Principal.Frequency.mad", 
            "Principal.Frequency.median", "Light.mean", 
            "Light.max", "Degrees.median", "Degrees.mean", "Degrees.mad", 
            "Degrees.sd", "Magnitude.mean")))
    
    expect_that(levels(f5), equals(c("active", "cycle", "drive", 
            "nonwear", "run", "sit", "sleep", "swim", "walk")))
    
    rm(list = c("p1", "p2", "f1", "f2", "f3", "f4", "f5", "t3", "t4"))
})



