% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/envelopes.r
\name{global_envelope_test}
\alias{global_envelope_test}
\alias{GET}
\alias{global_envelope_test2d}
\title{Global envelope test}
\usage{
global_envelope_test(
  curve_sets,
  type = "erl",
  alpha = 0.05,
  alternative = c("two.sided", "less", "greater"),
  ties = "erl",
  probs = c(0.025, 0.975),
  quantile.type = 7,
  central = "mean",
  nstep = 2,
  ...
)
}
\arguments{
\item{curve_sets}{A \code{curve_set} (see \code{\link{create_curve_set}})
or an \code{\link[spatstat]{envelope}} object containing a data function and simulated functions.
If an envelope object is given, it must contain the summary
functions from the simulated patterns which can be achieved by setting
\code{savefuns = TRUE} when calling \code{\link[spatstat]{envelope}}.
Alternatively, a list of \code{curve_set} or \code{\link[spatstat]{envelope}} objects can be given.}

\item{type}{The type of the global envelope with current options for 'rank', 'erl', 'cont', 'area',
'qdir', 'st' and 'unscaled'. See details.}

\item{alpha}{The significance level. The 100(1-alpha)\% global envelope will be calculated.}

\item{alternative}{A character string specifying the alternative hypothesis.
Must be one of the following: "two.sided" (default), "less" or "greater".
The last two options only available for types \code{'rank'}, \code{'erl'},
\code{'cont'} and \code{'area'}.}

\item{ties}{The method to obtain a unique p-value when  \code{type = 'rank'}.
Possible values are 'midrank', 'random', 'conservative', 'liberal' and 'erl'.
For 'conservative' the resulting p-value will be the highest possible.
For 'liberal' the p-value will be the lowest possible.
For 'random' the rank of the obs within the tied values is uniformly sampled so that the resulting
p-value is at most the conservative option and at least the liberal option.
For 'midrank' the mid-rank within the tied values is taken.
For 'erl' the extreme rank length p-value is calculated.
The default is 'erl'.}

\item{probs}{A two-element vector containing the lower and upper
quantiles for the measure 'q' or 'qdir', in that order and on the interval [0, 1].
The default values are 0.025 and 0.975, suggested by Myllymäki et al. (2015, 2017).}

\item{quantile.type}{As type argument of \code{\link[stats]{quantile}}, how to
calculate quantiles for 'q' or 'qdir'.}

\item{central}{Either "mean" or "median". If the curve sets do not contain the component
\code{theo} for the theoretical central function, then the central function (used for plotting only)
is calculated either as the mean or median of functions provided in the curve sets.}

\item{nstep}{1 or 2 for how to contruct a combined global envelope if list of curve sets
is provided. 2 (default) for a two-step combining procedure, 1 for one-step.}

\item{...}{Additional parameters to be passed to \code{\link{central_region}}.}
}
\value{
Either an object of class "global_envelope" and "fv"
(see \code{\link[spatstat]{fv.object}}) or "combined_global_envelope" for
combined tests. The objects can be printed and plotted directly.

The "global_envelope" is essentially a data frame containing columns
\itemize{
\item r = the vector of values of the argument r at which the test was made
\item obs = values of the data function
\item lo = the lower envelope based on the simulated functions
\item hi = the upper envelope based on the simulated functions
\item central = If the curve_set (or envelope object) contains a component 'theo',
      then this function is used as the central curve and returned in this component.
      Otherwise, the central_curve is the mean of the test functions T_i(r), i=2, ..., s+1.
      Used for visualization only.
}
Moreover, the return value has the same attributes as the object returned by
\code{\link{central_region}} and in addition
\itemize{
  \item p = A point estimate for the p-value (default is the mid-rank p-value).
}
and in the case that \code{type = 'rank'} also
\itemize{
  \item p_interval = The p-value interval [p_liberal, p_conservative].
  \item ties = As the argument \code{ties}.
}

The "combined_global_envelope" is a list of "global_envelope" objects
corresponding to the components of \code{curve_sets}. The second level envelope
on which the envelope construction is based on is saved in the attribute
"level2_ge".
}
\description{
Global envelope test, global envelopes and p-values
}
\details{
Given a \code{curve_set} (see \code{\link{create_curve_set}} for how to create such an object)
or an \code{\link[spatstat]{envelope}} object,
which contains both the data curve (or function or vector) \eqn{T_1(r)}{T_1(r)}
(in the component \code{obs}) and
the simulated curves \eqn{T_2(r),\dots,T_{s+1}(r)}{T_2(r),...,T_(s+1)(r)}
(in the component \code{sim_m}),
the function \code{global_envelope_test} performs a global envelope test.
The functionality of the function is rather similar to the function
\code{\link{central_region}}, but in addition to ordering the functions from
the most extreme one to the least extreme one using different measures
and providing the global envelopes with intrinsic
graphical interpretation, p-values are calculated for the test.
Thus, while \code{\link{central_region}} can be used to construct global
envelopes in a general setting, the function \code{\link{global_envelope_test}}
is devoted to testing as its name suggests.

The function \code{global_envelope_test} is the main function for global envelope tests
(for simple hypotheses).
Different \code{type} of global envelope tests can be performed.
We use such ordering of the functions for which we are able to construct global
envelopes with intrinsic graphical interpretation.
\itemize{
  \item \code{'rank'}: the completely non-parametric rank envelope test (Myllymäki et al., 2017)
  based on minimum of pointwise ranks
  \item \code{'erl'}: the completely non-parametric rank envelope test based on extreme rank lengths
  (Myllymäki et al., 2017; Mrkvička et al., 2018) based on number of minimal pointwise ranks
 \item \code{'cont'}: the completely non-parametric rank envelope test based on continuous rank
 (Hahn, 2015; Mrkvička et al., 2019) based on minimum of continuous pointwise ranks
 \item \code{'area'}: the completely non-parametric rank envelope test based on area rank
 (Mrkvička et al., 2019) based on area between continuous pointwise ranks and minimum
 pointwise ranks for those argument (r) values for which pointwise ranks achieve the minimum
 (it is a combination of erl and cont)
  \item "qdir", the directional quantile envelope test, protected against unequal variance and
  asymmetry of T(r) for different distances r (Myllymäki et al., 2015, 2017)
  \item "st", the studentised envelope test, protected against unequal variance of T(r) for
  different distances r (Myllymäki et al., 2015, 2017)
  \item "unscaled", the unscaled envelope (providing a baseline) that has a contant width and
  that corresponds to the classical maximum deviation test (Ripley, 1981).
}
See \code{\link{forder}} and \code{\link{central_region}} and the references
for more detailed description of the measures and the corresponding envelopes.

The first four types are global rank envelopes.
The \code{'rank'} envelope test is a completely non-parametric test,
which provides the 100(1-alpha)% global envelope for the chosen test function
T(r) on the chosen interval of distances and associated p-values.
The other three are modifications of \code{'rank'} to treat the ties in
the extreme rank ordering on which the \code{'rank'} test is based on.

The last three envelopes are global scaled maximum absolute difference (MAD)
envelope tests. The unscaled envelope test leads to envelopes with constant width over the
distances r. Thus, it suffers from unequal variance of T(r) over the distances r and
from the asymmetry of distribution of T(r). We recommend to use the other global
envelope tests available. The unscaled envelope is provided as a reference.
}
\section{Ranking of the curves}{

The options for measures to order the functions from the most extreme one to the least extreme one
are given by the argument \code{type}: 'rank', 'erl', 'cont', 'area', 'qdir', 'st', 'unscaled'.
The options are
\itemize{
\item 'rank': extreme ranks (Myllymäki et al., 2017)
\item 'erl': extreme rank lengths (Myllymäki et al., 2017; Mrkvička et al., 2018)
\item 'cont': continuous ranks (Hahn, 2015; Mrkvička et al., 2019)
\item 'area': area ranks (Mrkvička et al., 2019)
\item 'qdir': the directional quantile maximum absolute deviation (MAD) measure (Myllymäki et al., 2015, 2017)
\item 'st': the studentized MAD measure (Myllymäki et al., 2015, 2017)
\item 'unscaled': the unscaled MAD measure (Ripley, 1981)
}
See more detailed description of the envelopes and measures in \code{\link{central_region}}
and \code{\link{forder}}.
}

\section{Global envelope}{

Based on the measures used to rank the functions, the 100(1-alpha)\% global envelope is provided.
It corresponds to the 100*coverage\% central region.
}

\section{P-values}{

In the case \code{type="rank"}, based on the extreme ranks k_i, i=1, ..., s+1,
the p-interval is calculated. Because the extreme ranks contain ties, there is not just
one p-value. The p-interval is given by the most liberal and the most conservative p-value
estimate. Also a single p-value is calculated.
By default this single p-value is the extreme rank length p-value ("erl"),
but another option can be used by specifying \code{ties} argument.

If the case \code{type = "erl"}, the (single) p-value based on the extreme rank length ordering
of the functions is calculated and returned in the attribute \code{p}.
The same is done for other measures, the p-value always being correspondent to the chosen measure.
}

\section{Number of simulations}{

For the global \code{"rank"} envelope test, Myllymäki et al. (2017) recommended to use
at least 2500 simulations for testing at the significance level alpha = 0.05 for single
function tests, based on experiments with summary functions for point processes.
In this case, the width of the p-interval associated with the extreme rank measure tended
to be smaller than 0.01.
The tests \code{'erl'}, \code{'cont'} and \code{'area'}, similarly as
the MAD deviation/envelope tests \code{'qdir'}, \code{'st'} and \code{'unscaled'},
allow in principle a lower number of simulations to be used than the test based on
extreme ranks (\code{'rank'}), because no ties occur for these measures.
If affordable, we recommend in any case some thousands of simulations for all the measures
to achieve a good power and repeatability of the test.
}

\section{Tests based on several functions}{

If a list of (suitable) objects are provided in the argument \code{curve_sets},
then by default (\code{nstep = 2}) the two-step combining procedure is used to
perform the combined global test as described in Myllymäki and Mrkvička (2019).
If \code{nstep = 1} and the lengths of the multivariate vectors in each component
of the list are equal, then the one-step combining procedure is used where the
functions are concatenated together into a one long vector.
}

\examples{
# Goodness-of-fit testing for simple hypothesis
if(require("spatstat", quietly=TRUE)) {
  # Testing complete spatial randomness (CSR)
  #==========================================
  X <- unmark(spruces)

  \donttest{nsim <- 1999 # Number of simulations}
  \dontshow{nsim <- 19 # Number of simulations}

  # Illustration of general workflow for simple hypotheses
  #=======================================================
  # First illustrate the general workflow for the test by this example
  # of CSR test for a point pattern X using the empirical L-function.
  # Define the argument values at which the functions are evaluated
  obs.L <- Lest(X, correction = "translate")
  r <- obs.L[['r']]
  # The test function for the data
  obs <- obs.L[['trans']] - r
  # Prepare simulations and calculate test functions for them at same r as 'obs'
  sim <- matrix(nrow = length(r), ncol = nsim)
  for(i in 1:nsim) {
    sim.X <- runifpoint(ex = X) # simulation under CSR
    sim[, i] <- Lest(sim.X, correction = "translate", r = r)[['trans']] - r
  }
  # Create a curve_set containing argument values, observed and simulated functions
  cset <- create_curve_set(list(r = r, obs = obs, sim_m = sim))
  # Perform the test
  res <- global_envelope_test(cset, type="erl")
  plot(res, ylab = expression(italic(hat(L)(r)-r)))

  # Simple hypothesis for a point pattern utilizing the spatstat package
  #=====================================================================
  # Generate nsim simulations under CSR, calculate L-function for the data and simulations
  env <- envelope(X, fun="Lest", nsim=nsim,
                  savefuns=TRUE, # save the functions
                  correction="translate", # edge correction for L
                  transform = expression(.-r), # centering
                  simulate=expression(runifpoint(ex=X))) # Simulate CSR
  # The rank envelope test (ERL)
  res <- global_envelope_test(env, type="erl")
  # Plot the result
  plot(res)

  ## Advanced use:
  # Choose the interval of distances [r_min, r_max] (at the same time create a curve_set from 'env')
  cset <- crop_curves(env, r_min=1, r_max=7)
  # Do the rank envelope test (erl)
  res <- global_envelope_test(cset, type="erl")
  plot(res, ylab=expression(italic(L(r)-r)))

  \donttest{
  # Random labeling test
  #=====================
  mpp <- spruces
  # 1) Perform simulations under the random labelling hypothesis and calculate
  # the test function T(r) for the data pattern (mpp) and each simulation.
  # The command below specifies that the test function is T(r) = \hat{L}_mm(r),
  # which is an estimator of the mark-weighted L function, L_mm(r),
  # with translational edge correction.
  nsim <- 1999 # Number of simulations
  env <- envelope(mpp, fun=Kmark, nsim = nsim, f=function(m1, m2) { m1*m2 },
                  correction="translate", returnL=TRUE,
                  simulate=expression(rlabel(mpp, permute=TRUE)), # Permute the marks
                  savefuns=TRUE) # Save the functions
  # 2)
  # Crop curves to desired r-interval
  curve_set <- crop_curves(env, r_min=1.5, r_max=9.5)
  # Center the functions, i.e. take \hat{L}_mm(r)-T_0(r).
  # Below T_0(r) = \hat{L}(r) is the mean of simulated functions.
  # (This is only for visualization, does not affect the test result.)
  curve_set <- residual(curve_set)
  # 3) Do the rank envelope test
  res <- global_envelope_test(curve_set)
  # 4) Plot the test result
  plot(res, ylab=expression(italic(L[mm](r)-L(r))))

  # Goodness-of-fit test (typically conservative, see ?GET.composite for adjusted tests)
  #=====================
  X <- unmark(spruces)
  # Minimum distance between points in the pattern
  min(nndist(X))
  # Fit a model
  fittedmodel <- ppm(X, interaction=Hardcore(hc=1)) # Hardcore process

  # Simulating Gibbs process by 'envelope' is slow, because it uses the MCMC algorithm
  #env <- envelope(fittedmodel, fun="Jest", nsim=999, savefuns=TRUE,
  #                correction="none", r=seq(0, 4, length=500))

  # Using direct algorihm can be faster, because the perfect simulation is used here.
  simulations <- NULL
  nsim <- 999 # Number of simulations
  for(j in 1:nsim) {
     simulations[[j]] <- rHardcore(beta=exp(fittedmodel$coef[1]),
                                   R=fittedmodel$interaction$par$hc,
                                   W=X$window)
     if(j\%\%10==0) cat(j, "...", sep="")
  }
  env <- envelope(X, simulate=simulations, fun="Jest", nsim=length(simulations),
                  savefuns=TRUE, correction="none", r=seq(0, 4, length=500))
  curve_set <- crop_curves(env, r_min=1, r_max=3.5)
  res <- global_envelope_test(curve_set, type="erl"); plot(res, ylab=expression(italic(J(r))))
  }

  # A combined global envelope test
  #================================
  # As an example test CSR of the saplings point pattern by means of
  # L, F, G and J functions.
  data(saplings)
  X <- saplings

  \donttest{nsim <- 499 # Number of simulations}
  \dontshow{nsim <- 19 # Number of simulations}
  # Specify distances for different test functions
  n <- 500 # the number of r-values
  rmin <- 0; rmax <- 20; rstep <- (rmax-rmin)/n
  rminJ <- 0; rmaxJ <- 8; rstepJ <- (rmaxJ-rminJ)/n
  r <- seq(0, rmax, by=rstep)    # r-distances for Lest
  rJ <- seq(0, rmaxJ, by=rstepJ) # r-distances for Fest, Gest, Jest

  # Perform simulations of CSR and calculate the L-functions
  env_L <- envelope(X, nsim=nsim,
   simulate=expression(runifpoint(ex=X)),
   fun="Lest", correction="translate",
   transform=expression(.-r), # Take the L(r)-r function instead of L(r)
   r=r,                       # Specify the distance vector
   savefuns=TRUE,             # Save the estimated functions
   savepatterns=TRUE)         # Save the simulated patterns
  # Take the simulations from the returned object
  simulations <- attr(env_L, "simpatterns")
  # Then calculate the other test functions F, G, J for each simulated pattern
  env_F <- envelope(X, nsim=nsim,
                    simulate=simulations,
                    fun="Fest", correction="Kaplan", r=rJ,
                    savefuns=TRUE)
  env_G <- envelope(X, nsim=nsim,
                    simulate=simulations,
                    fun="Gest", correction="km", r=rJ,
                    savefuns=TRUE)
  env_J <- envelope(X, nsim=nsim,
                    simulate=simulations,
                    fun="Jest", correction="none", r=rJ,
                    savefuns=TRUE)

  # Crop the curves to the desired r-interval I
  curve_set_L <- crop_curves(env_L, r_min=rmin, r_max=rmax)
  curve_set_F <- crop_curves(env_F, r_min=rminJ, r_max=rmaxJ)
  curve_set_G <- crop_curves(env_G, r_min=rminJ, r_max=rmaxJ)
  curve_set_J <- crop_curves(env_J, r_min=rminJ, r_max=rmaxJ)

  res <- global_envelope_test(curve_sets=list(curve_set_L, curve_set_F,
                                              curve_set_G, curve_set_J))
  plot(res, labels=c("L(r)-r", "F(r)", "G(r)", "J(r)"))
}

# A test based on a low dimensional random vector
#================================================
# Let us generate some example data.
X <- matrix(c(-1.6,1.6),1,2) # data pattern X=(X_1,X_2)
if(requireNamespace("mvtnorm", quietly=TRUE)) {
  Y <- mvtnorm::rmvnorm(200,c(0,0),matrix(c(1,0.5,0.5,1),2,2)) # simulations
  plot(Y, xlim=c(min(X[,1],Y[,1]), max(X[,1],Y[,1])), ylim=c(min(X[,2],Y[,2]), max(X[,2],Y[,2])))
  points(X, col=2)

  # Test the null hypothesis is that X is from the distribution of Y's (or if it is an outlier).

  # Case 1. The test vector is (X_1, X_2)
  cset1 <- create_curve_set(list(r=1:2, obs=as.vector(X), sim_m=t(Y)))
  res1 <- global_envelope_test(cset1)
  plot(res1)

  # Case 2. The test vector is (X_1, X_2, (X_1-mean(Y_1))*(X_2-mean(Y_2))).
  t3 <- function(x, y) { (x[,1]-mean(y[,1]))*(x[,2]-mean(y[,2])) }
  cset2 <- create_curve_set(list(r=1:3, obs=c(X[,1],X[,2],t3(X,Y)), sim_m=rbind(t(Y), t3(Y,Y))))
  res2 <- global_envelope_test(cset2)
  plot(res2)
}

# Examples with image sets

# Example of spatial point pattern residuals
#===========================================
if(require("spatstat", quietly=TRUE)) {
  data(cells)
  X <- cells
  # Fit the hard-core process
  model <- ppm(X, interaction=Hardcore())
  summary(model)
  HD <- 0.08168525 # Hard-core process
  # Choose a bandwitdh by Scott's rule of thumb
  ds <- bw.scott(X); ds
  # Calculate raw residuals of the fitted model
  # (To use the default pixel array dimensions remove dimyx, see ?as.mask)
  u <- diagnose.ppm(model, type="raw", rbord = HD, which ="smooth",
                    sigma=ds, plot.it=FALSE, dimyx=32)
  obs <- u$smooth$Z$v
  # Generate simulations from the hard-core null model
  \dontshow{nsim <- 19}
  \donttest{nsim <- 499 # Number of simulations; increase for serious analysis!}
  simulations <- NULL
  ext.factor <- max(X$window$xrange[2]-X$window$xrange[1],
                    X$window$yrange[2]-X$window$yrange[1]) / 10
  win.extend <- owin(c(X$window$xrange[1]-ext.factor, X$window$xrange[2]+ext.factor),
                     c(X$window$yrange[1]-ext.factor, X$window$yrange[2]+ext.factor))
  mod02 <- list(cif="hardcore", par=list(beta=exp(model$fitin$coefs[1]),hc=HD), w=win.extend)
  # starting point pattern in an extended window
  x.start <- runifpoint(X$n, win=win.extend)
  # simulations
  for(sss in 1:nsim){
    uppp <- rmh(model=mod02, start=list(x.start=x.start), control=list(p=1,nrep=1e5,nverb=5000))
    f <- uppp$x > X$window$xrange[1] & uppp$x < X$window$xrange[2] &
         uppp$y > X$window$yrange[1] & uppp$y < X$window$yrange[2]
    simulations[[sss]] <- ppp(uppp$x[f], uppp$y[f], window=X$window)
  }
  # Calculate the raw residuals for simulations
  sim <- array(0, c(u$smooth$Z$dim, nsim))
  for(i in 1:length(simulations)) {
    model <- ppm(simulations[[i]],interaction=Hardcore(HD));
    u_sim <- diagnose.ppm(model, type="raw", rbord = HD, which ="smooth",
                          sigma=ds, plot.it=FALSE, dimyx=32)
    sim[,,i] <- u_sim$smooth$Z$v
    if((i \%\% 100)==0) cat(i, ' ')
  }
  # Constract the global envelope test for the (2D) raw residuals
  iset <- create_image_set(list(obs=obs, sim_m=sim))
  res <- global_envelope_test(iset, type="area")
  plot(res)
  plot(res) + ggplot2::scale_fill_gradient(low="black", high="white")
}
}
\references{
Mrkvička, T., Myllymäki, M. and Hahn, U. (2017). Multiple Monte Carlo testing, with applications in spatial point processes. Statistics & Computing 27 (5): 1239-1255. doi: 10.1007/s11222-016-9683-9

Mrkvička, T., Myllymäki, M., Jilek, M., and Hahn, U. (2018). A one-way ANOVA test for functional data with graphical interpretation. arXiv:1612.03608 [stat.ME]

Mrkvička, T., Myllymäki, M. and Narisetty, N. N. (2019) New methods for multiple testing in permutation inference for the general linear model. arXiv:1906.09004 [stat.ME]

Myllymäki, M., Grabarnik, P., Seijo, H. and Stoyan. D. (2015). Deviation test construction and power comparison for marked spatial point patterns. Spatial Statistics 11: 19-34. doi: 10.1016/j.spasta.2014.11.004

Myllymäki, M., Mrkvička, T., Grabarnik, P., Seijo, H. and Hahn, U. (2017). Global envelope tests for spatial point patterns. Journal of the Royal Statistical Society: Series B (Statistical Methodology), 79: 381–404. doi: 10.1111/rssb.12172

Myllymäki, M., Mrkvička, T. Global envelopes in R.

Ripley, B.D. (1981). Spatial statistics. Wiley, New Jersey.
}
\seealso{
\code{\link{plot.global_envelope}}, \code{\link{central_region}},
\code{\link{GET.composite}}
}
