#' Generate optimized code using the Generative AI model based on a prompt
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters and generates optimized code based on the prompt.
#'
#' @param model.parameter A character vector containing the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. A higher value results in more creative responses,
#' while a lower value produces more straightforward text.
#' @param prompt A character string representing the query for text generation.
#' @param language A character string representing the programming language of the code in
#' the prompt.
#' @param goal A character string representing the goal for code optimization.
#'
#' @return If successful, a character string (optimized code) generated by the Generative AI
#' model based on the provided query and parameters. If the API response indicates an error,
#' the function halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial
#' to ensure successful text generation by the Generative AI model. If any of the
#' provided parameters is incorrect, the function will respond with an error message based
#' on the information received from the API. Use the function \code{available.models} to
#' see all supported Generative AI models.
#'
#' A complete prompt behind the function is as follows:
#'
#' Optimize the following \code{language} code.
#'
#' The goal is: \code{goal}
#'
#' # Code starts #
#'
#' \code{prompt}
#'
#' # Code ends #
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Generate text
#'  temperature = 0.9
#'  prompt = "foo <- function(n) {
#'              if (n <= 0) {
#'                return(0)
#'              } else if (n == 1) {
#'                return(1)
#'              } else {
#'                return(foo(n - 1) + foo(n - 2))
#'              }
#'            }"
#'  goal = "Improve the runtime of the function"
#'  optimized.code = text.optimize.code(google.model,
#'                                      temperature,
#'                                      prompt,
#'                                      goal,
#'                                      language = "R")
#'  cat(optimized.code)
#'
#'  optimized.code = text.optimize.code(openai.model,
#'                                      temperature,
#'                                      prompt,
#'                                      goal,
#'                                      language = "R")
#'  cat(optimized.code)
#' }
#'
#' @export
#'
#' @importFrom GenAI moderation.openai
text.optimize.code = function (model.parameter,
                               temperature,
                               prompt,
                               goal,
                               language = "R") {
  if (prompt == "" || is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  if (goal == "" || is.na(goal) || !inherits(goal, "character")) {
    stop("Goal is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            requestBody = list(
              contents = list(parts = list(text = paste("Optimize the following",
                                                        language, "code.\n",
                                                        "The goal is:", goal, "\n",
                                                        "# Code starts #\n",
                                                        prompt, "\n",
                                                        "# Code ends #\n"))),
              generationConfig = list(temperature = temperature)
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("Safety: The prompt may contain harmful content.")
            }
            return (as.character(responseJSON$candidates[[1]]$content$parts[[1]]$text))
          },
          openai = {
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            requestBody = list(
              model = model.parameter["model"],
              messages = list(
                list(role = "system",
                     content = "You are a helpful assistant."),
                list(role = "user",
                     content = paste("Optimize the following",
                                     language, "code.\n",
                                     "The goal is:", goal, "\n",
                                     "# Code starts #\n",
                                     prompt, "\n",
                                     "# Code ends #\n"))
              ),
              temperature = temperature
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            return (as.character(responseJSON$choices[[1]]$message$content))
          })
}
