% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function_CleanSLCClasses.R
\name{CleanSLCClasses}
\alias{CleanSLCClasses}
\title{Clean Soil-Landuse classes (SLCs) from small fractions}
\usage{
CleanSLCClasses(
  gd,
  gcl,
  m1.file = NULL,
  m1.class = "s",
  m1.clean = rep(TRUE, 2),
  m1.precedence = rep(TRUE, 2),
  m2.frac = NULL,
  m2.abs = NULL,
  signif.digits = 3,
  verbose = TRUE,
  progbar = TRUE
)
}
\arguments{
\item{gd}{Data frame containing columns with SUBIDs, SUBID areas in m^2, and SLC fractions, typically a 'GeoData.txt' file
imported with \code{\link{ReadGeoData}}.}

\item{gcl}{Data frame containing columns with SLCs and corresponding land use and soil class IDs, typically a 'GeoClass.txt'
file imported with \code{\link{ReadGeoClass}}.}

\item{m1.file}{Character string, path and file name of the soil or land use class transfer table, a tab-separated text file. Format see details.
A value of \code{NULL} (default) prevents method 1 cleaning.}

\item{m1.class}{Character string, either "soil" or "landuse", can be abbreviated. Gives the type of transfer class table for method 1 cleaning.
See Details.}

\item{m1.clean}{A logical vector of length 2 which indicates if cleaning should be performed for area fraction thresholds (position 1) and/or
absolute area thresholds (position 2).}

\item{m1.precedence}{A logical vector of length 2 which indicates if areas below cleaning threshold should be moved to similar areas according to
precedence in the transfer table given in \code{m1.file} (\code{TRUE}) or to the largest area of available transfer classes (\code{FALSE}). Area
fraction thresholds (position 1) and absolute area thresholds (position 2).}

\item{m2.frac}{Numeric, area fraction threshold for method 2 cleaning, i.e. moving of small SLC areas to largest SLC in each SUBID without considering
similarity between classes. Either a single value or a vector of the same length as the number of SLC classes in \code{gd}, giving area fraction
thresholds for each SLC separately, with a value \code{0} for SLCs to omit from cleaning. A value of \code{NULL} (default) prevents method 2 area
fraction cleaning.}

\item{m2.abs}{Numeric, see \code{m2.frac}. Threshold(s) for absolute areas in \eqn{m^{2}}{m^2}.}

\item{signif.digits}{Integer, number of significant digits to round cleaned SLCs to. See also \code{\link{signif}}. Set to \code{NULL} to prevent rounding.}

\item{verbose}{Logical, print some information during runtime.}

\item{progbar}{Logical, display a progress bar while calculating SLC class fractions. Adds overhead to calculation time but useful when \code{subid}
is \code{NULL} or contains many SUBIDs.}
}
\value{
\code{CleanSLCClasses} returns the GeoData data frame passed to the function in argument \code{gd} with cleaned SLC class columns.
}
\description{
\code{CleanSLCClasses} attempts to clean small SLC fractions within each SUBID (sub-catchment) from an imported GeoData file using user-provided
area thresholds. Cleaning can be performed along class similarity rules or along SLC area alone.
}
\details{
\code{CleanSLCClasses} performs a clean-up of small SLC fractions in an imported GeoData file. Small SLCs are eliminated either by moving their
area to similar classes according to rules which are passed to the function in a text file (\emph{Method 1}), or by simply moving their area to the
largest SLC in the SUBID (\emph{Method 2}). Moving rules for the first method can be based on either soil classes or land use classes but these cannot
be combined in one function call. Run the function two times to combine soil and land use based clean-up. Method 1 and 2, however, can be combined
in one function call, in which case the rule-based classification will be executed first. Clean-up precedence in method 1: if
clean-ups based on area fractions and absolute areas are combined (\code{m1.clean = rep(TRUE, 2)}), then area fractions will be cleaned first. In
order to reverse precedence, call \code{CleanSLCClasses} two times with absolute area cleaning activated in first call and area fraction cleaning
in second. In both methods, SLCs in each SUBID are cleaned iteratively in numerical order, starting with SLC_1. This implies a greater likelihood of
eliminating SLCs with smaller indices.

\bold{Method 1}

For method one, small SLC fractions are moved to either similar land use classes within the same soil class, or vice versa. Similarities are
defined by the user in a tab-separated text file, which is read by \code{CleanSLCClasses} during runtime. Soil and land use classes correspond to
the classes given in column two and three in the \code{GeoClass} file. The file must have the following format:

\tabular{cccccc}{
\emph{class.1}\tab\emph{thres.frac.1}\tab\emph{thres.abs.1}\tab\emph{transfer.1}\tab\emph{...}\tab\emph{transfer.n}\cr
\emph{class.2}\tab\emph{thres.frac.2}\tab\emph{thres.abs.2}\tab\emph{transfer.1}\tab\emph{...}\tab\emph{transfer.o}\cr
\emph{...}\tab\emph{...}\tab\emph{...}\tab\emph{...}\tab\emph{...}\tab\emph{...}\cr
\emph{class.m}\tab\emph{thres.frac.m}\tab\emph{thres.abs.m}\tab\emph{transfer.1}\tab\emph{...}\tab\emph{transfer.p}\cr
}

Column 1 contains the source land use or soil classes subjected to clean-up, columns 2 and 3 contain threshold values for area fractions and
absolute areas. The remaining columns contain classes to which areas below threshold will be transferred, in order of precedence. Each class can
have one or several transfer classes. \code{CleanSLCClasses} will derive SLC classes to clean from the given soil or land use class using the
GeoClass table given in argument \code{gcl}.
No header is allowed. At least one transfer class must exist, but classes can be omitted and will then be ignored by \code{CleanSLCClasses}.
The order of transfer classes in the transfer file indicates transfer preference. \code{CleanSLCClasses} constructs a transfer list for each SLC
class in the model set-up and per default uses the order to choose a preferred SLC to transfer to. However, if several SLCs exist for a given soil
or land use class, one of them will be chosen without further sorting. If argument \code{m1.precedence} is set to \code{FALSE} for either area
fractions or absolute areas, precedence will be ignored and the largest area available will be chosen to transfer small areas to. Area fraction
thresholds are given as fractions of 1, absolute area thresholds as values in \eqn{m^{2}}{m^2}. If an area below threshold is identified but there
are no fitting SLCs available to transfer to, the area will remain unchanged.

\bold{Method 2}

This method is more rigid than method one and can also be applied as a post-processor after clean-up using method 1 to force a removal of all SLCs
below a given threshold from a GeoData file (method 1 cleaning can be be very selective, depending on how many transfer classes are provided in
the transfer table). Cleaning thresholds for method 2 area fractions and absolute areas are given in arguments \code{m2.frac} and \code{m2.abs}.
SLC areas below the given thresholds will be moved to the largest SLC in the given SUBID without considering any similarity between classes.
}
\examples{
# Import source data
te1 <- ReadGeoData(filename = system.file("demo_model", "GeoData.txt", package = "HYPEtools"))
te2 <- ReadGeoClass(filename = system.file("demo_model", "GeoClass.txt", package = "HYPEtools"))
# Clean-up using method 2, 0.5 \% area fraction threshold and 100 m^2 absolute area threshold
te3 <- CleanSLCClasses(gd = te1, gcl = te2, m2.frac = 0.005, m2.abs = 100)
# Detailed comparison with function CompareFiles
te4 <- CompareFiles(te1, te3, type = "GeoData")
te4

}
\seealso{
\code{\link{RescaleSLCClasses}} for re-scaling of SLC area fraction sums.
}
