% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transdistfuncs.r
\name{sim.epidemic}
\alias{sim.epidemic}
\title{Simulation of an epidemic in space and time}
\usage{
sim.epidemic(R, gen.t.mean, gen.t.sd, trans.kern.func,
  tot.generations = 10, min.cases = 0, max.try = 1000)
}
\arguments{
\item{R}{a scalar or a vector of length \code{tot.generations} providing the reproductive number for the epidemic. 
If scalar, the R value is constant. If a vector, the R value varies according to each generation in the vector.}

\item{gen.t.mean}{mean of generation time}

\item{gen.t.sd}{standard deviation of the generation time (assumed to be normally distributed)}

\item{trans.kern.func}{a function for the transmission kernel that takes \code{n} as an arguement. 
Function and associated parameters must be given in a list object.}

\item{tot.generations}{the total number of generations in the epidemic, where the index case (x,y,t = [0,0,0]) is considered generation zero (default = 10)}

\item{min.cases}{the minimum number of cases in the epidemic (default = 0)}

\item{max.try}{maximum number of tries to acheive the minimum number of cases (default = 1000)}
}
\value{
a numerical matrix with three columns giving the coordinates \code{x} and \code{y}, and time \code{t} of simulated cases
}
\description{
A function which simulates the spatial spread of infections through time given the reproductive number (\code{R}), 
a function describing the spatial transmission kernel (\code{trans.kern.func}), and the mean and standard deviation 
of the generation time distribution (\code{gen.t.mean} and \code{gen.t.sd}) for the infecting pathogen. The function returns 
the location (\code{x}, \code{y}) and time (\code{t}) for each case of infection in the simulation.
}
\examples{
\donttest{
     
set.seed(1)

dist.func <- alist(n=1, a=1/100, rexp(n, a)) # Exponential transmission kernel with mean = sd = 100

# Simulate epidemic with constant R value
a <- sim.epidemic(R=1.5,
             gen.t.mean=7,
             gen.t.sd=2,
             tot.generations=15,
             min.cases=100,
             trans.kern.func=dist.func)

sim.plot(a)

# Simulate an epidemic with variable R value
r1 <- 2
r2 <- 0.25
tg <- 25
R <- seq(r1, r2, (r2 -r1)/(tg - 1))

b <- sim.epidemic(R=R,
             gen.t.mean=7,
             gen.t.sd=2,
             tot.generations=tg,
             min.cases=100,
             trans.kern.func=dist.func)

sim.plot(b)

}
}
\author{
John Giles, Justin Lessler, and Henrik Salje
}
