% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estarfm_job.R
\name{estarfm_job}
\alias{estarfm_job}
\title{Execute a single self-contained time-series imagefusion job using ESTARFM}
\usage{
estarfm_job(
  input_filenames,
  input_resolutions,
  input_dates,
  pred_dates,
  pred_filenames,
  pred_area,
  winsize,
  date1,
  date3,
  n_cores,
  data_range_min,
  data_range_max,
  uncertainty_factor,
  number_classes,
  hightag,
  lowtag,
  MASKIMG_options,
  MASKRANGE_options,
  use_local_tol,
  use_quality_weighted_regression,
  output_masks,
  use_nodata_value,
  verbose = TRUE
)
}
\arguments{
\item{input_filenames}{A string vector containing the filenames of the input images}

\item{input_resolutions}{A string vector containing the resolution-tags (corresponding to the arguments \code{hightag} and \code{lowtag}, which are by default "high" and "low") of the input images.}

\item{input_dates}{An integer vector containing the dates of the input images.}

\item{pred_dates}{An integer vector containing the dates for which images should be predicted.}

\item{pred_filenames}{A string vector containing the filenames for the predicted images. Must match \code{pred_dates} in length and order. Must include an extension relating to one of the \href{https://gdal.org/drivers/raster/index.html}{drivers supported by GDAL}, such as ".tif".}

\item{pred_area}{(Optional) An integer vector containing parameters in image coordinates for a bounding box which specifies the prediction area. The prediction will only be done in this area. (x_min, y_min, width, height). By default will use the entire area of the first input image.}

\item{winsize}{(Optional) Window size of the rectangle around the current pixel. Default is 51.}

\item{date1}{(Optional) Set the date of the first input image pair. By default, will use the pair with the lowest date value.}

\item{date3}{(Optional) Set the date of the second input image pair. By default, will use the pair with the highest date value.}

\item{n_cores}{(Optional) Set the number of cores to use when using parallelization. Default is 1.}

\item{data_range_min}{(Optional) When predicting pixel values ESTARFM can exceed the values that appear in the image. To prevent from writing invalid values (out of a known data range) you can set bounds. By default, the value range will be limited by the natural data range (e. g. -32767 for INT2S).}

\item{data_range_max}{(Optional) When predicting pixel values ESTARFM can exceed the values that appear in the image. To prevent from writing invalid values (out of a known data range) you can set bounds. By default, the value range will be limited by the natural data range (e. g.  32767 for INT2S).}

\item{uncertainty_factor}{(Optional) Sets the uncertainty factor. This is multiplied to the upper limit of the high resolution range. The range can be given by a mask range. Default: 0.002 (0.2 per cent)}

\item{number_classes}{(Optional) The number of classes used for similarity. Note all channels of a pixel are considered for similarity. So this value holds for each channel, e. g. with 3 channels there are n^3 classes. Default: c-th root of 64, where c is the number of channels.}

\item{hightag}{(Optional) A string which is used in \code{input_resolutions} to describe the high-resolution images. Default is "high".}

\item{lowtag}{(Optional) A string which is used in \code{input_resolutions} to describe the low-resolution images.  Default is "low".}

\item{MASKIMG_options}{(Optional) A string containing information for a mask image (8-bit, boolean, i. e. consists of 0 and 255). "For all input images the pixel values at the locations where the mask is 0 is replaced by the mean value." Example: \code{--mask-img=some_image.png}}

\item{MASKRANGE_options}{(Optional) Specify one or more intervals for valid values. Locations with invalid values will be masked out. Ranges should be given in the format \code{'[<float>,<float>]'}, \code{'(<float>,<float>)'}, \code{'[<float>,<float>'} or \code{'<float>,<float>]'}. There are a couple of options:' \itemize{
\item{"--mask-valid-ranges"}{ Intervals which are marked as valid. Valid ranges can excluded from invalid ranges or vice versa, depending on the order of options.}
\item{"--mask-invalid-ranges"}{ Intervals which are marked as invalid. Invalid intervals can be excluded from valid ranges or vice versa, depending on the order of options.}
\item{"--mask-high-res-valid-ranges"}{ This is the same as --mask-valid-ranges, but is applied only for the high resolution images.}
\item{"--mask-high-res-invalid-ranges"}{ This is the same as --mask-invalid-ranges, but is applied only for the high resolution images.}
\item{"--mask-low-res-valid-ranges"}{ This is the same as --mask-valid-ranges, but is applied only for the low resolution images.}
\item{"--mask-low-res-invalid-ranges"}{ This is the same as --mask-invalid-ranges, but is applied only for the low resolution images.}
}}

\item{use_local_tol}{(Optional) This enables the usage of local tolerances to find similar pixels instead of using the global tolerance.  When searching similar pixels, a tolerance of \eqn{2\sigma/m} is used. This options sets whether is calculated only from the local window region around the central pixel or from the whole image. Default is "false".}

\item{use_quality_weighted_regression}{(Optional) This enables the smooth weighting of the regression coefficient by its quality. The regression coefficient is not limited strictly by the quality, but linearly blended to 1 in case of bad quality. Default is "false".}

\item{output_masks}{(Optional) Write mask images to disk? Default is "false".}

\item{use_nodata_value}{(Optional) Use the nodata value as invalid range for masking? Default is "true".}

\item{verbose}{(Optional) Print progress updates to console? Default is "true".}
}
\value{
Nothing. Output files are written to disk. The Geoinformation for the output images is adopted from the first input pair images.
}
\description{
A wrapper function for \code{execute_estarfm_job_cpp}. Intended to execute a single job, that is a number of predictions based on the same input pairs. It ensures that all of the arguments passed are of the correct type and creates sensible defaults.
}
\details{
Executes the ESTARFM algorithm to create a number of synthetic high-resolution images from two pairs of matching high- and low-resolution images.  Assumes that the input images already have matching size. See the original paper for details (Note: There is a difference to the algorithm as described in the paper though. The regression for R is now done with all candidates of one window. This complies to the reference implementation, but not to the paper, since there the regression is done only for the candidates that belong to one single coarse pixel. However, the coarse grid is not known at prediction and not necessarily trivial to find out (e. g. in case of higher order interpolation).
}
\examples{
# Load required libraries
library(ImageFusion)
library(raster)
# Get filesnames of high resolution images
landsat <- list.files(
  system.file("landsat/filled",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)

# Get filesnames of low resolution images
modis <- list.files(
  system.file("modis",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)

#Select the first two landsat images 
landsat_sel <- landsat[1:2]
#Select the corresponding modis images
modis_sel <- modis[1:10]
# Create output directory in temporary folder
out_dir <- file.path(tempdir(),"Outputs")
if(!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)

#Run the job, fusing two images
estarfm_job(input_filenames = c(landsat_sel,modis_sel),
            input_resolutions = c("high","high",
                                  "low","low","low",
                                  "low","low","low",
                                  "low","low","low","low"),
            input_dates = c(68,77,68,69,70,71,72,73,74,75,76,77),
            pred_dates = c(72,74),
            pred_filenames = c(file.path(out_dir,"estarfm_72.tif"),
                               file.path(out_dir,"estarfm_74.tif"))
)
# remove the output directory
unlink(out_dir,recursive = TRUE)

}
\references{
Zhu, X., Chen, J., Gao, F., Chen, X., & Masek, J. G. (2010). An enhanced spatial and temporal adaptive reflectance fusion model for complex heterogeneous regions. Remote Sensing of Environment, 114(11), 2610-2623.
}
\author{
Christof Kaufmann (C++)

Johannes Mast (R)
}
